/*
 * configaccess.cpp
 *
 * Copyright (c) 2001 Frerich Raabe <raabe@kde.org>
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. For licensing and distribution details, check the
 * accompanying file 'COPYING'.
 */
#include "configaccess.h"
#include "newsengine.h"

#include <qregexp.h>
#include <QFont>
#include <kdebug.h>
#include <kglobal.h>
#include <kglobalsettings.h>

static NewsSourceBase::Data NewsSourceDefault[DEFAULT_NEWSSOURCES] = {
	// Arts ---------------
		NewsSourceBase::Data(
		QLatin1String("Bureau 42"),
		QLatin1String("http://www.bureau42.com/rdf/"),
		QLatin1String("http://www.bureau42.com/favicon.ico"),
		NewsSourceBase::Arts, 5, false, false),
		NewsSourceBase::Data(
		QLatin1String("eFilmCritic"),
		QLatin1String("http://efilmcritic.com/fo.rdf"),
		QLatin1String("http://efilmcritic.com/favicon.ico"),
		NewsSourceBase::Arts, 3, false, false),
		NewsSourceBase::Data(
		QLatin1String("superhits.ch"),
		QLatin1String("http://www.superhits.ch/cgi-bin/superhits.cgi?page=rdf"),
		QLatin1String("http://www.superhits.ch/favicon.ico"),
		NewsSourceBase::Arts, 10, false, false, QLatin1String("de")),
	// Business -----------
		NewsSourceBase::Data(
		QLatin1String("Internet.com Business"),
		QLatin1String("http://headlines.internet.com/internetnews/bus-news/news.rss"),
		QString::null,
		NewsSourceBase::Business, 10, false, false),
		NewsSourceBase::Data(
		QLatin1String("TradeSims"),
		QLatin1String("http://www.tradesims.com/AEX.rdf"),
		QString::null,
		NewsSourceBase::Business, 10, false, false),
	// Computers ----------
		NewsSourceBase::Data(
		QLatin1String( "linuxartist.org" ),
		QLatin1String( "http://www.linuxartist.org/backend.php"),
		QLatin1String( "http://www.linuxartist.org/favicon.ico"),
		NewsSourceBase::Computers, 10, false, false ),
		NewsSourceBase::Data(
		QLatin1String("KDE Deutschland"),
		QLatin1String("http://www.kde.de/nachrichten/nachrichten.rdf"),
		QLatin1String("http://www.kde.de/favicon.ico"),
		NewsSourceBase::Computers, 10, true, false, QLatin1String("de")),
		NewsSourceBase::Data(
		QLatin1String("KDE France"),
		QLatin1String("http://www.kde-france.org/backend-breves.php3"),
		QString::null,
		NewsSourceBase::Computers, 10, true, false, QLatin1String("fr")),
		NewsSourceBase::Data(
		QLatin1String("FreeBSD Project News"),
		QLatin1String("http://www.freebsd.org/news/news.rdf"),
		QLatin1String("http://www.freebsd.org/favicon.ico"),
		NewsSourceBase::Computers, 10, false, false),
		NewsSourceBase::Data(
		QLatin1String("dot.kde.org"),
		QLatin1String("http://www.kde.org/dotkdeorg.rdf"),
		QLatin1String("http://www.kde.org/favicon.ico"),
		NewsSourceBase::Computers, 10, true, false),
		NewsSourceBase::Data(
		QLatin1String("GNOME News"),
		QLatin1String("http://www.gnomedesktop.org/backend.php"),
		QString::null,
		NewsSourceBase::Computers, 10, false, false),
		NewsSourceBase::Data(
		QLatin1String("Slashdot"),
		QLatin1String("http://slashdot.org/slashdot.rdf"),
		QLatin1String("http://slashdot.org/favicon.ico"),
		NewsSourceBase::Computers, 10, true, false),
		NewsSourceBase::Data(
		QLatin1String("Ask Slashdot"),
		QLatin1String("http://slashdot.org/askslashdot.rdf"),
		QLatin1String("http://slashdot.org/favicon.ico"),
		NewsSourceBase::Computers, 10, false, false),
		NewsSourceBase::Data(
		QLatin1String("Slashdot: Features"),
		QLatin1String("http://slashdot.org/features.rdf"),
		QLatin1String("http://slashdot.org/favicon.ico"),
		NewsSourceBase::Computers, 10, false, false),
		NewsSourceBase::Data(
		QLatin1String("Slashdot: Apache"),
		QLatin1String("http://slashdot.org/apache.rdf"),
		QLatin1String("http://slashdot.org/favicon.ico"),
		NewsSourceBase::Computers, 10, false, false),
		NewsSourceBase::Data(
		QLatin1String("Slashdot: Books"),
		QLatin1String("http://slashdot.org/books.rdf"),
		QLatin1String("http://slashdot.org/favicon.ico"),
		NewsSourceBase::Computers, 10, false, false),
		NewsSourceBase::Data(
		QLatin1String("Jabber News"),
		QLatin1String("http://www.jabber.org/news/rss.xml"),
		QString::null,
		NewsSourceBase::Computers, 10, false, false),
		NewsSourceBase::Data(
		QLatin1String("Freshmeat"),
		QLatin1String("http://freshmeat.net/backend/fm.rdf"),
		QLatin1String("http://freshmeat.net/favicon.ico"),
		NewsSourceBase::Computers, 10, false, false),
		NewsSourceBase::Data(
		QLatin1String("Linux Weekly News"),
		QLatin1String("http://www.lwn.net/headlines/rss"),
		QLatin1String("http://www.lwn.net/favicon.ico"),
		NewsSourceBase::Computers, 10, true, false),
		NewsSourceBase::Data(
		QLatin1String("heise online news"),
		QLatin1String("http://www.heise.de/newsticker/heise.rdf"),
		QLatin1String("http://www.heise.de/favicon.ico"),
		NewsSourceBase::Computers, 10, true, false, QLatin1String("de")),
		NewsSourceBase::Data(
		QLatin1String("RUS-CERT Ticker"),
		QLatin1String("http://cert.uni-stuttgart.de/ticker/rus-cert.rdf"),
		QLatin1String("http://cert.uni-stuttgart.de/favicon.ico"),
		NewsSourceBase::Computers, 10, true, false, QLatin1String("de")),
		NewsSourceBase::Data(
		QLatin1String("RUS-CERT Elsewhere"),
		QLatin1String("http://cert.uni-stuttgart.de/ticker/rus-cert-elsewhere.rdf"),
		QLatin1String("http://cert.uni-stuttgart.de/favicon.ico"),
		NewsSourceBase::Computers, 10, false, false, QLatin1String("de")),
		NewsSourceBase::Data(
		QLatin1String("Kuro5hin"),
		QLatin1String("http://kuro5hin.org/backend.rdf"),
		QLatin1String("http://kuro5hin.org/favicon.ico"),
		NewsSourceBase::Computers, 10, true, false),
		NewsSourceBase::Data(
		QLatin1String("Prolinux"),
		QLatin1String("http://www.pl-forum.de/backend/pro-linux.rdf"),
		QLatin1String("http://www.prolinux.de/favicon.ico"),
		NewsSourceBase::Computers, 10, false, false, QLatin1String("de")),
		NewsSourceBase::Data(
		QLatin1String("Linuxde.org"),
		QLatin1String("http://www.linuxde.org/backends/news.rdf"),
		QLatin1String("http://www.linuxde.org/favicon.ico"),
		NewsSourceBase::Computers, 10, false, false, QLatin1String("de")),
		NewsSourceBase::Data(
		QLatin1String("LinuxSecurity.com"),
		QLatin1String("http://www.linuxsecurity.com/linuxsecurity_hybrid.rdf"),
		QString::null,
		NewsSourceBase::Computers, 10, false, false),
		NewsSourceBase::Data(
		QLatin1String("Linux Game Tome"),
		QLatin1String("http://happypenguin.org/html/news.rdf"),
		QString::null,
		NewsSourceBase::Computers, 10, false, false),
		NewsSourceBase::Data(
		QLatin1String("Telefragged"),
		QLatin1String("http://www.telefragged.com/cgi-bin/rdf.pl"),
		QString::null,
		NewsSourceBase::Computers, 10, false, false),
/*		NewsSourceBase::Data(
		QLatin1String("Gimp News"),
		QLatin1String("http://www.xach.com/gimp/news/channel.rdf"),
		QString::null,
		NewsSourceBase::Computers, 10, false, false),*/
		NewsSourceBase::Data(
		QLatin1String("Mozilla"),
		QLatin1String("http://www.mozilla.org/news.rdf"),
		QLatin1String("http://www.mozillazine.org/favicon.ico"),
		NewsSourceBase::Computers, 10, false, false),
		NewsSourceBase::Data(
		QLatin1String("MozillaZine"),
		QLatin1String("http://www.mozillazine.org/contents.rdf"),
		QLatin1String("http://www.mozillazine.org/favicon.ico"),
		NewsSourceBase::Computers, 10, false, false),
		NewsSourceBase::Data(
		QLatin1String("Daemon News"),
		QLatin1String("http://daily.daemonnews.org/ddn.rdf.php3"),
		QString::null,
		NewsSourceBase::Computers, 10, false, false),
		NewsSourceBase::Data(
		QLatin1String("use Perl;"),
		QLatin1String("http://use.perl.org/useperl.rdf"),
		QString::null,
		NewsSourceBase::Computers, 10, false, false),
		NewsSourceBase::Data(
		QLatin1String("desktopian.org"),
		QLatin1String("http://www.desktopian.org/includes/headlines.xml"),
		QLatin1String("http://www.desktopian.org/favicon.ico"),
		NewsSourceBase::Computers, 10, false, false),
		NewsSourceBase::Data(
		QLatin1String("Root prompt"),
		QLatin1String("http://www.rootprompt.org/rss/"),
		QLatin1String("http://www.rootprompt.org/favicon.ico"),
		NewsSourceBase::Computers, 10, false, false),
		NewsSourceBase::Data(
		QLatin1String("SecurityFocus"),
		QLatin1String("http://www.securityfocus.com/rss/news.xml"),
		QLatin1String("http://www.securityfocus.com/favicon.ico"),
		NewsSourceBase::Computers, 10, false, false),
		NewsSourceBase::Data(
		QLatin1String("LinuxNewbie"),
		QLatin1String("http://www.linuxnewbie.org/news.cdf"),
		QLatin1String("http://www.linuxnewbie.org/favicon.ico"),
		NewsSourceBase::Computers, 5, false, false),
		NewsSourceBase::Data(
		QLatin1String("Arstechnica"),
		QLatin1String("http://arstechnica.com/etc/rdf/ars.rdf"),
		QLatin1String("http://arstechnica.com/favicon.ico"),
		NewsSourceBase::Computers, 10, false, false),
		NewsSourceBase::Data(
		QLatin1String("amiga-news.de - deutschsprachige Amiga Nachrichten"),
		QLatin1String("http://www.amiga-news.de/de/backends/news/index.rss"),
		QLatin1String("http://www.amiga-news.de/favicon.ico"),
		NewsSourceBase::Computers, 10, false, false, QLatin1String("de")),
		NewsSourceBase::Data(
		QLatin1String("amiga-news.de - english Amiga news"),
		QLatin1String("http://www.amiga-news.de/en/backends/news/index.rss"),
		QLatin1String("http://www.amiga-news.de/favicon.ico"),
		NewsSourceBase::Computers, 10, false, false),
		NewsSourceBase::Data(
		QLatin1String("Linux4Us (sowie RadioTux & Magazin42)"),
		QLatin1String("http://www.linux4us.de/newsticker.fd"),
		QString::null,
		NewsSourceBase::Computers, 10, false, false, QLatin1String("de")),
		NewsSourceBase::Data(
		QLatin1String("kdenews.unixcode.org"),
		QLatin1String("http://kdenews.unixcode.org/?node=news&action=rss"),
		QString::null,
		NewsSourceBase::Computers, 10, false, false),
		NewsSourceBase::Data(
		QLatin1String("FreshPorts - the place for ports"),
		QLatin1String("http://www.freshports.org/news.php"),
		QLatin1String("http://www.freshports.org/favicon.ico"),
		NewsSourceBase::Computers, 20, false, false),
		NewsSourceBase::Data(
		QLatin1String("NetPhoenix"),
		QLatin1String("http://www.netphoenix.at/rss/netphoenix.php"),
		QLatin1String("http://www.netphoenix.at/favicon.ico"),
		NewsSourceBase::Computers, 10, false, false, QLatin1String("de")),
		NewsSourceBase::Data(
		QLatin1String("ShortNews - by www.netphoenix.at"),
		QLatin1String("http://www.netphoenix.at/rss/shortnews.php"),
		QLatin1String("http://www.netphoenix.at/favicon.ico"),
		NewsSourceBase::Computers, 10, false, false, QLatin1String("de")),
		NewsSourceBase::Data(
		QLatin1String("zez.org - about code "),
		QLatin1String("http://zez.org/article/rssheadlines"),
		QString::null,
		NewsSourceBase::Computers, 10, false, false),
		NewsSourceBase::Data(
		QLatin1String("BSDatwork.com"),
		QLatin1String("http://BSDatwork.com/backend.php"),
		QLatin1String("http://BSDatwork.com/favicon.ico"),
		NewsSourceBase::Computers, 10, false, false),
		NewsSourceBase::Data(
		QLatin1String("FreshSource - the place for source"),
		QLatin1String("http://www.freshsource.org/news.php"),
		QLatin1String("http://www.freshsource.org/favicon.ico"),
		NewsSourceBase::Computers, 20, false, false),
		NewsSourceBase::Data(
		QLatin1String("The FreeBSD Diary"),
		QLatin1String("http://www.freebsddiary.org/news.php"),
		QLatin1String("http://www.freebsddiary.org/favicon.ico"),
		NewsSourceBase::Computers, 10, false, false),
		NewsSourceBase::Data(
		QLatin1String("MaximumBSD"),
		QLatin1String("http://www.maximumbsd.com/backend/mb.rdf"),
		QLatin1String("http://www.maximumbsd.com/favicon.ico"),
		NewsSourceBase::Computers, 10, false, false),
		NewsSourceBase::Data(
		QLatin1String("BR-Linux.org"),
		QLatin1String("http://br-linux.org/noticias/index.rdf"),
		QLatin1String("http://br-linux.org/noticias/favicon.ico"),
		NewsSourceBase::Computers, 10, false, false),
		NewsSourceBase::Data(
		QLatin1String("OSNews.com"),
		QLatin1String("http://www.osnews.com/files/recent.rdf"),
		QLatin1String("http://www.osnews.com/favicon.ico"),
		NewsSourceBase::Computers, 10, false, false),
	// Miscellaneous ------
		NewsSourceBase::Data(
		QLatin1String("tagesschau.de"),
		QLatin1String("http://www.tagesschau.de/newsticker.rdf"),
		QLatin1String("http://www.tagesschau.de/favicon.ico"),
		NewsSourceBase::Misc, 10, true, false, QLatin1String("de")),
		NewsSourceBase::Data(
		QLatin1String("N24.de"),
		QLatin1String("http://www.n24.de/rss/?rubrik=home"),
		QLatin1String("http://www.n24.de/favicon.ico"),
		NewsSourceBase::Misc, 10, true, false, QLatin1String("de")),
/*
		NewsSourceBase::Data(
		QLatin1String("CNN"),
		QLatin1String("http://www.cnn.com/cnn.rss"),
		QLatin1String("http://www.cnn.com/favicon.ico"),
		NewsSourceBase::Misc, 10, false, false),
		NewsSourceBase::Data(
		QLatin1String("CNN Europe"),
		QLatin1String("http://europe.cnn.com/cnn.rss"),
		QLatin1String("http://europe.cnn.com/favicon.ico"),
		NewsSourceBase::Misc, 10, false, false),
*/
		NewsSourceBase::Data(
		QLatin1String("HotWired"),
		QLatin1String("http://www.hotwired.com/webmonkey/meta/headlines.rdf"),
		QLatin1String("http://www.hotwired.com/favicon.ico"),
		NewsSourceBase::Misc, 10, false, false),
		NewsSourceBase::Data(
		QLatin1String("The Register"),
		QLatin1String("http://www.theregister.co.uk/tonys/slashdot.rdf"),
		QLatin1String("http://www.theregister.co.uk/favicon.ico"),
		NewsSourceBase::Misc, 10, false, false),
		NewsSourceBase::Data(
		QLatin1String( "Christian Science Monitor" ),
		QLatin1String( "http://www.csmonitor.com/rss/csm.rss"),
		QLatin1String( "http://www.csmonitor.com/favicon.ico"),
		NewsSourceBase::Misc, 10, false, false ),
	// Magazines ------
		NewsSourceBase::Data(
		QLatin1String("Revista do Linux"),
		QLatin1String("http://www.revistadolinux.com.br/noticias/arquivo/noticias.rdf"),
		QLatin1String("http://www.revistadolinux.com.br/favicon.ico"),
		NewsSourceBase::Magazines, 10, false, false /*, QLatin1String("br")*/ ),
		NewsSourceBase::Data(
		QLatin1String("Spiegel.de"),
		QLatin1String("http://www.spiegel.de/schlagzeilen/rss/0,5291,,00.xml"),
		QLatin1String("http://www.spiegel.de/favicon.ico"),
		NewsSourceBase::Magazines, 10, true, false, QLatin1String("de")),
		NewsSourceBase::Data(
		QLatin1String("FAZ.de"),
		QLatin1String("http://www.faz.net/s/Rub/Tpl~Epartner~SRss_.xml"),
		QLatin1String("http://www.faz.net/favicon.ico"),
		NewsSourceBase::Magazines, 10, true, false, QLatin1String("de")),
	// Recreation
		NewsSourceBase::Data(
		QLatin1String("Segfault"),
		QLatin1String("http://segfault.org/stories.xml"),
		QLatin1String("http://segfault.org/favicon.ico"),
		NewsSourceBase::Recreation, 10, false, false),
	// Society
		NewsSourceBase::Data(
		QLatin1String("nippon.it"),
		QLatin1String("http://www.nippon.it/backend.it.php"),
		QLatin1String("http://www.nippon.it/favicon.ico"),
		NewsSourceBase::Society, 10, false, false, QLatin1String("it")),
		NewsSourceBase::Data( 
		QLatin1String( "gflash" ),
		QLatin1String( "http://www.gflash.de/backend.php"),
		QLatin1String( "http://www.gflash.de/favicon.ico"),
		NewsSourceBase::Society, 10, false, false, QLatin1String( "de" ) ),
		NewsSourceBase::Data( 
		QLatin1String( "Quintessenz" ),
		QLatin1String( "http://quintessenz.at/cgi-bin/rdf"),
		QLatin1String( "http://quintessenz.at/favicon.ico"),
		NewsSourceBase::Society, 9, false, false, QLatin1String( "de" ) )
};

ArticleFilter::ArticleFilter(const QString &action, const QString &newsSource,
		const QString &condition, const QString &expression, bool enabled)
	: m_action(action),
	m_newsSource(newsSource),
	m_condition(condition),
	m_expression(expression),
	m_enabled(enabled)
{
}

bool ArticleFilter::matches(Article::Ptr a) const
{
	if (!enabled() ||
	    (a->newsSource()->data().name != newsSource() &&
	    newsSource() != i18n("all news sources")))
		return false;
	
	bool matches;

	if (condition() == i18n("contain"))
		matches = a->headline().contains(expression());
	else if (condition() == i18n("do not contain"))
		matches = !a->headline().contains(expression());
	else if (condition() == i18n("equal"))
		matches = (a->headline() == expression());
	else if (condition() == i18n("do not equal"))
		matches = (a->headline() != expression());
	else { // condition() == i18n("match")
		QRegExp regexp = QRegExp(expression());
		matches = regexp.exactMatch(a->headline());
	}
	
	if (action() == i18n("Show"))
		matches = !matches;

	return matches;
}
	
ConfigAccess::ConfigAccess()
	: m_defaultCfg(new KConfig(QString::null, true, false))
{
	m_cfg = m_defaultCfg;
}

ConfigAccess::ConfigAccess(KConfig *config)
	: m_cfg(config), m_defaultCfg(0L)
{
	m_cfg->setGroup("KNewsTicker");
}

ConfigAccess::~ConfigAccess()
{
	delete m_defaultCfg;
}

unsigned int ConfigAccess::interval() const
{
	return m_cfg->readEntry("Update interval", 30);
}

void ConfigAccess::setInterval(const unsigned int interval)
{
	m_cfg->writeEntry("Update interval", interval);
	m_cfg->sync();
}

unsigned int ConfigAccess::mouseWheelSpeed() const
{
	return m_cfg->readEntry("Mouse wheel speed", 5);
}

void ConfigAccess::setMouseWheelSpeed(const unsigned int mouseWheelSpeed)
{
	m_cfg->writeEntry("Mouse wheel speed", mouseWheelSpeed);
	m_cfg->sync();
}

QFont ConfigAccess::font() const
{
	QFont font = KGlobalSettings::fixedFont();
	return m_cfg->readEntry("Font", font);
}

void ConfigAccess::setFont(const QFont &font)
{
	m_cfg->writeEntry("Font", font);
	m_cfg->sync();
}

bool ConfigAccess::customNames() const
{
	return m_cfg->readEntry("Custom names", false);
}

void ConfigAccess::setCustomNames(bool customNames)
{
	m_cfg->writeEntry("Custom names", customNames);
	m_cfg->sync();
}

bool ConfigAccess::scrollMostRecentOnly() const
{
	return m_cfg->readEntry("Scroll most recent headlines only", false);
}

void ConfigAccess::setScrollMostRecentOnly(bool scrollMostRecentOnly)
{
	m_cfg->writeEntry("Scroll most recent headlines only", scrollMostRecentOnly);
	m_cfg->sync();
}

bool ConfigAccess::offlineMode() const
{
	return m_cfg->readEntry("Offline mode", false);
}

void ConfigAccess::setOfflineMode(bool offlineMode)
{
	m_cfg->writeEntry("Offline mode", offlineMode);
	m_cfg->sync();
}

QStringList ConfigAccess::newsSources() const
{
	QStringList tempList = m_cfg->readEntry("News sources",QStringList());
	if (tempList.isEmpty())
		for (unsigned int i = 0; i < DEFAULT_NEWSSOURCES; i++)
			tempList << NewsSourceDefault[i].name;
	return tempList;
}

ArticleFilter::List ConfigAccess::filters() const
{
	return m_cfg->readEntry("Filters", QList<int>());
}

void ConfigAccess::setNewsSources(const QStringList &newsSources)
{
	m_cfg->writeEntry("News sources", newsSources);
	m_cfg->sync();
}

void ConfigAccess::setFilters(const ArticleFilter::List &filters)
{
	m_cfg->writeEntry("Filters", filters);
	m_cfg->sync();
}

unsigned int ConfigAccess::scrollingSpeed() const
{
	return m_cfg->readEntry("Scrolling speed", 20);
}

void ConfigAccess::setScrollingSpeed(const unsigned int scrollingSpeed)
{
	m_cfg->writeEntry("Scrolling speed", scrollingSpeed);
	m_cfg->sync();
}

unsigned int ConfigAccess::scrollingDirection() const
{
	return m_cfg->readEntry("Scrolling direction", 0);
}

void ConfigAccess::setScrollingDirection(const unsigned int scrollingDirection)
{
	m_cfg->writeEntry("Scrolling direction", scrollingDirection);
	m_cfg->sync();
}

QColor ConfigAccess::foregroundColor() const
{
	QColor black(Qt::black);
	return m_cfg->readEntry("Foreground color", black);
}

void ConfigAccess::setForegroundColor(const QColor &foregroundColor)
{
	m_cfg->writeEntry("Foreground color", foregroundColor);
	m_cfg->sync();
}

QColor ConfigAccess::backgroundColor() const
{
	QColor white(Qt::white); 
	return m_cfg->readEntry("Background color", white);
}

void ConfigAccess::setBackgroundColor(const QColor &backgroundColor)
{
	m_cfg->writeEntry("Background color", backgroundColor);
	m_cfg->sync();
}

QColor ConfigAccess::highlightedColor() const
{
	QColor red(Qt::red);
	return m_cfg->readEntry("Highlighted color", red);
}

void ConfigAccess::setHighlightedColor(const QColor &highlightedColor)
{
	m_cfg->writeEntry("Highlighted color", highlightedColor);
	m_cfg->sync();
}

bool ConfigAccess::underlineHighlighted() const
{
	return m_cfg->readEntry("Underline highlighted headlines", true);
}

void ConfigAccess::setUnderlineHighlighted(bool underlineHighlighted)
{
	m_cfg->writeEntry("Underline highlighted headlines", underlineHighlighted);
	m_cfg->sync();
}

NewsSourceBase *ConfigAccess::newsSource(const QString &newsSource)
{
	NewsSourceBase::Data nsd;

	if (m_cfg->hasGroup(newsSource)) {
		m_cfg->setGroup(newsSource);
		nsd.name = newsSource;
		nsd.sourceFile = m_cfg->readPathEntry("Source file");
		nsd.isProgram = m_cfg->readEntry("Is program", false);
		nsd.subject = static_cast<NewsSourceBase::Subject>
			(m_cfg->readEntry("Subject", int(NewsSourceBase::Computers)));
		nsd.icon = m_cfg->readEntry("Icon");
		nsd.maxArticles = m_cfg->readEntry("Max articles", 10);
		nsd.enabled = m_cfg->readEntry("Enabled", true);
		nsd.language = m_cfg->readEntry("Language", QString("C"));
		m_cfg->setGroup("KNewsTicker");
	} else for (unsigned int i = 0; i < DEFAULT_NEWSSOURCES; i++)
		if (NewsSourceDefault[i].name == newsSource) {
			nsd = NewsSourceDefault[i];
			if (nsd.enabled)
				nsd.enabled = (nsd.language == QLatin1String("C") ||
						KGlobal::locale()->languageList().contains(nsd.language));
			break;
		}
	
	if (nsd.isProgram)
		return new ProgramNewsSource(nsd, this);
	else 
		return new SourceFileNewsSource(nsd, this);
	
	return 0L;
}

ArticleFilter ConfigAccess::filter(const unsigned int filterNo) const
{
	ArticleFilter f;
	f.setId(filterNo);

	if (m_cfg->hasGroup(QString::fromLatin1("Filter #%1").arg(filterNo))) {
		m_cfg->setGroup(QString::fromLatin1("Filter #%1").arg(filterNo));
		f.setAction(m_cfg->readEntry("Action", i18n("Show")));
		f.setNewsSource(m_cfg->readEntry("News source", i18n("all news sources")));
		f.setCondition(m_cfg->readEntry("Condition", i18n("contain")));
		f.setExpression(m_cfg->readEntry("Expression"));
		f.setEnabled(m_cfg->readEntry("Enabled", true));
		m_cfg->setGroup("KNewsTicker");
	}

	return f;
}

void ConfigAccess::setNewsSource(const NewsSourceBase::Data &ns)
{
	m_cfg->setGroup(ns.name);
	m_cfg->writePathEntry("Source file", ns.sourceFile);
	m_cfg->writeEntry("Is program", ns.isProgram);
	m_cfg->writeEntry("Max articles", ns.maxArticles);
	m_cfg->writeEntry("Subject", int(ns.subject));
	m_cfg->writeEntry("Icon", ns.icon);
	m_cfg->writeEntry("Enabled", ns.enabled);
	m_cfg->writeEntry("Language", ns.language);
	m_cfg->setGroup("KNewsTicker");
	m_cfg->sync();
}

void ConfigAccess::setFilter(const ArticleFilter &f)
{
	m_cfg->setGroup(QString::fromLatin1("Filter #%1").arg(f.id()));
	m_cfg->writeEntry("Action", f.action());
	m_cfg->writeEntry("News source", f.newsSource());
	m_cfg->writeEntry("Condition", f.condition());
	m_cfg->writeEntry("Expression", f.expression());
	m_cfg->writeEntry("Enabled", f.enabled());
	m_cfg->setGroup("KNewsTicker");
	m_cfg->sync();
}

bool ConfigAccess::showIcons() const
{
	return m_cfg->readEntry("Show icons", true);
}

void ConfigAccess::setShowIcons(bool showIcons)
{
	m_cfg->writeEntry("Show icons", showIcons);
	m_cfg->sync();
}

bool ConfigAccess::slowedScrolling() const
{
	return m_cfg->readEntry("Slowed scrolling", false);
}

void ConfigAccess::setSlowedScrolling(bool slowedScrolling)
{
	m_cfg->writeEntry("Slowed scrolling", slowedScrolling);
	m_cfg->sync();
}

