// systray.h
//
// Copyright 2004 Stefan Gehn <metz AT gehn DOT net>
// Copyright 2000 Neil Stevens <multivac@fcmail.com>
// Copyright 1999 Charles Samuels <charles@kde.org>
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
// THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
// AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
// CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//
// Except as contained in this notice, the name(s) of the author(s) shall not be
// used in advertising or otherwise to promote the sale, use or other dealings
// in this Software without prior written authorization from the author(s).


#include "systray.h"
#include "kitsystemtray.h"
#include "systray_prefs.h"
#include "yhconfig.h"

#include "noatun/global.h"
#include "noatun/player.h"
#include "noatun/stdaction.h"

#include <qfile.h>
#include <qimage.h>
//#include <qhbox.h>
//#include <qpainter.h>
#include <qpushbutton.h>
#include <qtooltip.h>
#include <qvbox.h>

#include <kaction.h>
#include <kapplication.h>
#include <kconfig.h>
#include <kglobal.h>
#include <kiconloader.h>
#include <kiconeffect.h>
#include <klocale.h>
#include <kxmlguiwindow.h>
#include <kpassivepopup.h>
#include <kstandardaction.h>

#include <kurl.h>
#include <kio/netaccess.h>
#include <kdebug.h>
#include <kstandarddirs.h>

#include <netwm.h>
#include <kglobalsettings.h>
#include <kgenericfactory.h>

// TODO: Maybe make this value configurable?
const int COVER_MAXW = 128;
const int COVER_MAXH = 128;

using namespace Noatun;

K_EXPORT_COMPONENT_FACTORY(noatun_systray, PluginFactory<NoatunSystray>("noatun_systray"))


// From JuK
class PassivePopup : public KPassivePopup
{
public:
	PassivePopup(QWidget *parent = 0, const char *name = 0) : KPassivePopup(parent, name)
	{
	}

protected:
	virtual void enterEvent(QEvent *)
	{
		setTimeout(3000000); // Make timeout damn near infinite
	}

	virtual void leaveEvent(QEvent *)
	{
		setTimeout(250); // Close quickly
	}
};


NoatunSystray::NoatunSystray(KComponentData inst, Global *glob, const char* name)
	: Plugin(inst, glob, name), mTray(0), mPassivePopup(0)
{
	showingTrayStatus = false;
	currentState = Stopped;
	tmpCoverPath = locateLocal("tmp", "youngHickoryCover.png");
	removeCover(); // make sure any old temp cover is gone

	connect(kapp, SIGNAL(iconChanged(int)), this, SLOT(slotIconThemeChanged()));
	createTrayPixmaps();

	mBlinkTimer = new QTimer(this);
	connect(mBlinkTimer, SIGNAL(timeout()), this, SLOT(slotBlinkTimer()));

	connect(glob->player(), SIGNAL(playing()), this, SLOT(slotPlayPause()));
	connect(glob->player(), SIGNAL(paused()), this, SLOT(slotPlayPause()));
	connect(glob->player(), SIGNAL(stopped()), this, SLOT(slotStopped()));

	// -------------------------------------------------------------------
	StdAction::back(glob, actionCollection(), "player_back");
	StdAction::stop(glob, actionCollection(), "player_stop");
	StdAction::playpause(glob, actionCollection(), "player_playpause");
	StdAction::forward(glob, actionCollection(), "player_forward");
	StdAction::loop(glob, actionCollection(), "player_loopstyle");
	StdAction::playlist(glob, actionCollection(), "view_playlist");
	StdAction::effects(glob, actionCollection(), "view_effects");
	StdAction::equalizer(glob, actionCollection(), "view_equalizer");
	StdAction::quit(glob, actionCollection(), "file_quit");
	StdAction::preferences(glob, actionCollection(), "options_configure");

	SystrayPrefs *prefsPage = new SystrayPrefs(glob, this);
	connect(prefsPage, SIGNAL(saved()), this, SLOT(slotLoadSettings()));

	connect(glob->frontend(), SIGNAL(newFrontend()), this, SLOT(slotLoadSettings()));
	slotLoadSettings();
}


NoatunSystray::~NoatunSystray()
{
	kDebug(66666) << "Called.";
	removeCover();
	if (mTray) // could have been destroyed if the UI is gone already
	{
		QWidget *win = static_cast<QWidget *>(mTray->parent());
		if (win)
			win->show();
		delete mTray;
	}
}


Interface *NoatunSystray::getInterface(const QString&)
{
	return 0;
}


void NoatunSystray::slotLoadSettings()
{
	kDebug(66666) << "Called.";
	YHConfig *c = YHConfig::self();
	c->readConfig();

	// tray is disabled, make sure we don't have one around right now
	// OR:
	// tray is enabled but we have no mainwindow right now
	if (!c->enabled() || !global()->frontend()->mainWindow())
	{
		mBlinkTimer->stop();
		delete mPassivePopup;
		mPassivePopup = 0;
		delete mTray;
		mTray = 0;
		return;
	}

	// ok, tray is enabled and we have a valid mainwindow, create a tray
	if (!mTray)
	{
		mTray = new KitSystemTray(this, global()->frontend()->mainWindow(), "NoatunSystray::mTray");
		mTray->show();
	}


	if(c->stateIconDisplay() == YHConfig::FlashingIcon)
		mBlinkTimer->start(1000);
	else
		mBlinkTimer->stop();
	slotBlinkTimer();

	if(c->tip())
		QToolTip::add(mTray, tipText);
	else
		QToolTip::remove(mTray);

	if (!c->passivePopupCovers())
		removeCover();

	if(c->passivePopup())
	{
		mPassivePopup = new PassivePopup(mTray, "NoatunPassivePopup");
	}
	else
	{
		delete mPassivePopup;
		mPassivePopup = 0;
	}
}


void NoatunSystray::slotPlayPause()
{
	if (!mTray)
		return;

	QString status;
	Player *pl = global()->player();

	if(pl->isPaused())
	{
		changeTray(Paused);
		status = i18n("Noatun - Paused");
	}
	else
	{
		changeTray(Playing);
		status = i18n("Noatun - Playing");
	}

	const PlaylistItem item = pl->current();
	QString s;

	if(!item.hasProperty("title"))
	{
		// No metadata
		s = QString("<nobr>%1</nobr>").arg(pl->currentTitle());
	}
	else
	{
		s = QString("<h2><nobr>%1</nobr></h2>").arg(item.property("title"));

		if(item.hasProperty("author"))
			s += QString("<nobr>%1</nobr><br>").arg(item.property("author"));

		if(item.hasProperty("album"))
		{
			if(item.hasProperty("date"))
				s += QString("<nobr>%1 (%2)</nobr><br>").arg(item.property("album")).arg(item.property("date"));
			else
				s += QString("<nobr>%1</nobr><br>").arg(item.property("album"));
		}
	}

	// prepare cover image for display
	if (YHConfig::self()->passivePopupCovers())
		updateCover();

	if(YHConfig::self()->passivePopupCovers() && QFile::exists(tmpCoverPath))
	{
		// QT always adds an empty line after the table so we add en empty line before the
		// table to get equal spacing on top and bottom
		setTipText(QString("<qt><br><table cellspacing=0 cellpadding=0><tr>" \
			"<td align=center valign=center><h4><nobr>%1</nobr></h4>%2</td>" \
			"<td valign=center><img src='%3'></td>" \
			"</qt></tr></table>").arg(status).arg(s).arg(tmpCoverPath));
	}
	else
	{
		setTipText(QString("<qt><center><h4><nobr>%1</nobr></h4>%2</center></qt>").arg(status).arg(s));
	}
}


void NoatunSystray::slotStopped()
{
	if (!mTray)
		return;

	if(!global()->player()->current())
		return;
	changeTray(Stopped);
	setTipText(QString("<qt><nobr><h4>%1</h4></nobr></qt>").arg(i18n("Noatun - Stopped")));
}


void NoatunSystray::changeTray(TrayIcons iconState)
{
	currentState = iconState;
	if(showingTrayStatus)
		slotBlinkTimer();
}


void NoatunSystray::slotBlinkTimer()
{
	if (!mTray)
	{
		mBlinkTimer->stop();
		return;
	}

	switch(YHConfig::self()->stateIconDisplay())
	{
		case (YHConfig::FlashingIcon):
			showingTrayStatus ^= true;
			break;
		case (YHConfig::StaticIcon):
			showingTrayStatus = true;
			break;
		case (YHConfig::NoIcon):
			showingTrayStatus = false;
			break;
	}

	if(showingTrayStatus)
		mTray->setPixmap(trayPixmaps[currentState]);
	else
		mTray->setPixmap(trayPixmaps[Base]);
}


// taken from patched karamba xmmssensor
// modified heavily to work in this place
void NoatunSystray::updateCover()
{
	//kDebug(66666) ;
	QString dir = global()->player()->current().url().directory();
	QString cover;

	// TODO: Maybe make these filenames configurable?
	if(QFile::exists(dir + "/folder.png"))
		cover = dir + "/folder.png";
	else if(QFile::exists(dir + "/.folder.png"))
		cover = dir + "/.folder.png";
	else if(QFile::exists(dir + "/cover.png"))
		cover = dir + "/cover.png";
	else if(QFile::exists(dir + "/cover.jpg"))
		cover = dir + "/cover.jpg";
	else if(QFile::exists(dir + "/cover.jpeg"))
		cover = dir + "/cover.jpeg";
	else // no cover
	{
		//kDebug(66666) << "NO COVER";
		removeCover();
		return;
	}

	QString title = global()->player()->currentTitle();

	QImage previmg;
	previmg.load(tmpCoverPath);

	if(previmg.text("Title") != title)
	{ //Verify song change to limit CPU usage
		/*kDebug(66666) << "Creating new temp cover for '" <<
			cover << "'" << endl;*/

		QImage src;
		QImage tmpimg;

		if(src.load(cover))
		{
			if(src.width() >= COVER_MAXW || src.height() >= COVER_MAXH)
				tmpimg = src.scale(COVER_MAXW, COVER_MAXH, QImage::ScaleMin);
			else
				tmpimg = src;

			tmpimg.setText("Title", 0, title); //add Title in the image text for cache usage
			tmpimg.save(tmpCoverPath, "PNG", 0);
		}
		else
		{
			removeCover();
		}
	}
}


void NoatunSystray::removeCover()
{
	if(QFile::exists(tmpCoverPath))
		KIO::NetAccess::del(KUrl(tmpCoverPath), mTray);
}


void NoatunSystray::setTipText(const QString& text)
{
	if(text == tipText) // save the planet, save cpu cycles ;)
		return;
	tipText = text;

	YHConfig *c = YHConfig::self();
	if(c->passivePopup())
		QTimer::singleShot(0, this, SLOT(showPassivePopup()));

	if(c->tip())
		QToolTip::add(mTray, tipText);
}


void NoatunSystray::showPassivePopup()
{
	if (!mTray)
		return;

	if (!mPassivePopup)
	{
		kDebug(66666) << "Called but no KPassivePopup created yet!";
		return;
	}

	mPassivePopup->reparent(0L, QPoint(0,0));
	if (YHConfig::self()->passivePopupButtons() && !global()->player()->isStopped())
	{
		QVBox *widget = mPassivePopup->standardView(QString(), tipText, QPixmap());
		QHBox *box = new QHBox(mPassivePopup, "popupbox");

		box->setSpacing(8);

		// Algorithm for determining popup location from kpassivepopup.cpp via JuK
		NETWinInfo ni(qt_xdisplay(), mTray->winId(), qt_xrootwin(),
			NET::WMIconGeometry | NET::WMKDESystemTrayWinFor);
		NETRect frame, win;
		ni.kdeGeometry(frame, win);

		QRect bounds = KGlobalSettings::desktopGeometry(QPoint(win.pos.x, win.pos.y));

		if(win.pos.x < bounds.center().x())
		{
			// Buttons to the left

			QVBox *buttonBox = new QVBox(box);
			buttonBox->setSpacing(3);

			QPushButton *forwardButton = new QPushButton(action("player_forward")->iconSet(), 0, buttonBox);
			forwardButton->setFlat(true);
			connect(forwardButton, SIGNAL(clicked()), action("player_forward"), SLOT(activate()));

			QPushButton *backButton = new QPushButton(action("player_back")->iconSet(), 0, buttonBox);
			backButton->setFlat(true);
			connect(backButton, SIGNAL(clicked()), action("player_back"), SLOT(activate()));

			QFrame *line = new QFrame(box);
			line->setFrameShape(QFrame::VLine);

			widget->reparent(box, QPoint(0, 0));
		}
		else
		{
			// Buttons to the right
			widget->reparent(box, QPoint(0, 0));

			QFrame *line = new QFrame(box);
			line->setFrameShape(QFrame::VLine);

			QVBox *buttonBox = new QVBox(box);
			buttonBox->setSpacing(3);

			QPushButton *forwardButton = new QPushButton(action("player_forward")->iconSet(), 0, buttonBox);
			forwardButton->setFlat(true);
			connect(forwardButton, SIGNAL(clicked()), action("player_forward"), SLOT(activate()));

			QPushButton *backButton = new QPushButton(action("player_back")->iconSet(), 0, buttonBox);
			backButton->setFlat(true);
			connect(backButton, SIGNAL(clicked()), action("player_back"), SLOT(activate()));
		}
		mPassivePopup->setView(box);
	}
	else
	{
		mPassivePopup->setView(QString(), tipText);
	}

	mPassivePopup->setTimeout(YHConfig::self()->passivePopupTimeout()*1000);
	mPassivePopup->show();
}


void NoatunSystray::slotIconThemeChanged()
{
	createTrayPixmaps();
	changeTray(currentState);
}

void NoatunSystray::createTrayPixmaps()
{
	trayPixmaps[Base] = renderIcon("noatun", QString());
	trayPixmaps[Stopped] = renderIcon("noatun", "media-playback-stop");
	trayPixmaps[Playing] = renderIcon("noatun", "media-playback-start");
	trayPixmaps[Paused] = renderIcon("noatun", "media-playback-pause");
}


QPixmap NoatunSystray::renderIcon(const QString& baseIcon, const QString &overlayIcon) const
{
	if (overlayIcon.isNull())
		return KSystemTray::loadIcon(baseIcon);

	QImage baseImg = KSystemTray::loadIcon(baseIcon).convertToImage();
	QImage overlayImg = KSystemTray::loadIcon(overlayIcon).convertToImage();
	KIconEffect::overlay(baseImg, overlayImg);

	QPixmap base;
	base.convertFromImage(baseImg);
	return base;
}

#include "systray.moc"
