/* This file is part of Noatun

  Copyright 2000-2006 Charles Samuels <charles@kde.org>
  Copyright 2000-2001 Neil Stevens <neil@qualityassistant.com>
  Copyright 2002 by Ryan Cumming <ryan@kde.org>
  Copyright 2003-2007 by Stefan Gehn <mETz81@web.de>

  Copyright (c) from the patches of:
                2001 Klas Kalass <klas.kalass@gmx.de>
                2001 Anno v. Heimburg <doktor.dos@gmx.de>

  Licensed under the "Artistic License"
*/

// Abandon All Hope, Ye Who Enter Here

#include <q3header.h>
#include <qlayout.h>
#include <qmap.h>
#include <qregexp.h>
#include <qtextstream.h>
#include <qpainter.h>

#include <kaction.h>
#include <kactioncollection.h>
#include <kdebug.h>
#include <kfiledialog.h>
#include <kfileitem.h>
#include <kjobuidelegate.h>
#include <kio/job.h>
#include <kio/netaccess.h>
#include <klocale.h>
#include <kpushbutton.h>
#include <kconfig.h>
#include <kstandarddirs.h>
#include <kstandardaction.h>
#include <kedittoolbar.h>
#include <k3tempfile.h>
#include <kmessagebox.h>
#include <kiconloader.h>

#include <noatun/player.h>
#include <noatun/playlistsaver.h>
#include <noatun/global.h>

#include "playlist.h"
#include "view.h"
#include "find.h"

using namespace Noatun;

#define SPL SplitPlaylist::SPL()

SafeListViewItem::SafeListViewItem(List *parent, Q3ListViewItem *after, const KUrl &url)
	: Q3CheckListItem(parent, 0, Q3CheckListItem::CheckBox)
{
	setOn(true);

	setProperty("url", url.url());
//	mProperties.setAutoDelete(true);

	if (after)
		static_cast<K3ListView*>(parent)->moveItem(this, 0, after);

	PlaylistItem pli(SPL->item(new ItemData(SPL, this)));
	pli.added(SPL->Plugin::global());
}

SafeListViewItem::SafeListViewItem(List *parent, Q3ListViewItem *after, const QMap<QString,QString> &props)
	: Q3CheckListItem(parent, 0, Q3CheckListItem::CheckBox)
{
	setOn(true);

	// A version of setProperty that assumes a key is unique,
	// and doesn't call modified for every new key.
	// Ugly, but this function is a very hot path on playlist loading
	for (QMap<QString,QString>::ConstIterator i=props.begin(); i!=props.end(); ++i )
	{
		const QString &key = i.key();
		const QString &val = i.value();

		if (key == "enabled")
		{
			setOn(val != "false" && val != "0");
		}
		else
		{
			Property p = {key, val};
			mProperties += p;
		}
	}

	if (after)
		static_cast<K3ListView*>(parent)->moveItem(this, 0, after);

	modified();

	PlaylistItem pli(SPL->item(new ItemData(SPL, this)));
	pli.added(SPL->Plugin::global());
}

SafeListViewItem::~SafeListViewItem()
{
	//kDebug(66666) << "For item " << text(0);
}

static void pad(QString &str)
{
	int len=str.length();
	int at = 0;
	int blocklen=0;

	static const int paddingsize=12;

	// not static for reason
	const QChar chars[paddingsize] =
	{
		QChar('0'), QChar('0'), QChar('0'), QChar('0'),
		QChar('0'), QChar('0'), QChar('0'), QChar('0'),
		QChar('0'), QChar('0'), QChar('0'), QChar('0')
	};

	for (int i=0; i < len; i++)
	{
		if (str[i].isNumber())
		{
			if (!blocklen)
				at = i;
			blocklen++;
		}
		else if (blocklen)
		{
			int pads=paddingsize;
			pads -= blocklen;
			str.insert(at, chars, pads);
			i += pads;
			blocklen = 0;
		}
	}
	if (blocklen)
	{
		int pads=paddingsize;
		pads -= blocklen;
		str.insert(at, chars, pads);
	}
}

int SafeListViewItem::compare(Q3ListViewItem * i, int col, bool) const
{
	QString text1 = text(col);
	QString text2 = i->text(col);

	pad(text1);
	pad(text2);
	return text1.compare(text2);
}

QString SafeListViewItem::property(const QString &n, const QString &def) const
{
	for (QList<Property>::ConstIterator i=mProperties.begin(); i != mProperties.end(); ++i)
	{
		if ((*i).key==n)
			return (*i).value;
	}

	if (n == "enabled")
		return (isOn() ? "true" : "false");
	return def;
}

void SafeListViewItem::setProperty(const QString &n, const QString &val)
{
	if (n == "enabled")
	{
		setOn( val != "false" && val != "0");
	}
	else
	{
		if ( property(n,"") == val )
			return;

		clearProperty(n);
		Property p={n,val};
		mProperties += p;
	}
	modified();
}

void SafeListViewItem::clearProperty(const QString &n)
{
	if (n == "enabled")
	{
		setOn(true);
		modified();
	}
	else
	{
		QList<Property>::Iterator      i(mProperties.begin());
		QList<Property>::ConstIterator end(mProperties.end());
		for ( ; i != end; ++i)
		{
			if ((*i).key == n)
			{
				mProperties.erase(i);
				modified();
				break;
			}
		}
	}
}

QStringList SafeListViewItem::properties() const
{
	QStringList list;
	for (QList<Property>::ConstIterator i=mProperties.begin(); i != mProperties.end(); ++i)
		list += (*i).key;
	list += "enabled";
	return list;
}

bool SafeListViewItem::hasProperty(const QString &n) const
{
	for (QList<Property>::ConstIterator i=mProperties.begin(); i != mProperties.end(); ++i)
	{
		if ((*i).key==n)
			return true;
	}
	return n == "enabled";
}

void SafeListViewItem::modified()
{
	PlaylistItem pli = SPL->item(new ItemData(SPL, this));
	const QString titlestring = SPL->Plugin::global()->player()->title(pli);

	setText(0, titlestring);
	setText(1, pli.lengthString());

	pli.modified(SPL->Plugin::global());
}

void SafeListViewItem::stateChange(bool s)
{
	// if you uncheck this, uncheck the others that
	// are selected too
	QList<Q3ListViewItem*> list = static_cast<K3ListView *>(listView())->selectedItems();

	// but not if I'm not selected
	if (list.contains(this))
	{
		Q3ListViewItem *i;
		foreach(i, list)
			static_cast<Q3CheckListItem*>(i)->setOn(s);
	}
	else
	{
		Q3CheckListItem::stateChange(s);
	}
}

void SafeListViewItem::remove()
{
	kDebug(66666) << "For item " << text(0);

	if (listView())
	{
		if (SPL->item(SPL->current())->item() == this)
			SPL->setCurrent(itemBelow(), false);

		listView()->takeItem(this);
	}
	else if (SPL->item(SPL->current())->item() == this)
	{
		SPL->setCurrent(0, false);
	}

#ifdef __GNUC__
#warning TODO
#endif
	//PlaylistItem pli(listView()->item(this));
	//pli.removed(Noatun::Global::self());
}

void SafeListViewItem::showPlaying(bool p)
{
	setPixmap(0, (p ? ::SmallIcon("media-playback-start") : QPixmap()));
}


List::List(View *parent)
	: K3ListView(parent), recursiveAddAfter(0), listJob(0)
{
	addColumn(i18n("File"));
	addColumn(i18n("Time"));
	setAcceptDrops(true);
	setSorting(-1);
	setDropVisualizer(true);
	setDragEnabled(true);
	setItemsMovable(true);
	setSelectionMode(Q3ListView::Extended);
	connect(this, SIGNAL(dropped(QDropEvent*, Q3ListViewItem*)),
		SLOT(dropEvent(QDropEvent*, Q3ListViewItem*)));
	connect(this, SIGNAL(moved()), SLOT(move()));
	connect(this, SIGNAL(aboutToMove()), parent, SLOT(setNoSorting()));
	connect(this, SIGNAL(deleteCurrentItem()), parent, SLOT(deleteSelected()));
}


List::~List()
{
	kDebug(66666) ;
}


void List::move()
{
	emit modified();
}


bool List::acceptDrag(QDropEvent *event) const
{
	if (KUrl::List::canDecode(event->mimeData()))
	{
		event->setAccepted(true);
		return true;
	}

	return K3ListView::acceptDrag(event);
}


void List::dropEvent(QDropEvent *event, Q3ListViewItem *after)
{
	if (!KUrl::List::canDecode(event->mimeData()))
		return;

	KUrl::List urlList = KUrl::List::fromMimeData(event->mimeData());
	if (!urlList.isEmpty())
	{
		static_cast<View*>(parent())->setNoSorting();

		foreach(KUrl u, urlList)
		{
			after = addFile(u, false, after);
		}
		emit modified();
	}
}


void List::keyPressEvent(QKeyEvent *e)
{
	if (e->key()==Qt::Key_Enter || e->key()==Qt::Key_Return)
	{
		if (currentItem())
		{
			emit K3ListView::executed(currentItem());
		}

		return;
	}

	if (e->key()==Qt::Key_Delete)
	{
		if (currentItem())
		{
			emit deleteCurrentItem();
		}

		return;
	}

	K3ListView::keyPressEvent(e);


}

/**
 * use this only once!!!
 **/
class NoatunSaver : public PlaylistSaver
{
	List *mList;
	SafeListViewItem *after, *mFirst;
public:
	NoatunSaver(List *l, Q3ListViewItem *after=0)
		: mList(l)
	{
		this->after = static_cast<SafeListViewItem*>(after);
		mFirst = 0;
	}

	Q3ListViewItem *getAfter() { return after; }
	Q3ListViewItem *getFirst() { return mFirst; }

protected:
	virtual void readItem(const Noatun::PropertyMap &properties)
	{
		after = new SafeListViewItem(mList, after, properties);
		if (mFirst==0)
			mFirst = after;
	}

	virtual PlaylistItem writeItem()
	{
		if (!after)
		{
			after = mList->firstChild();
		}
		else
		{
			after = after->nextSibling();
		}

		if (after)
			return SPL->item(new ItemData(SPL, after));
		else
			return PlaylistItem();
	}
}; // NoatunSaver


bool View::saveToURL(const KUrl &url, const QString &mimeType)
{
	kDebug(66666) << "url: " << url << "; mimeType: " << mimeType;

	NoatunSaver saver(list);
	if (saver.save(url, mimeType))
	{
		return true;
	}
	else
	{
		KMessageBox::error(this, i18n("Could not save playlist to '%1'.", url.prettyUrl()));
		return false;
	}
}


Q3ListViewItem *List::addFile(const KUrl& url, bool play, Q3ListViewItem *after)
{
	// when a new item is added, we don't want to sort anymore
	static_cast<View*>(parent())->setNoSorting();
	Q3ListViewItem *currentItem = 0;
	QString mimeType;

	mimeType = KMimeType::findByUrl(url, 0 /*mode*/,
		url.isLocalFile() /*is local*/, false /*fast mode*/)->name();
	if (mimeType == "application/octet-stream")
	{
		kDebug(66666) << "COULD NOT DETERMINE MIMETYPE FOR URL, ASKING KIO";
		mimeType = KIO::NetAccess::mimetype(url, 0);
	}
	kDebug(66666) << "==> Found MimeType " << mimeType;


	if (PlaylistSaver::loadableMimeTypes().contains(mimeType))
	{
		kDebug(66666) << "Passed KUrl ist supported by PlaylistSaver";
		// it's a playlist
		NoatunSaver saver(this, after);
		if (saver.load(url, mimeType))
		{
			// return the first item added from this playlist
			// that way noatun can start playing the first item
			currentItem = saver.getFirst();

			// failsafe in case nothing was added, getFirst() may return 0
			if (!currentItem)
				currentItem = saver.getAfter();
		}
	}
	else if (SPL->Plugin::global()->player()->mimeTypes().contains(mimeType))
	{
		kDebug(66666) << "Passed KUrl ist supported by Player";
		if (!after)
			after = lastItem();

		KFileItem fileItem(KFileItem::Unknown, KFileItem::Unknown,url);
		if (fileItem.isDir())
		{
			addDirectoryRecursive(url, after);
			currentItem = after; // don't (and can't) know better!?
		}
		else
		{
			currentItem = new SafeListViewItem(this, after, url);
		}
	}
	else
	{
		kDebug(66666) << "Passed KUrl ist unsupported by Noatun";
	}

	if (play && currentItem)
		SPL->listItemSelected(currentItem);
	return currentItem;
}


void List::addDirectoryRecursive(const KUrl &dir, Q3ListViewItem *after)
{
	recursiveAddAfter = (after ? after : lastItem());

	if (dir.upUrl().equals(currentJobURL, KUrl::CompareWithoutTrailingSlash))
	{
		// We are a subdir of our currentJobURL and need to get listed next,
		// NOT after all the other dirs that are on the same level as
		// currentJobURL!
		lastAddedSubDirectory = pendingAddDirectories.insert(lastAddedSubDirectory, dir);
		lastAddedSubDirectory++;
	}
	else
	{
		pendingAddDirectories.append(dir);
	}
	addNextPendingDirectory();
}


// starts a new listJob if there is no active but work to do
void List::addNextPendingDirectory()
{
	KUrl::List::Iterator      pendingIt  = pendingAddDirectories.begin();
	KUrl::List::ConstIterator pendingEnd = pendingAddDirectories.end();

	if (!listJob && (pendingIt != pendingEnd))
	{
		currentJobURL = *pendingIt;
		listJob = KIO::listDir(currentJobURL, false, false);
		connect(
				listJob, SIGNAL(entries(KIO::Job*, const KIO::UDSEntryList&)),
				SLOT(slotEntries(KIO::Job*, const KIO::UDSEntryList&))
			);
		connect(
				listJob, SIGNAL(result(KJob *)),
				SLOT(slotResult(KJob *))
			);
		connect(
				listJob, SIGNAL(redirection(KIO::Job *, const KUrl &)),
				SLOT(slotRedirection(KIO::Job *, const KUrl &))
			);
		pendingAddDirectories.erase(pendingIt);
		lastAddedSubDirectory = pendingAddDirectories.begin();
	}
}

void List::slotResult(KJob *job)
{
	listJob = 0;
	if (job && job->error() && job->uiDelegate())
		job->uiDelegate()->showErrorMessage();
	addNextPendingDirectory();
}

void List::slotEntries(KIO::Job *, const KIO::UDSEntryList &entries)
{
	QMap<QString,KUrl> __list; // temp list to sort entries

	foreach (const KIO::UDSEntry entry, entries)
	{
		KFileItem file(entry, currentJobURL, false, true);
		// "prudhomm:
		// insert the path + url in the map to sort automatically by path
		// note also that you use audiocd to rip your CDs then it will be sorted the right way
		// now it is an easy fix to have a nice sort BUT it is not the best
		// we should sort based on the tracknumber"
		// - copied over from old kdirlister hack <hans_meine@gmx.de>
		__list.insert(file.url().path(), file.url());
	}

	QMap<QString,KUrl>::Iterator __it;
	for( __it = __list.begin(); __it != __list.end(); ++__it )
	{
		recursiveAddAfter= addFile(__it.value(), false, recursiveAddAfter);
	}
}

void List::slotRedirection(KIO::Job *, const KUrl & url)
{
	currentJobURL = url;
}





///////////////////////////////////////////////////////////////////////////////

View::View(Plugin *pl) : Noatun::NMainWindow(pl, false, false)
{
	kDebug(66666) ;

	mFinder = new Finder(this);
	list = new List(this);
	setCentralWidget(list);

	connect(list, SIGNAL(modified()), SLOT(setModified()));
	// connect the click on the header with sorting
	connect(list->header(), SIGNAL(clicked(int)), SLOT(headerClicked(int)));

	// --- "Playlist" menu ---
	mOpenNew = KStandardAction::openNew(this, SLOT(openNew()), actionCollection());
	mOpenpl  = KStandardAction::open(this, SLOT(open()), actionCollection());
	mSave    = KStandardAction::save(this, SLOT(save()), actionCollection());
	mSaveAs  = KStandardAction::saveAs(this, SLOT(saveAs()), actionCollection());

	KAction *addFiles = new KAction(actionCollection());
	addFiles->setText(i18n("Add &Files..."));
	addFiles->setIcon(KIcon("list-add"));
	connect(addFiles, SIGNAL(triggered()), SLOT(addFiles()));
	actionCollection()->addAction("file_add_files", addFiles);

	KAction *addFld = new KAction(actionCollection());
	addFld->setText(i18n("Add Fol&der..."));
	addFld->setIcon(KIcon("folder")); //TODO: folder_add icon needed
	connect(addFld, SIGNAL(triggered()), SLOT(addDirectory()));
	actionCollection()->addAction("file_add_dir", addFld);

	KStandardAction::close(this, SLOT(close()), actionCollection());

	// --- "Edit" menu ---

	mDelete = new KAction(actionCollection());
	mDelete->setText(i18n("Delete"));
	mDelete->setIcon(KIcon("edit-delete"));
	connect(mDelete, SIGNAL(triggered()), SLOT(deleteSelected()));
	actionCollection()->addAction("edit_delete", mDelete);

	KStandardAction::find(this, SLOT(find()), actionCollection());
	KStandardAction::clear(list, SLOT(clear()), actionCollection());

	KAction *shuf = new KAction(actionCollection());
	shuf->setText(i18n("Shuffle"));
	shuf->setIcon(KIcon("misc"));
	connect(shuf, SIGNAL(triggered()), SPL, SLOT(randomize()));
	actionCollection()->addAction("shuffle", shuf);
	// ---

	setStandardToolBarMenuEnabled(true);
	//TODO: use standard name for rc-file
	setupGUI(ToolBar | Keys | Create, "noatun/splui.rc");
	// enable loading/saving if window settings including window-size
	setAutoSaveSettings(QLatin1String("SPL Window"), true);
}


View::~View()
{
	kDebug(66666) ;
	//delete list;
}


void View::saveState(KConfigGroup &grp)
{
	kDebug(66666) ;

	grp.writePathEntry("PlaylistFile", mPlaylistFile.url());
	grp.writeEntry("PlaylistMimeType", mPlaylistMimeType);

	unsigned int i;
	PlaylistItem item = SPL->getFirst();
	for (i = 0; item && item != SPL->current(); )
	{
		item = SPL->getAfter(item), i++;
	}
	grp.writeEntry("CurrentPlaylistIndex", i);

	if (mModified)
	{
		if (KMessageBox::warningYesNoCancel(this,
			i18n("Playlist \"%1\" has unsaved changes.\nSave changes to playlist?",
				mPlaylistFile.fileName()))
			== KMessageBox::Yes)
		{
			save();
		}
	}
}

void View::loadState(const KConfigGroup &grp)
{
	kDebug(66666) ;

	mPlaylistFile     = KUrl(grp.readPathEntry("PlaylistFile"));
	mPlaylistMimeType = grp.readEntry("PlaylistMimeType");

	if (mPlaylistFile.isEmpty() || !mPlaylistFile.isValid())
		return;

	list->clear();
	NoatunSaver playlistLoader(list);
	playlistLoader.load(mPlaylistFile, mPlaylistMimeType);
	setModified(false);

	SPL->reset();
	unsigned int savedIndex = grp.readEntry("CurrentPlaylistIndex", 0);
	PlaylistItem item = SPL->getFirst();
	for (unsigned int i = 0 ; i < savedIndex ; i++)
	{
		item = SPL->getAfter(item);
	}
	if (item)
		SPL->setCurrent(item);
}


void View::find()
{
	mFinder->show();
	connect(mFinder, SIGNAL(search(Finder*)), SLOT(findIt(Finder*)));
}


static bool testWord(PlaylistItem &item, const QString &finder)
{
	if (item.property("author").contains(finder, Qt::CaseInsensitive))
		return true;
	if (item.property("title").contains(finder, Qt::CaseInsensitive))
		return true;
	if (item.property("album").contains(finder, Qt::CaseInsensitive))
		return true;
	if (item.property("comment").contains(finder, Qt::CaseInsensitive))
		return true;
	if (item.url().path().contains(finder, Qt::CaseInsensitive))
		return true;

	return false;
}


static bool testWord(PlaylistItem &item, const QRegExp &finder)
{
	if (item.property("author").contains(finder))
		return true;
	if (item.property("title").contains(finder))
		return true;
	if (item.property("album").contains(finder))
		return true;
	if (item.property("comment").contains(finder))
		return true;
	if (item.url().path().contains(finder))
		return true;

	return false;
}


void View::findIt(Finder *f)
{
	SafeListViewItem *search = static_cast<SafeListViewItem*>(list->currentItem());

	if (search)
	{
		if (f->isForward())
			search = search->itemBelow();
		else
			search = search->itemAbove();
	}
	else
	{
		if (f->isForward())
			search = list->firstChild();
		else
			search = static_cast<SafeListViewItem*>(list->lastChild());
	}


	while (search)
	{
		PlaylistItem item = SPL->item(new ItemData(SPL, search));
		if (f->regexp())
		{
			if (testWord(item, QRegExp(f->string(), Qt::CaseInsensitive)))
				break;
		}
		else
		{
			if (testWord(item, f->string()))
				break;
		}

		if (f->isForward())
			search = search->itemBelow();
		else
			search = search->itemAbove();

		if (!search)
		{
			if (f->isForward())
			{
				if (KMessageBox::questionYesNo(this, i18n("End of playlist reached. Continue searching from beginning?")) == KMessageBox::Yes)
					search=list->firstChild();
			}
			else
			{
				if (KMessageBox::questionYesNo(this, i18n("Beginning of playlist reached. Continue searching from end?")) == KMessageBox::Yes)
					search = static_cast<SafeListViewItem*>(list->lastChild());
			}
		}
	}

	if (search)
	{
		{ // select none
			QList<Q3ListViewItem*> sel=list->selectedItems();
			Q3ListViewItem *i;
			foreach (i, sel)
				list->setSelected(i, false);
		}
		list->setSelected(search, true);
		list->setCurrentItem(search);
		list->ensureItemVisible(search);
	}
}


void View::save()
{
	if (mPlaylistFile.isEmpty() || !mPlaylistFile.isValid())
	{
		saveAs();
	}
	else
	{
		if (saveToURL(mPlaylistFile, mPlaylistMimeType))
			setModified(false);
	}
}


void View::saveAs()
{
	const QStringList saveTypes = PlaylistSaver::savableMimeTypes();

	KFileDialog dlg(KUrl("kfiledialog:///splitplaylistdir"), QString(), this);
	dlg.setOperationMode(KFileDialog::Saving);
	dlg.setWindowTitle(i18n("Save Playlist"));
	dlg.setMode(KFile::File);
	dlg.setMimeFilter(saveTypes, saveTypes.first());

	if (dlg.exec() == QDialog::Accepted && dlg.selectedUrl().isValid())
	{
		mPlaylistFile     = dlg.selectedUrl();
		mPlaylistMimeType = dlg.currentFilterMimeType()->name();
		save();
	}
}


void View::open()
{
	KFileDialog dlg(KUrl("kfiledialog:///splitplaylistdir"), QString(), this);
	dlg.setOperationMode(KFileDialog::Opening);
	dlg.setWindowTitle(i18n("Open Playlist"));
	dlg.setMode(KFile::File | KFile::ExistingOnly);
	dlg.setMimeFilter(PlaylistSaver::loadableMimeTypes());

	if (dlg.exec() == QDialog::Accepted && dlg.selectedUrl().isValid())
	{
		kDebug(66666) << "selected " << dlg.selectedUrl();

		list->clear();
		NoatunSaver playlistLoader(list);
		if (playlistLoader.load(dlg.selectedUrl()))
		{
			mPlaylistFile     = dlg.selectedUrl();
			mPlaylistMimeType = playlistLoader.lastMimeType();
		}
		setModified(false);
	}
}


void View::openNew()
{
	mPlaylistFile.clear();
	mPlaylistMimeType.clear();
	listView()->clear();
	setModified(false);
}


void List::clear()
{
	kDebug(66666) << "Clearing all items";
	SPL->setCurrent(0, false);

	// We cannot use QListView::clear() in here as this would delete all
	// PlaylistItemData and produce PlaylistItem objects with dangling mData
	// pointers around!!!
	clearSelection();

	SafeListViewItem *currentItem = 0;
	SafeListViewItem *nextItem = static_cast<SafeListViewItem*>(firstChild());
	while (nextItem)
	{
		currentItem = nextItem;
		nextItem = static_cast<SafeListViewItem*>(currentItem->itemBelow());
		currentItem->remove();
	}
	triggerUpdate();
}


void View::deleteSelected()
{
	QList<Q3ListViewItem*> items(list->selectedItems());

	bool stopped=false;
	// noatun shouldn't play files for now
	SafeListViewItem *afterLast=0;

	for (QList<Q3ListViewItem*>::Iterator it(items.begin()); it != items.end(); ++it)
	{
		SafeListViewItem *i = static_cast<SafeListViewItem*>(*it);
		if (!stopped && SPL->item(SPL->current())->item() == i)
		{
			SPL->Plugin::global()->player()->stop();
			SPL->setCurrent(0, false);
			stopped = true;
		}
		i->remove();

		afterLast = i->itemBelow();
	}

	if (stopped)
		SPL->setCurrent(afterLast, false);

	setModified(true);
}


void View::addFiles()
{
	const QStringList types = Noatun::PlaylistSaver::loadableMimeTypes() +
		SPL->Plugin::global()->player()->mimeTypes();

	KFileDialog dlg(KUrl("kfiledialog:///splitplaylistdir"), QString(), this);
	dlg.setOperationMode(KFileDialog::Opening);
	dlg.setWindowTitle(i18n("Select File(s) to Add"));
	dlg.setMode(KFile::Files);
	dlg.setMimeFilter(types);
	dlg.okButton()->setText(i18n("&Add"));

	dlg.exec();

	foreach(const KUrl &u, dlg.selectedUrls())
		addFile(u, false);

	setModified(true);
}

void View::addDirectory()
{
	QString folder = KFileDialog::getExistingDirectory(KUrl(), this,
		i18n("Select folder to add"));

	if (folder.isEmpty())
		return;

	KUrl url;
	url.setPath(folder);
	list->addDirectoryRecursive(url);

	setModified(true);
}


void View::setModified(bool b)
{
	mModified = b;
	setCaption(mPlaylistFile.prettyUrl(), mModified);
}


void View::setModified(void)
{
	setModified(true);
}


void View::configureToolBars()
{
	kDebug(66666) ;
	KConfigGroup cg(KGlobal::config(), "SPL Window");
	saveMainWindowSettings(cg);

	KEditToolBar dlg(guiFactory(), this);
	connect(&dlg, SIGNAL(newToolbarConfig()), SLOT(newToolBarConfig()));
	dlg.exec();
}


void View::newToolBarConfig()
{
	kDebug(66666) ;
	createGUI("noatun/splui.rc");
	applyMainWindowSettings(KGlobal::config()->group("SPL Window"));
}


// turns the sorting on or off
void View::setSorting(bool on, int column)
{
	if (on)
	{
		list->setSorting(column,true);
		list->setShowSortIndicator(true);
	}
	else
	{
		list->setShowSortIndicator(false);
		list->setSorting(-1);
	}
}


void View::headerClicked(int column)
{
	// this is to avoid that if we already have it sorted,
	// we sort it again ascendingly this way, clicking on
	// the header a second time will correctly toggle
	// ascending/descending sort
	if (list->showSortIndicator())
		return;
	setSorting(true,column);
}

#include "view.moc"

