/* This file is part of Noatun

  Copyright 2001-2006 by Charles Samuels <charles@kde.org>
  Copyright 2002 by Stefan Westerfeld <stefan@space.twc.de>
  Copyright 2003-2007 by Stefan Gehn <mETz81@web.de>

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions
  are met:

  1. Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.
  2. Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.

  THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
  IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
  OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
  NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#ifndef N_PLAYLIST_H
#define N_PLAYLIST_H

#include <kurl.h>

#include <noatun/playlistitem.h>
#include <noatun/noatun_export.h>

class QStringList;
class KXmlGuiWindow;

namespace Noatun
{
class Global;
class PlaylistPrivate;
class PlaylistInterfaceGeneric;
class PlaylistItem;
class PlaylistNotifier;
class Plugin;


// ----------------------------------------------------------------------------


/**
 * @brief Central playlist object
 *
 * This object is instantiated in class Global. Use Global::playlist() to get
 * a pointer to it.
 **/
class NOATUN_EXPORT Playlist : public QObject
{
Q_OBJECT
	friend class PlaylistInterfaceGeneric; // Needed to emit Playlist signals from PlaylistInterface
	friend class PlaylistItem;             // Needs access to notifiers
	friend class PlaylistNotifier;         // Needs access to notifiers

public:
	explicit Playlist(Global *parent);
	~Playlist();

	KXmlGuiWindow *mainWindow() const;

	/**
	 * @brief Add a list of files/urls to the playlist
	 **/
	void addFile(const KUrl::List &f, bool purgeList = false, bool autoplay = false);
	/**
	 * @brief Add a single file/url to the playlist
	 **/
	void addFile(const KUrl &url, bool purgeList = false, bool autoplay = false);

	PlaylistItem getFirst() const;
	PlaylistItem getAfter(const PlaylistItem &item) const;

	bool listVisible() const;

	QList<PlaylistItem> select(const QStringList &keys,
		const QStringList &values, int limit = -1, bool exact = false,
		bool caseSensitive=false );

	QList<PlaylistItem> select(const QString &key, const QString &value,
		int limit = -1, bool exact = false, bool caseSensitive=false);

	/*QList<PlaylistItem> select(const char *key, const char *value,
		int limit = -1, bool exact = false, bool caseSensitive = false)
	{
		return select(QString(key), QString(value), limit, exact, caseSensitive);
	}*/

	/**
	 * returns a friendly representation of the length
	 * of this item
	 **/
	QString lengthString(const PlaylistItem &data) const;


public Q_SLOTS:
	/**
	 * Shows the playlist-window
	 */
	void showList();
	/**
	 * Hides the playlist-window
	 */
	void hideList();
	/**
	 * toggle visibility
	 */
	void toggleList();

Q_SIGNALS:
	/**
	 * Emitted when the list is hidden
	 **/
	void listHidden();

	/**
	 * Emitted when the list is shown
	 **/
	void listShown();

	void newPlaylist();

private Q_SLOTS:
	void slotPluginLoaded(Plugin *);
	void slotPluginUnloaded(Plugin *plug);

protected:
	bool eventFilter(QObject *o, QEvent *e);

protected:
	QList<PlaylistNotifier *> *notifiers();

private:
	PlaylistPrivate * const d;
}; // END class Playlist


/**
 * this class's methods will be called whenever
 * something happens to the playlist or its
 * items.
 *
 * If the playlist plugin changes, you don't have to do
 * anything.
 **/
class NOATUN_EXPORT PlaylistNotifier
{
public:
	explicit PlaylistNotifier(Global *glob);
	virtual ~PlaylistNotifier();

	/**
	 * a new item is added to the playlist.
	 * @p item contains a reference to the item that was added.
	 **/
	virtual void added(PlaylistItem &item);

	/**
	 * The passed item was removed from the playlist.
	 * @p item contains a reference to the item that was added.
	 **/
	virtual void removed(PlaylistItem &item);

	/**
	 * The passed item was modified (usually due to a changed,
	 * removed or added property).
	 * @p item contains a reference to the item that was added.
	 **/
	virtual void modified(PlaylistItem &item);

private:
	Noatun::Global *mGlobal;
}; // END class PlaylistNotifier

} // END namespace Noatun
#endif // N_PLAYLIST_H
