/* This file is part of Noatun

  Copyright 2004-2007 by Stefan Gehn <mETz81@web.de>
  Copyright 2005-2006 by Charles Samuels <charles@kde.org>

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions
  are met:

  1. Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.
  2. Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.

  THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
  IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
  OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
  NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#ifndef NOATUN_GLOBAL_H
#define NOATUN_GLOBAL_H

#include <QtCore/QObject>
#include <noatun/noatun_export.h>

class KComponentData;

namespace Noatun
{
class Frontend;
class Player;
class Playlist;
class Plugin;
class PluginHandler;
class PreferencesDialog;
class Config;

/**
 * @brief Central point that keeps Noatun tied together
 * @author Stefan Gehn
 * @todo rename to NInstance, the Noatun-counterpart of KInstance ;)
 * @todo Revise shutdown handling so that calling QApplication::quit() would
 *       work as expected. This is probably also needed for proper session
 *       management.
 **/
class NOATUN_EXPORT Global : public QObject
{
Q_OBJECT

public:
	/**
	 * @brief Constructor
	 *
	 * parent ideally should be a KApplication object
	 **/
	Global(QObject *parent, const KComponentData &componentData);

	/**
	 * @brief Destructor
	 *
	 * You shouldn't have to delete this object manually as long as it has a
	 * proper parent object.
	 **/
	~Global();

	/**
	 * @brief Central playlist object
	 **/
	Playlist *playlist() const;
	/**
	 * @brief Central userinterface object
	 **/
	Frontend *frontend() const;
	/**
	 * @brief Central player object
	 **/
	Player *player() const;
	/**
	 * @brief Handler for finding, loading and unloading plugins
	 **/
	PluginHandler *pluginHandler() const;
	/**
	 * @brief global plugin-independent configuration settings
	 *
	 * Not const because of lazy initialization
	 **/
	Config *config();
	/**
	 * @brief Preferences Dialog
	 *
	 * Not const because of lazy initialization
	 **/
	PreferencesDialog *preferencesDialog();

Q_SIGNALS:
	/**
	 * @brief Emitted when all plugins have been unloaded
	 *
	 * You can safely delete this NInstance after having received this signal
	 **/
	void shutdown();

public Q_SLOTS:
	/**
	 * @brief Quits this noatun-instance
	 * @note Only use from applications but not from a KPart
	 *
	 * A simple way to invoke this method is creating a standard KAction:
	 * @code
	 * Noatun::StdAction::quit(actionCollection());
	 * @endcode
	 **/
	void quit();

private:
	class Private;
	Private * const d;
}; // class NInstance

/**
 * @brief Format a duration honoring current locale
 *
 * Output examples:
 * "1 day 12:02:55"
 * "23:57:03"
 * "02:15"
 *
 * @param duration Duration in milliseconds, negative values are allowed too
 */
NOATUN_EXPORT QString formatDuration(int duration);

} // namespace Noatun
#endif
