/*  This file is part of the QbtGStreamer project, a Qt GStreamer Wrapper
    Copyright (C) 2006 Tim Beaulen <tbscope@gmail.com>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License version 2 as published by the Free Software Foundation.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/

#ifndef QBTGSTREAMER
#define QBTGSTREAMER

#include <gst/gst.h>
#include "qbtgstreamer_export.h"
#include <QString>

class QBTGSTREAMER_EXPORT QbtGStreamer
{
public:
    QbtGStreamer();
    virtual ~QbtGStreamer();

    /*!
     \brief Initialise GStreamer
     
     Initializes the GStreamer library, setting up internal path lists, registering built-in elements, and loading standard plugins.
     
     Always call this function first before using GStreamer.
     This function will terminate your program if it was unable to initialize GStreamer for some reason.
     
     \param argc The argument count of the application.
     \param argv List of parameters of the application.
     */
    static void init(int argc, char *argv[]);

    /*!
     \brief Check the initialisation of GStreamer
     
     Initializes the GStreamer library, setting up internal path lists, registering built-in elements, and loading standard plugins.
     
     Always call this function first before using GStreamer.
     This function will return false if it was unable to initialize GStreamer for some reason.
     
     \param argc The argument count of the application.
     \param argv List of parameters of the application.
     \return True if GStreamer could be initialised
     */
    static bool initCheck(int argc, char *argv[]);

    /*!
     \brief Deinitialise GStreamer
     
     Clean up any resources created by GStreamer in gst_init(). 
     
     It is normally not needed to call this function in a normal application as the resources will automatically be freed when the program terminates. This function is therefore mostly used by testsuites and other memory profiling tools. 
     
     After this call GStreamer (including this method) should not be used anymore.
     */
    static void deInit();

    /*!
     \brief GStreamer version number
     
     Gets the version number of the GStreamer library.
     
     \param major returns the major version number.
     \param minor returns the minor version number.
     \param micro returns the micro version number.
     \param nano returns the nano version number, actual releases have 0, CVS versions have 1, prerelease versions have 2.
     */
    static void version(uint &major, uint &minor, uint &micro, uint &nano);

    /*!
     \brief GStreamer version as a string
     
     This function returns a string that is useful for describing this version of GStreamer to the outside world: user agent strings, logging, ...
     
     \return A string describing the GStreamer version.
     */
    static QString versionString();
};

#endif
