/***************************************************************************
 * form.cpp
 * This file is part of the KDE project
 * copyright (C)2006-2007 by Sebastian Sauer (mail@dipe.org)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 * You should have received a copy of the GNU Library General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 ***************************************************************************/

#include "form.h"

#include <QtCore/QByteRef>
#include <QtCore/QBuffer>
#include <QtCore/QRegExp>
#include <QtCore/QFile>
#include <QtCore/QArgument>
#include <QtCore/QMetaEnum>
#include <QtGui/QKeyEvent>
#include <QtGui/QDialog>
#include <QtGui/QBoxLayout>
#include <QtGui/QStackedLayout>
#include <QtGui/QSizePolicy>
#include <QtGui/QApplication>
#include <QtGui/QProgressDialog>
#include <QUiLoader>
#include <QtDesigner/QFormBuilder>

#include <kdebug.h>
#include <kurl.h>
#include <kpushbutton.h>
//#include <kurlcombobox.h>
//#include <kdiroperator.h>
//#include <kshell.h>
#include <kicon.h>
#include <kaction.h>
#include <kactioncollection.h>
#include <kmessagebox.h>
#include <klibloader.h>
#include <kparts/part.h>
//#include <kio/netaccess.h>
//#include <klocale.h>
//#include <kmimetype.h>
//#include <kstandarddirs.h>
#include <kfilewidget.h>
#include <kurlcombobox.h>
#include <kshell.h>

extern "C"
{
    KDE_EXPORT QObject* krossmodule()
    {
        return new Kross::FormModule();
    }
}

using namespace Kross;

/*********************************************************************************
 * FormDialog
 */

namespace Kross {

    /// \internal d-pointer class.
    class FormFileWidget::Private
    {
        public:
            KFileWidget* filewidget;
    };

}

FormFileWidget::FormFileWidget(QWidget* parent, const QString& startDirOrVariable)
    : QWidget(parent), d(new Private())
{
    QVBoxLayout* layout = new QVBoxLayout(this);
    layout->setSpacing(0);
    layout->setMargin(0);
    setLayout(layout);

    d->filewidget = new KFileWidget(KUrl(startDirOrVariable), this);
    layout->addWidget( d->filewidget );
    //QMetaObject::invokeMethod(d->filewidget, "toggleSpeedbar", Q_ARG(bool,false));
    //KFileDialog::setMode( KFile::File | KFile::LocalOnly );

    QObject::connect(d->filewidget, SIGNAL(fileSelected(const QString&)), this, SIGNAL(fileSelected(const QString&)));
    QObject::connect(d->filewidget, SIGNAL(fileHighlighted(const QString&)), this, SIGNAL(fileHighlighted(const QString&)));
    QObject::connect(d->filewidget, SIGNAL(selectionChanged()), this, SIGNAL(selectionChanged()));
    QObject::connect(d->filewidget, SIGNAL(filterChanged(const QString&)), this, SIGNAL(filterChanged(const QString&)));

//     d->impl->setOperationMode(d->mode);
//     if( d->mimeFilter.count() > 0 )
//         d->impl->setMimeFilter(d->mimeFilter);
//     else if( ! d->filter.isEmpty() )
//         d->impl->setFilter(d->filter);

    if( parent && parent->layout() )
        parent->layout()->addWidget(this);
    setMinimumSize( QSize(480,360) );
}

FormFileWidget::~FormFileWidget()
{
    delete d;
}

void FormFileWidget::setMode(const QString& mode)
{
    QMetaEnum e = metaObject()->enumerator( metaObject()->indexOfEnumerator("Mode") );
    KFileWidget::OperationMode m = (KFileWidget::OperationMode) e.keysToValue( mode.toLatin1() );
    d->filewidget->setOperationMode(m);
}

QString FormFileWidget::currentFilter() const
{
    return d->filewidget->currentFilter();
}

void FormFileWidget::setFilter(const QString &filter)
{
    QString f = filter;
    f.replace(QRegExp("([^\\\\]{1,1})/"), "\\1\\/"); // escape '/' chars else KFileDialog assumes they are mimetypes :-/
    d->filewidget->setFilter(f);
}

QString FormFileWidget::currentMimeFilter() const
{
    return d->filewidget->currentMimeFilter();
}

void FormFileWidget::setMimeFilter(const QStringList& filter)
{
    d->filewidget->setMimeFilter(filter);
}

QString FormFileWidget::selectedFile() const
{
    KUrl selectedUrl;
    QString locationText = d->filewidget->locationEdit()->currentText();
    if( locationText.contains( '/' ) ) { // relative path? -> prepend the current directory
        KUrl u( d->filewidget->baseUrl(), KShell::tildeExpand(locationText) );
        selectedUrl = u.isValid() ? u : selectedUrl = d->filewidget->baseUrl();
    }
    else { // simple filename -> just use the current URL
        selectedUrl = d->filewidget->baseUrl();
    }
    QFileInfo fi( selectedUrl.path(), d->filewidget->locationEdit()->currentText() );
    return fi.absoluteFilePath();
}

/*********************************************************************************
 * FormDialog
 */

namespace Kross {

    /// \internal d-pointer class.
    class FormDialog::Private
    {
        public:
            KDialog::ButtonCode buttoncode;
            QHash<QString, KPageWidgetItem*> items;
    };

}

FormDialog::FormDialog(const QString& caption)
    : KPageDialog( /*0, Qt::WShowModal | Qt::WDestructiveClose*/ )
    , d( new Private() )
{
    setCaption(caption);
    KDialog::setButtons(KDialog::Ok);
    setSizePolicy( QSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding) );

    connect(this, SIGNAL(currentPageChanged(KPageWidgetItem*,KPageWidgetItem*)),
            this, SLOT(slotCurrentPageChanged(KPageWidgetItem*)));
}

FormDialog::~FormDialog()
{
    delete d;
}

bool FormDialog::setButtons(const QString& buttons)
{
    int i = metaObject()->indexOfEnumerator("ButtonCode");
    Q_ASSERT( i >= 0 );
    QMetaEnum e = metaObject()->enumerator(i);
    int v = e.keysToValue( buttons.toUtf8() );
    if( v < 0 )
        return false;
    KDialog::setButtons( (KDialog::ButtonCode) v );
    return true;
}

bool FormDialog::setFaceType(const QString& facetype)
{
    int i = KPageView::staticMetaObject.indexOfEnumerator("FaceType");
    Q_ASSERT( i >= 0 );
    QMetaEnum e = KPageView::staticMetaObject.enumerator(i);
    int v = e.keysToValue( facetype.toUtf8() );
    if( v < 0 )
        return false;
    KPageDialog::setFaceType( (KPageDialog::FaceType) v );
    return true;
}

QString FormDialog::currentPage() const
{
    KPageWidgetItem* item = KPageDialog::currentPage();
    return item ? item->name() : QString();
}

bool FormDialog::setCurrentPage(const QString& name)
{
    if( ! d->items.contains(name) )
        return false;
    KPageDialog::setCurrentPage( d->items[name] );
    return true;
}

QWidget* FormDialog::page(const QString& name) const
{
    return d->items.contains(name) ? d->items[name]->widget() : 0;
}

QWidget* FormDialog::addPage(const QString& name, const QString& header, const QString& iconname)
{
    QWidget* widget = new QWidget( mainWidget() );
    QVBoxLayout* boxlayout = new QVBoxLayout(widget);
    boxlayout->setSpacing(0);
    boxlayout->setMargin(0);
    widget->setLayout(boxlayout);

    KPageWidgetItem* item = KPageDialog::addPage(widget, name);
    item->setHeader(header.isNull() ? name : header);
    if( ! iconname.isEmpty() )
        item->setIcon( KIcon(iconname) );
    d->items.insert(name, item);

    return item->widget();
}

QString FormDialog::result()
{
    int i = metaObject()->indexOfEnumerator("ButtonCode");
    if( i < 0 ) {
        kWarning() << "Kross::FormDialog::setButtons No such enumerator \"ButtonCode\"";
        return QString();
    }
    QMetaEnum e = metaObject()->enumerator(i);
    return e.valueToKey(d->buttoncode);
}

void FormDialog::slotButtonClicked(int button)
{
    d->buttoncode = (KDialog::ButtonCode) button;
    KDialog::slotButtonClicked(button);
}

void FormDialog::slotCurrentPageChanged(KPageWidgetItem* current)
{
    Q_UNUSED(current);
    //kDebug() << "FormDialog::slotCurrentPageChanged current=" << current->name();
    //foreach(QWidget* widget, current->widget()->findChildren< QWidget* >("")) widget->setFocus();
}

/*********************************************************************************
 * FormModule
 */

namespace Kross {

    /// \internal extension of the QUiLoader class.
    class UiLoader : public QUiLoader
    {
        public:
            UiLoader() : QUiLoader() {}
            virtual ~UiLoader() {}

            /*
            virtual QAction* createAction(QObject* parent = 0, const QString& name = QString())
            {
            }

            virtual QActionGroup* createActionGroup(QObject* parent = 0, const QString& name = QString())
            {
            }

            virtual QLayout* createLayout(const QString& className, QObject* parent = 0, const QString& name = QString())
            {
            }

            virtual QWidget* createWidget(const QString& className, QWidget* parent = 0, const QString& name = QString())
            {
            }
            */
    };

    /// \internal d-pointer class.
    class FormModule::Private
    {
        public:
    };

}

FormModule::FormModule()
    : QObject()
    , d( new Private() )
{
}

FormModule::~FormModule()
{
    delete d;
}

QWidget* FormModule::activeModalWidget()
{
    return QApplication::activeModalWidget();
}

QWidget* FormModule::activeWindow()
{
    return QApplication::activeWindow();
}

QString FormModule::showMessageBox(const QString& dialogtype, const QString& caption, const QString& message, const QString& details)
{
    KMessageBox::DialogType type;
    if(dialogtype == "Error") {
        if( ! details.isNull() ) {
            KMessageBox::detailedError(0, message, details, caption);
            return QString();
        }
        type = KMessageBox::Error;
    }
    else if(dialogtype == "Sorry") {
        if( ! details.isNull() ) {
            KMessageBox::detailedSorry(0, message, details, caption);
            return QString();
        }
        type = KMessageBox::Sorry;
    }
    else if(dialogtype == "QuestionYesNo") type = KMessageBox::QuestionYesNo;
    else if(dialogtype == "WarningYesNo") type = KMessageBox::WarningYesNo;
    else if(dialogtype == "WarningContinueCancel") type = KMessageBox::WarningContinueCancel;
    else if(dialogtype == "WarningYesNoCancel") type = KMessageBox::WarningYesNoCancel;
    else if(dialogtype == "QuestionYesNoCancel") type = KMessageBox::QuestionYesNoCancel;
    else /*if(dialogtype == "Information")*/ type = KMessageBox::Information;
    switch( KMessageBox::messageBox(0, type, message, caption) ) {
        case KMessageBox::Ok: return "Ok";
        case KMessageBox::Cancel: return "Cancel";
        case KMessageBox::Yes: return "Yes";
        case KMessageBox::No: return "No";
        case KMessageBox::Continue: return "Continue";
        default: break;
    }
    return QString();
}

QWidget* FormModule::showProgressDialog(const QString& caption, const QString& labelText)
{
    QProgressDialog* progress = new QProgressDialog();
    //progress->setWindowModality(Qt::WindowModal);
    progress->setModal(true);
    progress->setWindowTitle(caption);
    progress->setLabelText(labelText);
    progress->setAutoClose(true);
    progress->setAutoReset(true);
    progress->setCancelButtonText(QString());
    progress->setMinimumWidth(300);
    progress->setSizePolicy(QSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding));
    progress->show();
    return progress;
}

QWidget* FormModule::createDialog(const QString& caption)
{
    return new FormDialog(caption);
}

QObject* FormModule::createLayout(QWidget* parent, const QString& layout)
{
    QLayout* l = 0;
    if( layout == "QVBoxLayout" )
        l = new QVBoxLayout();
    else if( layout == "QHBoxLayout" )
        l = new QHBoxLayout();
    else if( layout == "QStackedLayout" )
        l = new QStackedLayout();
    if( parent && l )
        parent->setLayout(l);
    return l;
}

QWidget* FormModule::createWidget(const QString& className)
{
    UiLoader loader;
    QWidget* widget = loader.createWidget(className);
    return widget;
}

QWidget* FormModule::createWidget(QWidget* parent, const QString& className, const QString& name)
{
    UiLoader loader;
    QWidget* widget = loader.createWidget(className, parent, name);
    if( parent && parent->layout() )
        parent->layout()->addWidget(widget);
    return widget;
}

QWidget* FormModule::createWidgetFromUI(QWidget* parent, const QString& xml)
{
    QFormBuilder builder;
    QByteArray ba = xml.toUtf8();
    QBuffer buffer(&ba);
    buffer.open(QIODevice::ReadOnly);
    QWidget* widget = builder.load(&buffer, parent);
    if( widget && parent && parent->layout() )
        parent->layout()->addWidget(widget);
    return widget;
}

QWidget* FormModule::createWidgetFromUIFile(QWidget* parent, const QString& filename)
{
    QFile file(filename);
    if( ! file.exists() ) {
        kDebug() << QString("Kross::FormModule::createWidgetFromUIFile: There exists no such file \"%1\"").arg(filename);
        return false;
    }
    if( ! file.open(QFile::ReadOnly) ) {
        kDebug() << QString("Kross::FormModule::createWidgetFromUIFile: Failed to open the file \"%1\"").arg(filename);
        return false;
    }
    const QString xml = file.readAll();
    file.close();
    return createWidgetFromUI(parent, xml);
}

QWidget* FormModule::createFileWidget(QWidget* parent, const QString& startDirOrVariable)
{
    FormFileWidget* widget = new FormFileWidget(parent, startDirOrVariable);
    if( parent && parent->layout() )
        parent->layout()->addWidget(widget);
    return widget;
}

QObject* FormModule::loadPart(QWidget* parent, const QString& name, const QUrl& url)
{
    //name e.g. "libkghostview"
    KLibFactory* factory = KLibLoader::self()->factory( name.toLatin1() );
    if( ! factory ) {
        kWarning() << QString("Kross::FormModule::loadPart: No such library \"%1\"").arg(name);
        return 0;
    }
    KParts::ReadOnlyPart* part = factory->create< KParts::ReadOnlyPart >( parent );
    if( ! part ) {
        kWarning() << QString("Kross::FormModule::loadPart: Library \"%1\" is not a KPart").arg(name);
        return 0;
    }
    if( url.isValid() )
        part->openUrl(url);
    if( parent && parent->layout() && part->widget() )
        parent->layout()->addWidget( part->widget() );
    return part;
}

#include "form.moc"
