/* This file is part of the KDE libraries
    Copyright (C) 2001,2002 Ellis Whitehead <ellis@kde.org>
    Copyright (C) 2006 Hamish Rodda <rodda@kde.org>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/

#ifndef _KGLOBALACCEL_H_
#define _KGLOBALACCEL_H_

#include <kdeui_export.h>
#include "kaction.h"

#include <QtCore/QObject>

class QWidget;
class KShortcut;

/**
 * @short Configurable global shortcut support
 *
 * KGlobalAccel allows you to have global accelerators that are independent of
 * the focused window.  Unlike regular shortcuts, the application's window does not need focus
 * for them to be activated.
 *
 * @see KKeyChooser
 * @see KKeyDialog
 */
class KDEUI_EXPORT KGlobalAccel : public QObject
{
    friend class KGlobalAccelImpl;
    Q_OBJECT

public:
    /**
     * Returns (and creates if necessary) the singleton instance
     */
    static KGlobalAccel *self();

    /**
     * Checks whether the accelerators are enabled.
     * @return true if the KGlobalAccel is enabled
     */
    bool isEnabled();

    /**
     * Enables or disables the KGlobalAccel
     * @param enabled true if the KGlobalAccel should be enabled, false if it
     *  should be disabled.
     */
    void setEnabled(bool enabled);

    /**
     * Return the name of the action that uses the given key sequence. This applies to
     * all actions with global shortcuts in any KDE application.
     *
     * @see promptStealShortcutSystemwide(), stealShorctutSystemwide()
     */
    static QStringList findActionNameSystemwide(const QKeySequence &seq);

    /**
     * Show a messagebox to inform the user that a global shorcut is already occupied,
     * and ask to take it away from its current action. This is GUI only, so nothing will
     * be actually changed.
     *
     * @see stealShorctutSystemwide()
     */
    static bool promptStealShortcutSystemwide(QWidget *parent, const QStringList &actionIdentifier, const QKeySequence &seq);

    /**
     * Take away the given shortcut from the named action it belongs to.
     * This applies to all actions with global shortcuts in any KDE application.
     *
     * @see promptStealShortcutSystemwide()
     */
    static void stealShortcutSystemwide(const QKeySequence &seq);

private:
    friend class KAction;
    ///Propagate any shortcut changes to the KDED module that does the bookkeeping
    ///and the key grabbing.
    ///If this is called with an action that has an empty active global shortcut and
    ///an empty default shortcut, the record of that action will be deleted.
    void updateGlobalShortcut(KAction *action, /*KAction::ShortcutTypes*/uint flags);

    ///Register or unregister the action in this class, and notify the KDED module
    void updateGlobalShortcutAllowed(KAction *action, /*KAction::ShortcutTypes*/uint flags);

    QList<int> intListFromShortcut(const KShortcut &cut);
    KShortcut shortcutFromIntList(const QList<int> &list);

//slots
private Q_SLOTS:
    void invokeAction(const QStringList &actionId);
    void shortcutGotChanged(const QStringList &actionId,
                            const QList<int> &shortcut);

private:
    /// Creates a new KGlobalAccel object
    KGlobalAccel();

    /// Destructor
    ~KGlobalAccel();

    class KGlobalAccelPrivate *const d;
};

#endif // _KGLOBALACCEL_H_
