/**
 * Copyright (C) 2000-2002 the KGhostView authors. See file AUTHORS.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#include <assert.h>
#include <stdlib.h>
#include <errno.h>
#include <kaction.h>
#include <kactioncollection.h>
#include <kapplication.h>
#include <kfiledialog.h>
#include <kiconloader.h>
#include <kicon.h>
#include <klibloader.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kstandarddirs.h>
#include <kstatusbar.h>
#include <kstandardaction.h>
#include <kstandardshortcut.h>
#include <ktemporaryfile.h>
#include <kmenubar.h>
#include <kedittoolbar.h>
#include <kdebug.h>
#include <kxmlguifactory.h>
#include <kicontheme.h>
#include <kglobal.h>
#include <kmenu.h>
#include <kparts/componentfactory.h>
#include <ktoolbar.h>
#include <krecentfilesaction.h>
#include <ktoggleaction.h>
#include <kactioncollection.h>
#include <ktogglefullscreenaction.h>
#include <kwin.h>

#include <qcursor.h>

#include "kgv_miniwidget.h"
#include "kgv_view.h"
#include "kgvpageview.h"
#include "displayoptions.h"
#include "fullscreenfilter.h"

#undef Always  // avoid X11/Qt namespace clash
#include "kgvshell.moc"

//TODO -- disable GUI when no file
//TODO -- don't stay open when no file, go directly to KFileDialog

KGVShell::KGVShell() :
    _tmpFile( 0 )
{
    m_gvpart = KParts::ComponentFactory::createPartInstanceFromLibrary< KGVPart >( "libkghostviewpart", this,
                                                                                   this, QStringList("kgvpart") );

    /*---- File -----------------------------------------------------------*/
    openact =
	    KStandardAction::open( this, SLOT( slotFileOpen() ),
			      actionCollection() );
    recent =
	    KStandardAction::openRecent( this, SLOT( openUrl( const KUrl& ) ),
				    actionCollection() );
	    KStandardAction::print( m_gvpart->document(), SLOT( print() ),
			       actionCollection() );
    (void)
	    KStandardAction::quit( this, SLOT( slotQuit() ), actionCollection() );

    /*---- View -----------------------------------------------------------*/
            QAction *action = actionCollection()->addAction( "reload" );
            action->setIcon( KIcon("reload") );
            action->setText( i18n("&Reload") );
            connect(action, SIGNAL(triggered(bool) ), m_gvpart, SLOT(  reloadFile() ));
            action->setShortcuts(KStandardShortcut::shortcut( KStandardShortcut::Reload ));
	    action = actionCollection()->addAction( "maximize" );
            action->setText( i18n("&Maximize") );
	    connect(action, SIGNAL(triggered(bool) ), SLOT( slotMaximize() ));
	    action->setShortcut(Qt::Key_M);
    _showMenuBarAction = KStandardAction::showMenubar( this, SLOT( slotShowMenubar() ), actionCollection() );

    /*---- Settings -------------------------------------------------------*/
    createStandardStatusBarAction();
    setAutoSaveSettings();
    setStandardToolBarMenuEnabled(true);
    m_fullScreenAction = KStandardAction::fullScreen( this, SLOT( slotUpdateFullScreen() ), this, actionCollection() );
    KStandardAction::configureToolbars( this, SLOT( slotConfigureToolbars() ), actionCollection() );
    KStandardAction::keyBindings(guiFactory(), SLOT(configureShortcuts()),
actionCollection());

    //_popup = new KMenu( i18n( "Full Screen Options" ), this, "rmb popup" );
    _popup = new KMenu( this );
    _popup->setObjectName( "rmb popup" );
    _popup->addTitle( i18n( "Full Screen Options" ) );
    _popup->addAction( m_fullScreenAction );
    _popup->addAction( _showMenuBarAction );

    m_fsFilter = new FullScreenFilter( *this );

    // Just save them automatically is destructor. (TODO: of kgv_view!)
    //KStandardAction::saveOptions ( this, SLOT (slotWriteSettings()), actionCollection());

    setXMLFile( "kghostviewui.rc" );

    // We could, at the user's option, make this connection and kghostview
    // will always resize to fit the width of the page.  But, for now,
    // let's not.
    // connect ( m_gvpart->widget(), SIGNAL (sizeHintChanged()),	    this, SLOT (slotResize ()) );

    setCentralWidget( m_gvpart->widget() );
    createGUI( m_gvpart );

    connect( m_gvpart->pageView(), SIGNAL( rightClick() ),SLOT( slotRMBClick() ) );
    connect( m_gvpart, SIGNAL( canceled(const QString&) ),SLOT( slotReset() ) );
    connect( m_gvpart, SIGNAL( completed() ), SLOT( slotDocumentState() ) );

    if (!initialGeometrySet())
        resize(640,400);

    readSettings();
    stateChanged( "initState" );

    // Make sure the view has the keyboard focus.
    m_gvpart->widget()->setFocus();
}

KGVShell::~KGVShell()
{
    writeSettings();

    if( _tmpFile )
    {
	_tmpFile->setAutoRemove( true );
	delete _tmpFile;
	_tmpFile = 0;
    }

    // delete m_gvpart;
}

void
KGVShell::slotQuit()
{
    kapp->closeAllWindows();
}

void KGVShell::slotShowMenubar()
{
    if ( _showMenuBarAction->isChecked() ) menuBar()->show();
    else menuBar()->hide();
}

void
KGVShell::setDisplayOptions( const DisplayOptions& options )
{
    m_gvpart->setDisplayOptions( options );
}

void KGVShell::slotReset()
{
    kDebug( 4500 ) << "KGVShell::slotReset()" << endl;
    stateChanged( "initState" );
}

void
KGVShell::readProperties( KConfig *config )
{
    KUrl url = KUrl::fromPathOrUrl( config->readPathEntry( "URL" ) );
    if ( url.isValid() ) {
        openUrl( url );
	DisplayOptions options;
	if ( DisplayOptions::fromString( options, config->readEntry( "Display Options" ) ) ) m_gvpart->setDisplayOptions( options );
    }
}

void
KGVShell::saveProperties( KConfig* config )
{
    config->writePathEntry( "URL", m_gvpart->url().prettyUrl() );
    config->writeEntry( "Display Options", DisplayOptions::toString( m_gvpart->miniWidget()->displayOptions() ) );
}

void
KGVShell::readSettings()
{
    recent->loadEntries( KGlobal::config().data() );
    QStringList items = recent->items();

// Code copied from kviewshell.cpp:
// Constant source of annoyance in KDVI < 1.0: the 'recent-files'
// menu contains lots of files which don't exist (any longer). Thus,
// we'll sort out the non-existent files here.

    for ( QStringList::Iterator it = items.begin(); it != items.end(); ++it ) {
        KUrl url(*it);
        if (url.isLocalFile()) {
            QFileInfo info(url.path());
            if (!info.exists())
                recent->removeUrl(url);
        }
    }

    applyMainWindowSettings(KGlobal::config().data(), "MainWindow");

    KGlobal::config()->setDesktopGroup();
    bool fullScreen = KGlobal::config()->readEntry( "FullScreen", false );
    setFullScreen( fullScreen );
    _showMenuBarAction->setChecked( menuBar()->isVisible() );
}

void
KGVShell::writeSettings()
{
    saveMainWindowSettings(KGlobal::config().data(), "MainWindow");

    recent->saveEntries( KGlobal::config().data() );

    KGlobal::config()->setDesktopGroup();
    KGlobal::config()->writeEntry( "FullScreen", m_fullScreenAction->isChecked());

    KGlobal::config()->sync();
}

void
KGVShell::openUrl( const KUrl & url )
{
    if( m_gvpart->openUrl( url ) ) recent->addUrl (url);
}

void
KGVShell::openStdin()
{
    if( _tmpFile )
    {
	_tmpFile->setAutoRemove( true );
	delete _tmpFile;
    }

    _tmpFile = new KTemporaryFile;
    if( !_tmpFile->open() ) {
	KMessageBox::error( this,
		i18n( "Could not create temporary file." ) );
	return;
    }

    QByteArray buf( BUFSIZ );
    int read = 0, wrtn = 0;
    while( ( read = fread( buf.data(), sizeof(char), buf.size(), stdin ) )
	    > 0 ) {
	wrtn = _tmpFile->write( buf.data(), read );
	if( read != wrtn )
	    break;
	kapp->processEvents();
    }

    if( read != 0 ) {
	KMessageBox::error( this,
		i18n( "Could not open standard input stream: %1" ,
		  strerror( errno ) ) );
	return;
    }

    _tmpFile->flush();

    if( m_gvpart->openUrl( KUrl::fromPathOrUrl( _tmpFile->fileName() ) ) ) setCaption( "stdin" );
}

void KGVShell::slotFileOpen()
{
    KUrl url = KFileDialog::getOpenUrl( cwd, i18n(
		    "*.ps *.ps.bz2 *.ps.gz *.eps *.eps.gz *.pdf|All Document Files\n"
		    "*.ps *.ps.bz2 *.ps.gz|PostScript Files\n"
		    "*.pdf *.pdf.gz *.pdf.bz2|Portable Document Format (PDF) Files\n"
		    "*.eps *.eps.gz *.eps.bz2|Encapsulated PostScript Files\n"
		    "*|All Files" ) );

    if( !url.isEmpty() ) {
	openUrl( url );
    }
}

void KGVShell::slotDocumentState()
{
    stateChanged( "documentState" );
}


void KGVShell::slotMaximize()
{
    kDebug(4500) << "KGVShell::slotMaximize()" << endl;
#ifdef Q_WS_X11
    KWin::setState( winId(), NET::MaxHoriz | NET::MaxVert );
#endif
    // If we do it now, it comes to nothing since it would work
    // on the current (non-maximized) size
    QTimer::singleShot( 800, m_gvpart, SLOT( slotFitToPage() ) );
}

void KGVShell::slotResize()
{
    resize( m_gvpart->pageView()->sizeHint().width(), height() );
}

void KGVShell::setFullScreen( bool useFullScreen )
{
    if( useFullScreen )
        showFullScreen();
    else if( isFullScreen())
        showNormal();
}

void KGVShell::slotUpdateFullScreen()
{
    if( m_fullScreenAction->isChecked())
    {
	menuBar()->hide();
	statusBar()->hide();
	toolBar()->hide();
	m_gvpart->updateFullScreen( true );
	showFullScreen();
	kapp->installEventFilter( m_fsFilter );
	if ( m_gvpart->document()->isOpen() )
		m_gvpart->slotFitToPage();
    }
    else
    {
	kapp->removeEventFilter( m_fsFilter );
	m_gvpart->updateFullScreen( false );
	menuBar()->show();
	KToggleAction *statusbarAction = dynamic_cast<KToggleAction *>(actionCollection()->action(KStandardAction::name(KStandardAction::ShowStatusbar)));
	assert( statusbarAction );
	if (statusbarAction->isChecked()) statusBar()->show();
	toolBar()->show();
	showNormal();
    }
}

void KGVShell::slotConfigureToolbars()
{
    saveMainWindowSettings( KGlobal::config().data(), "MainWindow" );
    KEditToolbar dlg( factory() );
    connect(&dlg,SIGNAL(newToolbarConfig()),this,SLOT(slotNewToolbarConfig()));
    dlg.exec();
}

void KGVShell::slotNewToolbarConfig()
{
    applyMainWindowSettings( KGlobal::config().data(), "MainWindow" );
}

void KGVShell::slotRMBClick()
{
    _popup->exec( QCursor::pos() );
}


// vim:sw=4:sts=4:ts=8:noet
