
/* Yo Emacs, this -*- C++ -*-

  Copyright (C) 1999-2001 Jens Hoefkens
  jens@hoefkens.com

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

  $Id: kbgfibschat.cpp 721729 2007-10-05 23:25:56Z apaku $

*/


#include "kbgfibschat.h"
#include "kbgfibschat.moc"



#include <QLayout>
#include <QLabel>
#include <QTabWidget>
#include <QRegExp>
#include <QFont>
#include <QApplication>
#include <ktoggleaction.h>
#include <kactioncollection.h>
#include <QDateTime>
#include <QClipboard>
//#include <q3simplerichtext.h>
#include <QRegion>
#include <QPalette>
#include <QPainter>
#include <QPoint>
#include <q3listbox.h>
#include <QIcon>
#include <QStringList>
//#include <q3dict.h>
#include <QMultiHash>
//Added by qt3to4:
//#include <QGridLayout>
//#include <QHideEvent>
//#include <Q3Frame>
//#include <QShowEvent>

#include <klocale.h>
#include <kconfig.h>
#include <kdebug.h>
#include <kstandardaction.h>
#include <kaction.h>
#include <kiconloader.h>
#include <kglobal.h>
#include <kicon.h>
#include <kconfiggroup.h>

#include "clip.h"
#include "version.h"


/*
 * Private utility class that might become more generally useful in
 * the future. Basically, it implements rich text QListBox items.
 */
// class KLBT : public Q3ListBoxText
// {
// 
// public:
// 
//     /*
//      * Constructor
//      */
//     KLBT(QWidget *parent, const QString &text = QString(), const QString &player = QString())
//         : Q3ListBoxText(text)
//     {
//         w = parent;
//         n = new QString(player);
//         t = new Q3SimpleRichText(text, w->font());
// 
//         // FIXME: this is not yet perfect
//         t->setWidth(w->width()-20);
//     }
// 
//     /*
//      * Destructor
//      */
//     virtual ~KLBT()
//     {
//         delete t;
//         delete n;
//     }
// 
//     /*
//      * Overloaded required members returning height
//      */
//     virtual int height(const Q3ListBox *) const
//     {
//         return (1+t->height());
//     }
// 
//     /*
//      * Overloaded required members returning width
//      */
//     virtual int width(const Q3ListBox *) const
//     {
//         return t->width();
//     }
// 
//     /*
//      * The context menu needs the name of the player. It's easier
//      * than extracting it from the text.
//      */
//     QString player() const
//     {
//         return *n;
//     }
// 
// protected:
// 
//     /*
//      * Required overloaded member to paint the text on the painter p.
//      */
//     virtual void paint(QPainter *p)
//     {
//         t->draw(p, 1, 1, QRegion(p->viewport()), QColorGroup( w->palette() ));
//     }
// 
// private:
// 
//     Q3SimpleRichText *t;
//     QWidget *w;
//     QString *n;
// 
// };


class KBgChatPrivate
{
public:

    /*
     * Name of the users
     */
    QString mName[2];

    /*
     * Hold and assemble info text
     */
    QString mText;

    /*
     * Numbers of the private action list.
     */
    enum Privact {Inquire, InviteD, Invite1, Invite2, Invite3, Invite4,
                  Invite5, Invite6, Invite7, InviteR, InviteU, Silent,
                  Talk, Gag, Ungag, Cleargag, Copy, Clear, Close, MaxAction};

    /*
     * Available actions
     */
    QAction *mAct[MaxAction];

    /*
     * Context menu and invitation menu
     */
    QMenu *mChat, *mInvt;

    /*
     * list of users we do not want to hear shouting
     */
    QStringList mGag;

    /*
     * Listbox needed by the setup dialog
     */
    Q3ListBox *mLb;

    /*
     * Internal ID to name mapping
     */
    //Q3Dict<int> *mName2ID;
    QMultiHash<QString, int *> *mName2ID;
};


// == constructor, destructor ==================================================

/*
 * Constructor of the chat window.
 */
KBgChat::KBgChat(QWidget *parent)
    : KChat(parent, false)
{
    d = new KBgChatPrivate();
    KActionCollection* actions = new KActionCollection(this);

    d->mName[0].clear();
    d->mChat = 0;
    d->mInvt = new QMenu();

    setAutoAddMessages(false); // we get an echo from FIBS
    setFromNickname(i18n("%1 user", QLatin1String(PROG_NAME)));

    if (!addSendingEntry(i18n("Kibitz to watchers and players"), CLIP_YOU_KIBITZ))
        kDebug(10500) << "adding kibitz";
    if (!addSendingEntry(i18n("Whisper to watchers only"), CLIP_YOU_WHISPER))
        kDebug(10500) << "adding whisper";

    connect(this, SIGNAL(rightButtonClicked(Q3ListBoxItem *, const QPoint &)),
            this, SLOT(contextMenu(Q3ListBoxItem *, const QPoint &)));
    connect(this, SIGNAL(signalSendMessage(int, const QString &)),
            this, SLOT(handleCommand(int, const QString &)));

    //d->mName2ID = new Q3Dict<int>(17, true);
    d->mName2ID = new QMultiHash<QString, int *>;
    //d->mName2ID->setAutoDelete(true);

    /*
     * some eye candy :)
     */
    setWindowIcon(qApp->windowIcon().pixmap(IconSize(KIconLoader::Small),IconSize(KIconLoader::Small)));
    setWindowTitle(i18n("Chat Window"));

    this->setWhatsThis( i18n("This is the chat window.\n\n"
                               "The text in this window is colored depending on whether "
                               "it is directed at you personally, shouted to the general "
                               "FIBS population, has been said by you, or is of general "
                               "interest. If you select the name of a player, the context "
                               "contains entries specifically geared towards that player."));
    /*
     * Define set of available actions
     */
    d->mAct[KBgChatPrivate::Inquire] = actions->addAction("fibs_info_on");
    d->mAct[KBgChatPrivate::Inquire]->setIcon(KIcon("help-contents"));
    d->mAct[KBgChatPrivate::Inquire]->setText(i18n("Info On"));
    connect(d->mAct[KBgChatPrivate::Inquire], SIGNAL(triggered(bool) ), SLOT(slotInquire()));
    d->mAct[KBgChatPrivate::Talk]    = actions->addAction("fibs_talk");
    d->mAct[KBgChatPrivate::Talk]->setIcon(KIcon(PROG_NAME "-chat.png"));
    d->mAct[KBgChatPrivate::Talk]->setText(i18n("Talk To"));
    connect(d->mAct[KBgChatPrivate::Talk], SIGNAL(triggered(bool) ), SLOT(slotTalk()));

    d->mAct[KBgChatPrivate::InviteD] = actions->addAction("fibs_use_dialog");
    d->mAct[KBgChatPrivate::InviteD]->setText(i18n("Use Dialog"));
    connect(d->mAct[KBgChatPrivate::InviteD], SIGNAL(triggered(bool) ), SLOT(slotInviteD()));
    d->mAct[KBgChatPrivate::Invite1] = actions->addAction("fibs_pm_1");
    d->mAct[KBgChatPrivate::Invite1]->setText(i18n("1 Point Match"));
    connect(d->mAct[KBgChatPrivate::Invite1], SIGNAL(triggered(bool) ), SLOT(slotInvite1()));
    d->mAct[KBgChatPrivate::Invite2] = actions->addAction("fibs_pm_2");
    d->mAct[KBgChatPrivate::Invite2]->setText(i18n("2 Point Match"));
    connect(d->mAct[KBgChatPrivate::Invite2], SIGNAL(triggered(bool) ), SLOT(slotInvite2()));
    d->mAct[KBgChatPrivate::Invite3] = actions->addAction("fibs_pm_3");
    d->mAct[KBgChatPrivate::Invite3]->setText(i18n("3 Point Match"));
    connect(d->mAct[KBgChatPrivate::Invite3], SIGNAL(triggered(bool) ), SLOT(slotInvite3()));
    d->mAct[KBgChatPrivate::Invite4] = actions->addAction("fibs_pm_4");
    d->mAct[KBgChatPrivate::Invite4]->setText(i18n("4 Point Match"));
    connect(d->mAct[KBgChatPrivate::Invite4], SIGNAL(triggered(bool) ), SLOT(slotInvite4()));
    d->mAct[KBgChatPrivate::Invite5] = actions->addAction("fibs_pm_5");
    d->mAct[KBgChatPrivate::Invite5]->setText(i18n("5 Point Match"));
    connect(d->mAct[KBgChatPrivate::Invite5], SIGNAL(triggered(bool) ), SLOT(slotInvite5()));
    d->mAct[KBgChatPrivate::Invite6] = actions->addAction("fibs_pm_6");
    d->mAct[KBgChatPrivate::Invite6]->setText(i18n("6 Point Match"));
    connect(d->mAct[KBgChatPrivate::Invite6], SIGNAL(triggered(bool) ), SLOT(slotInvite6()));
    d->mAct[KBgChatPrivate::Invite7] = actions->addAction("fibs_pm_7");
    d->mAct[KBgChatPrivate::Invite7]->setText(i18n("7 Point Match"));
    connect(d->mAct[KBgChatPrivate::Invite7], SIGNAL(triggered(bool) ), SLOT(slotInvite7()));
    d->mAct[KBgChatPrivate::InviteU] = actions->addAction("fibs_unlimited");
    d->mAct[KBgChatPrivate::InviteU]->setText(i18n("Unlimited"));
    connect(d->mAct[KBgChatPrivate::InviteU], SIGNAL(triggered(bool) ), SLOT(slotInviteU()));
    d->mAct[KBgChatPrivate::InviteR] = actions->addAction("fibs_resume");
    d->mAct[KBgChatPrivate::InviteR]->setText(i18n("Resume"));
    connect(d->mAct[KBgChatPrivate::InviteR], SIGNAL(triggered(bool) ), SLOT(slotInviteR()));

    d->mInvt->addAction( d->mAct[KBgChatPrivate::InviteD] );

    d->mInvt->addSeparator();

    d->mInvt->addAction( d->mAct[KBgChatPrivate::Invite1] );
    d->mInvt->addAction( d->mAct[KBgChatPrivate::Invite2] );
    d->mInvt->addAction( d->mAct[KBgChatPrivate::Invite3] );
    d->mInvt->addAction( d->mAct[KBgChatPrivate::Invite4] );
    d->mInvt->addAction( d->mAct[KBgChatPrivate::Invite5] );
    d->mInvt->addAction( d->mAct[KBgChatPrivate::Invite6] );
    d->mInvt->addAction( d->mAct[KBgChatPrivate::Invite7] );

    d->mInvt->addSeparator();

    d->mInvt->addAction( d->mAct[KBgChatPrivate::InviteU] );
    d->mInvt->addAction( d->mAct[KBgChatPrivate::InviteR] );

    d->mAct[KBgChatPrivate::Gag] = actions->addAction("fibs_gag");
    d->mAct[KBgChatPrivate::Gag]->setText(i18n("Gag"));
    connect(d->mAct[KBgChatPrivate::Gag], SIGNAL(triggered(bool) ), SLOT(slotGag()));
    d->mAct[KBgChatPrivate::Ungag] = actions->addAction("fibs_ungag");
    d->mAct[KBgChatPrivate::Ungag]->setText(i18n("Ungag"));
    connect(d->mAct[KBgChatPrivate::Ungag], SIGNAL(triggered(bool) ), SLOT(slotUngag()));
    d->mAct[KBgChatPrivate::Cleargag] = actions->addAction("fibs_clear_gags");
    d->mAct[KBgChatPrivate::Cleargag]->setText(i18n("Clear Gag List"));
    connect(d->mAct[KBgChatPrivate::Cleargag], SIGNAL(triggered(bool) ), SLOT(slotCleargag()));
    d->mAct[KBgChatPrivate::Copy]     = KStandardAction::copy(this, SLOT(slotCopy()), this);
    actions->addAction(d->mAct[KBgChatPrivate::Copy]->objectName(), d->mAct[KBgChatPrivate::Copy]);
    d->mAct[KBgChatPrivate::Clear] = actions->addAction("fibs_clear");
    d->mAct[KBgChatPrivate::Clear]->setText(i18n("Clear"));
    connect(d->mAct[KBgChatPrivate::Clear], SIGNAL(triggered(bool) ), SLOT(slotClear()));
    d->mAct[KBgChatPrivate::Close]    = KStandardAction::close(this, SLOT(hide()), this);
    actions->addAction(d->mAct[KBgChatPrivate::Close]->objectName(), d->mAct[KBgChatPrivate::Close]);
    d->mAct[KBgChatPrivate::Silent] = new KToggleAction(i18n("Silent"), this);
    actions->addAction(QString(), d->mAct[KBgChatPrivate::Silent]);
    connect(d->mAct[KBgChatPrivate::Silent], SIGNAL(triggered(bool) ), SLOT(slotSilent()));
}


/*
 * Destructor
 */
KBgChat::~KBgChat()
{
    delete d->mName2ID;
    delete d->mChat; // save to delete NULL pointers
    delete d->mInvt;
    delete d;
}


// == configuration handling ===================================================

/*
 * Restore the previously stored settings
 */
void KBgChat::readConfig()
{
    KConfigGroup config = KGlobal::config()->group("chat window");

    QPoint pos = config.readEntry("ori", QPoint( 10, 10 ));
    setGeometry(pos.x(), pos.y(), config.readEntry("wdt",460), config.readEntry("hgt",200));

    config.readEntry("vis", false) ? show() : hide();

    ((KToggleAction *)d->mAct[KBgChatPrivate::Silent])->setChecked(config.readEntry("sil", false));

    d->mGag = config.readEntry("gag",QStringList());
}

/*
 * Save the current settings to disk
 */
void KBgChat::saveConfig()
{
    KConfigGroup config = KGlobal::config()->group("chat window");

    config.writeEntry("ori", pos());
    config.writeEntry("hgt", height());
    config.writeEntry("wdt", width());
    config.writeEntry("vis", isVisible());

    config.writeEntry("sil", ((KToggleAction *)d->mAct[KBgChatPrivate::Silent])->isChecked());

    config.writeEntry("gag", d->mGag);
}


/*
 * Setup dialog page of the player list - allow the user to select the
 * columns to show
 *
 * FIXME: need to be able to set font here KChatBase::setBothFont(const QFont& font)
 */
void KBgChat::getSetupPages(QTabWidget *nb, int space)
{
    /*
     * Main Widget
     * ===========
     */
    QWidget *w = new QWidget(nb);
    QGridLayout *gl = new QGridLayout(w);
    gl->setMargin(space);

    d->mLb = new Q3ListBox(w);
    d->mLb->setMultiSelection(true);

    d->mLb->insertStringList(d->mGag);

    QLabel *info = new QLabel(w);
    info->setText(i18n("Select users to be removed from the gag list."));

    w->setWhatsThis( i18n("Select all the users you want "
                            "to remove from the gag list "
                            "and then click OK. Afterwards "
                            "you will again hear what they shout."));

    gl->addWidget(d->mLb, 0, 0);
    gl->addWidget(info, 1, 0);

    /*
     * put in the page
     * ===============
     */
    gl->activate();
    w->adjustSize();
    w->setMinimumSize(w->size());
    nb->addTab(w, i18n("&Gag List"));
}

/*
 * Remove all the selected entries from the gag list
 */
void KBgChat::setupOk()
{
    for (uint i = 0; i < d->mLb->count(); ++i) {
        if (d->mLb->isSelected(i))
            d->mGag.removeAll(d->mLb->text(i));
    }
    d->mLb->clear();
    d->mLb->insertStringList(d->mGag);
}

/*
 * Don't do anything
 */
void KBgChat::setupCancel()
{
    // empty
}

/*
 * By default, all players stay in the gag list
 */
void KBgChat::setupDefault()
{
    d->mLb->clearSelection();
}


// == various slots and functions ==============================================

/*
 * Overloaded member to create a QListBoxItem for the chat window.
 */
// Q3ListBoxItem* KBgChat::layoutMessage(const QString& fromName, const QString& text)
// {
//     Q3ListBoxText* message = new KLBT(this, text, fromName);
//     return message;
// }

/*
 * Catch hide events, so the engine's menu can be update.
 */
void KBgChat::showEvent(QShowEvent *e)
{
    Q_UNUSED(e)
    //Q3Frame::showEvent(e);
    emit windowVisible(true);
}

/*
 * Catch hide events, so the engine's menu can be update.
 */
void KBgChat::hideEvent(QHideEvent *e)
{
    Q_UNUSED(e)
    emit windowVisible(false);
    //Q3Frame::hideEvent(e);
}

/*
 * At the beginning of a game, add the name to the list and switch to
 * kibitz mode.
 */
void KBgChat::startGame(const QString &name)
{
    int *id = *d->mName2ID->find(d->mName[1] = name);
    if (!id) {
        id = new int(nextId());
        d->mName2ID->insert(name, id);
        addSendingEntry(i18n("Talk to %1", name), *id);
    }
    setSendingEntry(CLIP_YOU_KIBITZ);
}

/*
 * At the end of a game, we switch to talk mode.
 */
void KBgChat::endGame()
{
    int *id = *d->mName2ID->find(d->mName[1]);
    if (id)
        setSendingEntry(*id);
    else
        setSendingEntry(SendToAll);
}

/*
 * Set the chat window ready to talk to name
 */
void KBgChat::fibsTalk(const QString &name)
{
    int *id = *d->mName2ID->find(name);
    if (!id) {
        id = new int(nextId());
        d->mName2ID->insert(name, id);
        addSendingEntry(i18n("Talk to %1", name), *id);
    }
    setSendingEntry(*id);
}

/*
 * Remove the player from the combo box when he/she logs out.
 */
void KBgChat::deletePlayer(const QString &name)
{
    int *id = *d->mName2ID->find(name);
    if (id) {
        //removeSendingEntry(*id);//FIXME this line causes a crash
	//d->mName2ID->remove(name); //dict.remove(key, value);delete hash.take(key);
	delete d->mName2ID->take(name);
    }
}

/*
 * Take action when the user presses return in the line edit control.
 */
void KBgChat::handleCommand(int id, const QString& msg)
{
    Q_UNUSED(id)
    int realID = sendingEntry();

    switch (realID) {
    case SendToAll:
        emit fibsCommand("shout " + msg);
        break;
    case CLIP_YOU_KIBITZ:
        emit fibsCommand("kibitz " + msg);
        break;
    case CLIP_YOU_WHISPER:
        emit fibsCommand("whisper " + msg);
        break;
    default:
        //Q3DictIterator<int> it(*d->mName2ID);
        QHashIterator<QString, int *> it(*d->mName2ID);
	while (it.value()) {
	    it.next();
            if (*it.value() == realID) {
                emit fibsCommand("tell " + it.key() + ' ' + msg);
                return;
            }
            //++it;
        }
        kDebug(10500) << "unrecognized ID in KBgChat::handleCommand";
    }
}


// == handle strings from the server ===========================================

/*
 * A message from the server that should be handled by us. If necessary,
 * we replace the CLIP number by a string and put the line into the window.
 *
 * This function emits the string in rich text format with the signal
 * personalMessage - again: the string contains rich text!
 */
void KBgChat::handleData(const QString &msg)
{
    QString clip = msg.left(msg.indexOf(' ')), user, cMsg = msg;
    QDateTime date;

    bool flag = false;
    int cmd = clip.toInt(&flag);

    if (flag) {
        cMsg.replace(0, cMsg.indexOf(' ')+1, "");

        user = cMsg.left(cMsg.indexOf(' '));

        switch (cmd) {
        case CLIP_SAYS:
            if (!d->mGag.contains(user)) {
                cMsg = i18n("<u>%1 tells you:</u> %2", user, cMsg.replace(QRegExp('^' + user), ""));
                cMsg = "<font color=\"red\">" + cMsg + "</font>";
                emit personalMessage(cMsg);
            } else
                cMsg = "";
            break;

        case CLIP_SHOUTS:
            if ((!((KToggleAction *)d->mAct[KBgChatPrivate::Silent])->isChecked()) && (!d->mGag.contains(user))) {
                cMsg = i18n("<u>%1 shouts:</u> %2", user, cMsg.replace(QRegExp('^' + user), ""));
                cMsg = "<font color=\"black\">" + cMsg + "</font>";
            } else
                cMsg = "";
            break;

        case CLIP_WHISPERS:
            if (!d->mGag.contains(user)) {
                cMsg = i18n("<u>%1 whispers:</u> %2", user, cMsg.replace(QRegExp('^' + user), ""));
                cMsg = "<font color=\"red\">" + cMsg + "</font>";
                emit personalMessage(cMsg);
            } else
                cMsg = "";
            break;

        case CLIP_KIBITZES:
            if (!d->mGag.contains(user)) {
                cMsg = i18n("<u>%1 kibitzes:</u> %2", user, cMsg.replace(QRegExp('^' + user), ""));
                cMsg = "<font color=\"red\">" + cMsg + "</font>";
                emit personalMessage(cMsg);
            } else
                cMsg = "";
            break;

        case CLIP_YOU_SAY:
            cMsg = i18n("<u>You tell %1:</u> %2", user, cMsg.replace(QRegExp('^' + user), ""));
            cMsg = "<font color=\"darkgreen\">" + cMsg + "</font>";
            emit personalMessage(cMsg);
            user.clear();
            break;

        case CLIP_YOU_SHOUT:
            cMsg = i18n("<u>You shout:</u> %1", cMsg);
            cMsg = "<font color=\"darkgreen\">" + cMsg + "</font>";
            emit personalMessage(cMsg);
            user.clear();
            break;

        case CLIP_YOU_WHISPER:
            cMsg = i18n("<u>You whisper:</u> %1", cMsg);
            cMsg = "<font color=\"darkgreen\">" + cMsg + "</font>";
            emit personalMessage(cMsg);
            user.clear();
            break;

        case CLIP_YOU_KIBITZ:
            cMsg = i18n("<u>You kibitz:</u> %1", cMsg);
            cMsg = "<font color=\"darkgreen\">" + cMsg + "</font>";
            emit personalMessage(cMsg);
            user.clear();
            break;

        case CLIP_MESSAGE:
            user = cMsg.left(cMsg.indexOf(' ')+1);
            cMsg.remove(0, cMsg.indexOf(' ')+1);
            date.setTime_t(cMsg.left(cMsg.indexOf(' ')+1).toUInt());
            cMsg.remove(0, cMsg.indexOf(' '));
            cMsg = i18n("<u>User %1 left a message at %2</u>: %3", user, date.toString(), cMsg);
            cMsg = "<font color=\"red\">" + cMsg + "</font>";
            emit personalMessage(cMsg);
            user.clear();
            break;

        case CLIP_MESSAGE_DELIVERED:
            cMsg = i18n("Your message for %1 has been delivered.", user);
            cMsg = QString("<font color=\"darkgreen\">") + cMsg + "</font>";
            emit personalMessage(cMsg);
            user.clear();
            break;

        case CLIP_MESSAGE_SAVED:
            cMsg = i18n("Your message for %1 has been saved.", user);
            cMsg = QString("<font color=\"darkgreen\">") + cMsg + "</font>";
            emit personalMessage(cMsg);
            user.clear();
            break;

        default: // ignore the message
            return;
        }

    } else {

        /*
         * Special treatment for non-CLIP messages
         */
        if (cMsg.contains(QRegExp("^You say to yourself: "))) {
            cMsg.replace(QRegExp("^You say to yourself: "),
                         i18n("<u>You say to yourself:</u> "));
        } else {
            kDebug(user.isNull(), 10500) << "KBgChat::handleData unhandled message: "
                                                  << cMsg.toLatin1();
            return;
        }
    }

    if (!cMsg.isEmpty())
        addMessage(user, cMsg);
}


// == context menu and related slots ===========================================

/*
 * RMB opens a context menu.
 */
// void KBgChat::contextMenu(Q3ListBoxItem *i, const QPoint &p)
// {
//     /*
//      * Even if i is non-null, user might still be QString()
//      */
//     d->mName[0] = (i == 0) ? QString() : ((KLBT *)i)->player();
//     d->mText = (i == 0) ? QString() : ((KLBT *)i)->text();
// 
//     /*
//      * Get a new context menu every time. Safe to delete the 0
//      * pointer.
//      */
//     delete d->mChat; d->mChat = new QMenu();
// 
//     /*
//      * Fill the context menu with actions
//      */
//     if (!d->mName[0].isNull()) {
// 
//         d->mAct[KBgChatPrivate::Talk]->setText(i18n("Talk to %1", d->mName[0]));
//         d->mChat->addAction( d->mAct[KBgChatPrivate::Talk] );
// 
//         d->mAct[KBgChatPrivate::Inquire]->setText(i18n("Info on %1", d->mName[0]));
//         d->mChat->addAction( d->mAct[KBgChatPrivate::Inquire] );
// 
//         // invite menu is always the same
//         d->mInvt->setTitle(i18n("Invite %1", d->mName[0]));
//         d->mChat->addMenu(d->mInvt);
// 
//         d->mChat->addSeparator();
// 
//         if (d->mGag.contains(d->mName[0]) <= 0) {
//             d->mAct[KBgChatPrivate::Gag]->setText(i18n("Gag %1", d->mName[0]));
//             d->mChat->addAction( d->mAct[KBgChatPrivate::Gag] );
//         } else {
//             d->mAct[KBgChatPrivate::Ungag]->setText(i18n("Ungag %1", d->mName[0]));
//             d->mChat->addAction( d->mAct[KBgChatPrivate::Ungag] );
//         }
//     }
//     if (d->mGag.count() > 0)
//         d->mChat->addAction( d->mAct[KBgChatPrivate::Cleargag] );
// 
//     if ((d->mGag.count() > 0) || (!d->mName[0].isNull()))
//         d->mChat->addSeparator();
// 
//     d->mChat->addAction( d->mAct[KBgChatPrivate::Silent] );
// 
//     d->mChat->addSeparator();
// 
//     d->mChat->addAction( d->mAct[KBgChatPrivate::Copy ] );
//     d->mChat->addAction( d->mAct[KBgChatPrivate::Clear] );
//     d->mChat->addAction( d->mAct[KBgChatPrivate::Close] );
// 
//     d->mChat->popup(p);
// }

/*
 * Clear the gag list
 */
void KBgChat::slotCleargag()
{
    d->mGag.clear();

    QString msg("<font color=\"blue\">");
    msg += i18n("The gag list is now empty.");
    msg += "</font>";

    addMessage(QString(), msg);
}

/*
 * Gag the selected user
 */
void KBgChat::slotGag()
{
    d->mGag.append(d->mName[0]);

    QString msg("<font color=\"blue\">");
    msg += i18n("You will not hear what %1 says and shouts.", d->mName[0]);
    msg += "</font>";

    addMessage(QString(), msg);
}

/*
 * Simple interface to the actual talk slot
 */
void KBgChat::slotTalk()
{
    fibsTalk(d->mName[0]);
}

/*
 * Remove selected user from gag list
 */
void KBgChat::slotUngag()
{
    d->mGag.removeAll(d->mName[0]);

    QString msg("<font color=\"blue\">");
    msg += i18n("You will again hear what %1 says and shouts.", d->mName[0]);
    msg += "</font>";

    addMessage(QString(), msg);
}

/*
 * Get information on selected user
 */
void KBgChat::slotInquire()
{
    kDebug(d->mName[0].isNull(), 10500) << "KBgChat::slotInquire: user == null";
    emit fibsCommand("whois " + d->mName[0]);
}

/*
 * Block all shouts from the chat window
 */
void KBgChat::slotSilent()
{
    QString msg;
    if (((KToggleAction *)d->mAct[KBgChatPrivate::Silent])->isChecked())
        msg = "<font color=\"blue\">" + i18n("You will not hear what people shout.") + "</font>";
    else
        msg = "<font color=\"blue\">" + i18n("You will hear what people shout.") + "</font>";
    addMessage(QString(), msg);
}

/*
 * Copy the selected line to the clipboard. Strip the additional HTML
 * from the text before copying.
 */
void KBgChat::slotCopy()
{
    d->mText.replace(QRegExp("<u>"), "");
    d->mText.replace(QRegExp("</u>"), "");
    d->mText.replace(QRegExp("</font>"), "");
    d->mText.replace(QRegExp("^.*\">"), "");

    QApplication::clipboard()->setText(d->mText);
}

/*
 * Invite the selected player.
 */
void KBgChat::slotInviteD()
{
    kDebug(d->mName[0].isNull(), 10500) << "KBgChat::slotInvite: user == null";
    emit fibsRequestInvitation(d->mName[0]);
}
void KBgChat::slotInvite1() { emit fibsCommand("invite " + d->mName[0] + " 1"); }
void KBgChat::slotInvite2() { emit fibsCommand("invite " + d->mName[0] + " 2"); }
void KBgChat::slotInvite3() { emit fibsCommand("invite " + d->mName[0] + " 3"); }
void KBgChat::slotInvite4() { emit fibsCommand("invite " + d->mName[0] + " 4"); }
void KBgChat::slotInvite5() { emit fibsCommand("invite " + d->mName[0] + " 5"); }
void KBgChat::slotInvite6() { emit fibsCommand("invite " + d->mName[0] + " 6"); }
void KBgChat::slotInvite7() { emit fibsCommand("invite " + d->mName[0] + " 7"); }

void KBgChat::slotInviteU() { emit fibsCommand("invite " + d->mName[0] + " unlimited"); }
void KBgChat::slotInviteR() { emit fibsCommand("invite " + d->mName[0]); }


// EOF
