/*
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */


// QT includes
#include <qnamespace.h>
#include <qcursor.h>
//Added by qt3to4:
#include <QMouseEvent>
#include <QLabel>
#include <QEvent>
#include <QMenu>

// KDE includes
#include <kmenubar.h>
#include <kstatusbar.h>
#include <kmessagebox.h>
#include <kconfig.h>
#include <klocale.h>
#include <kstandardshortcut.h>
#include <kstandardaction.h>
#include <kstandardgameaction.h>
#include <kaction.h>
#include <kactioncollection.h>
#include <kshortcut.h>
#include <kxmlguifactory.h>
#include <kglobal.h>
#include <ktoggleaction.h>

// own includes
#include "top.h"
#include "kpoker.h"
#include "defines.h"
#include "version.h"


PokerWindow::PokerWindow()
{
  m_kpok = new kpok(this);
  setCentralWidget( m_kpok );
  m_kpok->show();

  clickToHoldIsShown = false;

  LHLabel = new QLabel(statusBar());
  LHLabel->adjustSize();

  connect(m_kpok, SIGNAL(changeLastHand(const QString &, bool)),
	  this,   SLOT(setHand(const QString &, bool)));
  connect(m_kpok, SIGNAL(showClickToHold(bool)),
	  this,   SLOT(showClickToHold(bool)));
  connect(m_kpok, SIGNAL(clearStatusBar()),
	  this,   SLOT(clearStatusBar()));
  connect(m_kpok, SIGNAL(statusBarMessage(QString)),
	  this,   SLOT(statusBarMessage(QString)));

  statusBar()->addPermanentWidget(LHLabel);
  m_kpok->updateLHLabel();
  //FIXME: LHLabel is shown twize until the bar is repainted!

  initKAction();
  readOptions();
}


PokerWindow::~PokerWindow()
{
}


// ----------------------------------------------------------------


void PokerWindow::initKAction()
{
  QAction *action;

  //Game
  action = KStandardGameAction::gameNew(m_kpok, SLOT(newGame()), this);
  actionCollection()->addAction(action->objectName(), action);
  action = KStandardGameAction::save(m_kpok, SLOT(saveGame()), this);
  actionCollection()->addAction(action->objectName(), action);
  action = KStandardGameAction::quit(this, SLOT(close()), this);
  actionCollection()->addAction(action->objectName(), action);

  //Settings
  showMenubarAction =
    KStandardAction::showMenubar(this, SLOT(toggleMenubar()), this);
  actionCollection()->addAction(showMenubarAction->objectName(), showMenubarAction);

  soundAction = new KToggleAction(i18n("Soun&d"), this);
  actionCollection()->addAction("options_sound", soundAction);
  connect(soundAction, SIGNAL(triggered(bool) ), m_kpok, SLOT(toggleSound()));
  if (m_kpok->getSound())
    m_kpok->toggleSound();
  blinkingAction = new KToggleAction(i18n("&Blinking Cards"), this);
  actionCollection()->addAction("options_blinking", blinkingAction);
  connect(blinkingAction, SIGNAL(triggered(bool) ), m_kpok, SLOT(toggleBlinking()));
  if (m_kpok->getBlinking())
    m_kpok->toggleBlinking();
  adjustAction = new KToggleAction(i18n("&Adjust Bet is Default"), this);
  actionCollection()->addAction("options_adjust", adjustAction);
  connect(adjustAction, SIGNAL(triggered(bool) ), m_kpok, SLOT(toggleAdjust()));
  if (m_kpok->getAdjust())
    m_kpok->toggleAdjust();

  showStatusbarAction =
    KStandardAction::showStatusbar(this, SLOT(toggleStatusbar()), this);
  actionCollection()->addAction(showStatusbarAction->objectName(), showStatusbarAction);

  action = KStandardAction::saveOptions(this,     SLOT(saveOptions()), this);
  actionCollection()->addAction(action->objectName(), action);
  action = KStandardGameAction::carddecks(m_kpok, SLOT(slotCardDeck()), this);
  actionCollection()->addAction(action->objectName(), action);
  action = KStandardAction::preferences(m_kpok,   SLOT(slotPreferences()), this);
  actionCollection()->addAction(action->objectName(), action);

  // Keyboard shortcuts.
  action = actionCollection()->addAction("draw");
  action->setText(i18n("Draw"));
  connect(action, SIGNAL(triggered(bool) ), m_kpok, SLOT(drawClick()));
  action->setShortcuts(KShortcut(Qt::Key_Return));
  action = actionCollection()->addAction("exchange_card_1");
  action->setText(i18n("Exchange Card 1"));
  connect(action, SIGNAL(triggered(bool) ), m_kpok, SLOT(exchangeCard1()));
  action->setShortcuts(KShortcut(Qt::Key_1));
  action = actionCollection()->addAction("exchange_card_2");
  action->setText(i18n("Exchange Card 2"));
  connect(action, SIGNAL(triggered(bool) ), m_kpok, SLOT(exchangeCard2()));
  action->setShortcuts(KShortcut(Qt::Key_2));
  action = actionCollection()->addAction("exchange_card_3");
  action->setText(i18n("Exchange Card 3"));
  connect(action, SIGNAL(triggered(bool) ), m_kpok, SLOT(exchangeCard3()));
  action->setShortcuts(KShortcut(Qt::Key_3));
  action = actionCollection()->addAction("exchange_card_4");
  action->setText(i18n("Exchange Card 4"));
  connect(action, SIGNAL(triggered(bool) ), m_kpok, SLOT(exchangeCard4()));
  action->setShortcuts(KShortcut(Qt::Key_4));
  action = actionCollection()->addAction("exchange_card_5");
  action->setText(i18n("Exchange Card 5"));
  connect(action, SIGNAL(triggered(bool) ), m_kpok, SLOT(exchangeCard5()));
  action->setShortcuts(KShortcut(Qt::Key_5));

  setupGUI( KMainWindow::Save | StatusBar | Keys | Create);
}


void PokerWindow::readOptions()
{
  KSharedConfig::Ptr conf = KGlobal::config();
  conf->setGroup("General");

  if (m_kpok->getSound() != conf->readEntry("Sound", true))
    soundAction->trigger();

  if (m_kpok->getBlinking() != conf->readEntry("Blinking", true))
    blinkingAction->trigger();

  if (m_kpok->getAdjust() != conf->readEntry("Adjust", true))
    adjustAction->trigger();

  if ( showMenubarAction->isChecked() !=
       conf->readEntry("ShowMenubar", true))
    showMenubarAction->trigger();

  if ( showStatusbarAction->isChecked() !=
       conf->readEntry("ShowStatusbar", true))
    showStatusbarAction->trigger();
}


void PokerWindow::toggleMenubar()
{
  if (!menuBar()->isHidden())
    menuBar()->hide();
  else
    menuBar()->show();
}


void PokerWindow::toggleStatusbar()
{
  if (!statusBar()->isHidden())
    statusBar()->hide();
  else
    statusBar()->show();
}


/* Ask the user if he/she wants to save the game.  This virtual method
 * is called from the Quit KAction (I think).
 */

bool PokerWindow::queryClose()
{
  if (!m_kpok->isDirty())
    return true;

  // Only ask if the game is changed in some way.
  switch(KMessageBox::warningYesNoCancel(this, i18n("Do you want to save this game?"), QString::null, KStandardGuiItem::save(), KStandardGuiItem::dontSave())) {
  case KMessageBox::Yes :
    m_kpok->saveGame();
    return true;
  case KMessageBox::No :
    return true;
  default :
    return false;
  }
}


/* Show the hand or winner in the status bar at the lower right.
 *
 * Which is shown depends on wether this is a one player game or a two
 * player game.
 */

void PokerWindow::setHand(const QString &newHand, bool lastHand)
{
  if (lastHand)
    LHLabel->setText(i18n("Last hand: ") + newHand);
  else
    LHLabel->setText(i18n("Last winner: ") + newHand);
  LHLabel->adjustSize();
}


void PokerWindow::showClickToHold(bool show)
{
  if (show) {
    statusBar()->clearMessage();
    statusBar()->showMessage(i18n("Click a card to hold it"));
    clickToHoldIsShown = true;
  } else if (clickToHoldIsShown) {
    statusBar()->clearMessage();
    clickToHoldIsShown = false;
  }
}


void PokerWindow::statusBarMessage(QString s)
{
  clearStatusBar();
  statusBar()->showMessage(s);
  clickToHoldIsShown = false;
}


void PokerWindow::clearStatusBar()
{
  if (!clickToHoldIsShown)
    statusBar()->clearMessage();
}


void PokerWindow::saveOptions()
{
  KSharedConfig::Ptr conf = KGlobal::config();
  conf->setGroup("General");

  conf->writeEntry("Sound",         soundAction->isChecked());
  conf->writeEntry("Blinking",      blinkingAction->isChecked());
  conf->writeEntry("Adjust",        adjustAction->isChecked());
  conf->writeEntry("ShowMenubar",   showMenubarAction->isChecked());
  conf->writeEntry("ShowStatusbar", showStatusbarAction->isChecked());
}


bool PokerWindow::eventFilter(QObject*, QEvent* e)
{
  if (e->type() == QEvent::MouseButtonPress) {

    if (((QMouseEvent*)e)->button() == Qt::RightButton) {
      QMenu*  popup = (QMenu*) factory()->container("popup", this);
      if (popup)
	popup->popup(QCursor::pos());
      return true;
    } else
      return false;
  }

  return false;
}

#include "top.moc"

