/*
 * KAsteroids - Copyright Martin R. Jones 1997 <mjones@kde.org>
 *
 * Part of the KDE project
 */

#include <stdlib.h>
#include <math.h>
#include <kstandarddirs.h>
#include <kglobalsettings.h>

#include "settings.h"
#include "sprites.h"
#include "view.h"
#include "view.moc"
#include <QPixmap>
#include <QTimerEvent>
#include <QResizeEvent>
#include <QFileInfo>
#include <QDir>
#include <QTimer>
#include <QGraphicsView>
#include <QGraphicsScene>

#define IMG_BACKGROUND "bg.png"
#define SPRITES_PREFIX kapp->kde_datadir() + "/kasteroids/"

#define REFRESH_DELAY           33
#define SHIP_SPEED              0.3
#define MISSILE_SPEED           10.0
#define SHIP_STEPS              64
#define ROTATE_RATE             2
#define SHIELD_ON_COST          1
#define SHIELD_HIT_COST         30
#define BRAKE_ON_COST           4

#define MAX_ROCK_SPEED          2.5
#define MAX_POWERUP_SPEED       1.5
#define MAX_SHIP_SPEED		12
#define MAX_BRAKES              5
#define MAX_SHIELDS             5
#define MAX_FIREPOWER		5

#define TEXT_SPEED              4

#define PI_X_2                  6.283185307

static struct
{
    int id;
    const char *path;
}
kas_animations [] =
{
    { ID_ROCK_LARGE,       "rock1/rock1*.png" },
    { ID_ROCK_MEDIUM,      "rock2/rock2*.png" },
    { ID_ROCK_SMALL,       "rock3/rock3*.png" },
    { ID_SHIP,             "ship/ship*.png" },
    { ID_MISSILE,          "missile/missile.png" },
    { ID_BIT,              "bits/bits*.png" },
    { ID_EXHAUST,          "exhaust/exhaust.png" },
    { ID_ENERGY_POWERUP,   "powerups/energy.png" },
//    { ID_TELEPORT_POWERUP, "powerups/teleport*.png" },
    { ID_BRAKE_POWERUP,    "powerups/brake.png" },
    { ID_SHIELD_POWERUP,   "powerups/shield.png" },
    { ID_SHOOT_POWERUP,    "powerups/shoot.png" },
    { ID_SHIELD,           "shield/shield*.png" },
    { 0,                   0 }
};



KAsteroidsView::KAsteroidsView( QWidget *parent )
    : QWidget( parent ), shield(0), ship(0)
{
    field = new QGraphicsScene( 0, 0, 640, 440 );
    field->setItemIndexMethod( QGraphicsScene::NoIndex );
    view =  new QGraphicsView( field, this );
    view->setVerticalScrollBarPolicy( Qt::ScrollBarAlwaysOff );
    view->setHorizontalScrollBarPolicy( Qt::ScrollBarAlwaysOff );
    view->setCacheMode( QGraphicsView::CacheBackground );

    QPixmap pm( KStandardDirs::locate("sprite", IMG_BACKGROUND) );
    field->setBackgroundBrush( pm );

    textSprite = new QGraphicsSimpleTextItem( 0, field );
    QFont font( KGlobalSettings::generalFont().family(), 18 );
    textSprite->setFont( font );

    shieldOn = false;
    refreshRate = REFRESH_DELAY;

    readSprites();

    shieldTimer = new QTimer( this );
    connect( shieldTimer, SIGNAL(timeout()), this, SLOT(hideShield()) );
    mTimerId = -1;

    shipPower = MAX_POWER_LEVEL;
    vitalsChanged = true;
    mBrakeCount = 0;
    mShootCount = 0;
    mShieldCount = 0;
}

// - - -

KAsteroidsView::~KAsteroidsView()
{
    delete textSprite;
    delete shield;
    delete ship;
    delete field;
    // delete all pixmap arrays
    qDeleteAll( mAnimation.begin(), mAnimation.end() );
    mAnimation.clear();
}

// - - -

void KAsteroidsView::reset()
{
    qDeleteAll( rocks );
    qDeleteAll( missiles );
    qDeleteAll( bits );
    qDeleteAll( exhaust );
    qDeleteAll( powerups );

    rocks.clear();
    missiles.clear();
    bits.clear();
    exhaust.clear();
    powerups.clear();

    shotsFired = 0;
    shotsHit = 0;

    rockSpeed = 1.0;
    powerupSpeed = 1.0;
    mFrameNum = 0;
    mPaused = false;

    ship->hide();
    shield->hide();

    if ( mTimerId >= 0 ) {
	killTimer( mTimerId );
	mTimerId = -1;
    }
}

// - --

void KAsteroidsView::newGame()
{
    if ( shieldOn )
    {
      shield->hide();
      shieldOn = false;
    }
    reset();
    if ( mTimerId < 0 )
        mTimerId = startTimer( REFRESH_DELAY );
    emit updateVitals();
}

// - - -

void KAsteroidsView::endGame()
{
}

void KAsteroidsView::pause( bool p )
{
    if ( !mPaused && p ) {
	if ( mTimerId >= 0 ) {
	    killTimer( mTimerId );
	    mTimerId = -1;
	}
    } else if ( mPaused && !p )
        mTimerId = startTimer( REFRESH_DELAY );
    mPaused = p;
}

// - - -

void KAsteroidsView::newShip()
{
    ship->setPos( field->width()/2, field->height()/2 );
    ship->setFrame(0);
    shield->setPos( field->width()/2, field->height()/2 );
    shield->setFrame(0);
    ship->setVelocity( 0.0, 0.0 );
    shipDx = 0;
    shipDy = 0;
    shipAngle = 0;
    rotateL = false;
    rotateR = false;
    thrustShip = false;
    shootShip = false;
    brakeShip = false;
    teleportShip = false;
    shieldOn = true;
    shootDelay = 0;
    shipPower = MAX_POWER_LEVEL;
    rotateRate = ROTATE_RATE;
    rotateSlow = 0;

    mBrakeCount = 0;
    mTeleportCount = 0;
    mShootCount = 0;

    ship->show();
    shield->show();
    mShieldCount = 1;   // just in case the ship appears on a rock.
    shieldTimer->setSingleShot( true );
    shieldTimer->start( 1000 );
}

void KAsteroidsView::setShield( bool s )
{
    if ( shieldTimer->isActive() && !s ) {
        shieldTimer->stop();
        hideShield();
    } else {
        shieldOn = s && mShieldCount;
    }
}

void KAsteroidsView::brake( bool b )
{
    if ( mBrakeCount )
    {
        if ( brakeShip && !b )
        {
            rotateL = false;
            rotateR = false;
            thrustShip = false;
            rotateRate = ROTATE_RATE;
        }

        brakeShip = b;
    }
}

// - - -

void KAsteroidsView::readSprites()
{
    QString sprites_prefix =
        KGlobal::dirs()->findResourceDir("sprite", "rock1/rock10000.png");

    int i = 0;
    while( kas_animations[i].id )
    {
        QFileInfo fi( sprites_prefix + kas_animations[i].path );
        QList<QPixmap> *frames = new QList<QPixmap>;
        foreach( const QString &entry, QDir( fi.path(), fi.fileName() ).entryList() )
            frames->append( QPixmap( fi.path() + '/' + entry ) );
        mAnimation.insert( kas_animations[i].id, frames );
        i++;
    }

    ship = new KAstSprite( mAnimation.value(ID_SHIP), field );
    ship->hide();

    shield = new KShield( mAnimation.value(ID_SHIELD), field );
    shield->hide();
}

// - - -

void KAsteroidsView::addRocks( int num )
{
    for ( int i = 0; i < num; i++ )
    {
        KRock *rock = new KRock( mAnimation.value(ID_ROCK_LARGE), field,
                                 ID_ROCK_LARGE, krandom.getLong(2),
				 krandom.getLong(2) ? -1 : 1 );
        double dx = (2.0 - krandom.getDouble()*4.0) * rockSpeed;
        double dy = (2.0 - krandom.getDouble()*4.0) * rockSpeed;
        rock->setVelocity( dx, dy );
	rock->setFrame( krandom.getLong( rock->frameCount() ) );
        if ( dx > 0 )
        {
            if ( dy > 0 )
                rock->setPos( 5, 5 );
            else
                rock->setPos( 5, field->height() - 25 );
            rock->setFrame(0);
        }
        else
        {
            if ( dy > 0 )
                rock->setPos( field->width() - 25, 5 );
            else
                rock->setPos( field->width() - 25, field->height() - 25 );
            rock->setFrame(0);
        }
        rock->show( );
        rocks.append( rock );
    }
}

// - - -

void KAsteroidsView::showText( const QString &text, const QColor &color, bool scroll )
{
//    textSprite->setTextFlags( AlignHCenter | AlignVCenter );
    textSprite->setText( text );
    textSprite->setPen( color );

    if ( scroll ) {
	textSprite->setPos( (field->width()-textSprite->boundingRect().width()) / 2,
			    -textSprite->boundingRect().height() );
	textDy = TEXT_SPEED;
    } else {
	textSprite->setPos( (field->width()-textSprite->boundingRect().width()) / 2,
			  (field->height()-textSprite->boundingRect().height()) / 2 );
	textDy = 0;
    }
    textSprite->show();
}

// - - -

void KAsteroidsView::hideText()
{
    textDy = -TEXT_SPEED;
}

// - - -

void KAsteroidsView::resizeEvent(QResizeEvent* event)
{
    QWidget::resizeEvent(event);
    field->setSceneRect(0, 0, width()-4, height()-4);
    view->resize(width(),height());
}

// - - -

void KAsteroidsView::timerEvent( QTimerEvent * )
{
    field->advance();

    // move rocks forward
    QLinkedList<KRock*>::iterator it;
    for ( it = rocks.begin(); it != rocks.end(); ++it ) {
        (*it)->nextFrame();
        wrapSprite( *it );
    }

    wrapSprite( ship );

    // check for missile collision with rocks.
    processMissiles();

    // these are generated when a ship explodes
    QLinkedList<KBit*>::iterator it1 = bits.begin();
    while( it1 != bits.end() )
    {
        if ( (*it1)->expired() )
        {
            delete *it1;
            it1 = bits.erase( it1 );
        }
        else
        {
            (*it1)->growOlder();
            (*it1)->setFrame( ( (*it1)->frame()+1 ) % (*it1)->frameCount() );
            ++it1;
        }
    }

    qDeleteAll( exhaust );
    exhaust.clear();

    // move / rotate ship.
    // check for collision with a rock.
    processShip();

    // move powerups and check for collision with player and missiles
    processPowerups();

    if ( textSprite->isVisible() )
    {
        if ( textDy < 0 &&
	     textSprite->boundingRect().y() <= -textSprite->boundingRect().height() )
            textSprite->hide();
        else
        {
            textSprite->moveBy( 0, textDy );
        }
        if ( textSprite->sceneBoundingRect().y() > (field->height()-textSprite->boundingRect().height())/2 )
            textDy = 0;
    }

    if ( vitalsChanged && !(mFrameNum % 10) ) {
        emit updateVitals();
        vitalsChanged = false;
    }

    mFrameNum++;
}

void KAsteroidsView::wrapSprite( KAstSprite *s )
{
    int x = int(s->x() + s->boundingRect().width() / 2);
    int y = int(s->y() + s->boundingRect().height() / 2);

    if ( x > field->width() )
        s->setPos( s->x() - field->width(), s->y() );
    else if ( x < 0 )
        s->setPos( field->width() + s->x(), s->y() );

    if ( y > field->height() )
        s->setPos( s->x(), s->y() - field->height() );
    else if ( y < 0 )
        s->setPos( s->x(), field->height() + s->y() );
}

// - - -

void KAsteroidsView::rockHit( KRock *hit )
{
    KPowerup *nPup = 0;
    int rnd = static_cast<int>(krandom.getDouble()*30.0) % 30;
    switch( rnd )
    {
      case 4:
      case 5:
        nPup = new KPowerup( mAnimation.value(ID_ENERGY_POWERUP), field,
                             ID_ENERGY_POWERUP );
        break;
      case 10:
//        nPup = new KPowerup( animation[ID_TELEPORT_POWERUP], field,
//                             ID_TELEPORT_POWERUP );
        break;
      case 15:
        nPup = new KPowerup( mAnimation.value(ID_BRAKE_POWERUP), field,
                                  ID_BRAKE_POWERUP );
        break;
      case 20:
        nPup = new KPowerup( mAnimation.value(ID_SHIELD_POWERUP), field,
                                  ID_SHIELD_POWERUP );
        break;
      case 24:
      case 25:
        nPup = new KPowerup( mAnimation.value(ID_SHOOT_POWERUP), field,
                                  ID_SHOOT_POWERUP );
        break;
    }
    if ( nPup )
    {
        double r = 0.5 - krandom.getDouble();
        nPup->setPos( hit->x(), hit->y() );
        nPup->setFrame(0);
        nPup->setVelocity( hit->xVelocity() + r, hit->yVelocity() + r );
        nPup->show( );
        powerups.append( nPup );
    }

    if ( hit->type() == ID_ROCK_LARGE || hit->type() == ID_ROCK_MEDIUM )
    {
        // break into smaller rocks
        double addx[4] = { 1.0, 1.0, -1.0, -1.0 };
        double addy[4] = { -1.0, 1.0, -1.0, 1.0 };

        double dx = hit->xVelocity();
        double dy = hit->yVelocity();

	double maxRockSpeed = MAX_ROCK_SPEED * rockSpeed;
        if ( dx > maxRockSpeed )
            dx = maxRockSpeed;
        else if ( dx < -maxRockSpeed )
            dx = -maxRockSpeed;
        if ( dy > maxRockSpeed )
            dy = maxRockSpeed;
        else if ( dy < -maxRockSpeed )
            dy = -maxRockSpeed;

        KRock *nrock;

        for ( int i = 0; i < 4; i++ )
        {
            double r = rockSpeed/2 - krandom.getDouble()*rockSpeed;
            if ( hit->type() == ID_ROCK_LARGE )
            {
                nrock = new KRock( mAnimation.value(ID_ROCK_MEDIUM), field,
                                   ID_ROCK_MEDIUM, krandom.getLong(2),
				   krandom.getLong(2) ? -1 : 1 );
                emit rockHit( 0 );
            }
            else
            {
                nrock = new KRock( mAnimation.value(ID_ROCK_SMALL), field,
                                   ID_ROCK_SMALL, krandom.getLong(2),
				   krandom.getLong(2) ? -1 : 1 );
                emit rockHit( 1 );
            }

            nrock->setPos( hit->x(), hit->y() );
            nrock->setVelocity( dx+addx[i]*rockSpeed+r, dy+addy[i]*rockSpeed+r );
	    nrock->setFrame( krandom.getLong( nrock->frameCount() ) );
            nrock->show( );
            rocks.append( nrock );
        }
    }
    else if ( hit->type() == ID_ROCK_SMALL )
        emit rockHit( 2 );
    delete hit;
    rocks.removeAll( hit );
    if ( rocks.count() == 0 )
        emit rocksRemoved();
}

void KAsteroidsView::reducePower( int val )
{
    shipPower -= val;
    if ( shipPower <= 0 )
    {
        shipPower = 0;
        thrustShip = false;
        if ( shieldOn )
        {
            shieldOn = false;
            shield->hide();
        }
    }
    vitalsChanged = true;
}

void KAsteroidsView::addExhaust( double x, double y, double dx,
                                 double dy, int count )
{
    for ( int i = 0; i < count; i++ )
    {
        KExhaust *e = new KExhaust( mAnimation.value(ID_EXHAUST), field );
        e->setPos( x + 2 - krandom.getDouble()*4, y + 2 - krandom.getDouble()*4 );
        e->setVelocity( dx, dy );
        e->show( );
        exhaust.append( e );
    }
}

void KAsteroidsView::processMissiles()
{
    KMissile *missile;

    // if a missile has hit a rock, remove missile and break rock into smaller
    // rocks or remove completely.
    QLinkedList<KMissile*>::iterator it = missiles.begin();
    while( it != missiles.end() )
    {
        missile = *it;
        missile->growOlder();

        if ( missile->expired() )
        {
            // it will point to next-after-erased missile
            delete *it;
            it = missiles.erase( it );
            continue;
        }

        wrapSprite( missile );

        QList<QGraphicsItem*> hits = missile->collidingItems();
        QList<QGraphicsItem*>::Iterator hit;
        bool missileErased = false;
        for ( hit = hits.begin(); hit != hits.end(); ++hit )
        {
            if ( (*hit)->type() >= ID_ROCK_LARGE &&
                 (*hit)->type() <= ID_ROCK_SMALL )
            {
                shotsHit++;
                rockHit( dynamic_cast<KRock*>(*hit) );
                delete *it;
                it = missiles.erase( it );
                missileErased = true;
                break;
            }
        }
        // we need to increment only if erase haven't done this for us already
        if(!missileErased)
            ++it;
    }
}

// - - -

void KAsteroidsView::processShip()
{
    if ( ship->isVisible() )
    {
        if ( shieldOn )
        {
            shield->show();
            reducePower( SHIELD_ON_COST );
            static int sf = 0;
            sf++;

            if ( sf % 2 )
                shield->setFrame( (shield->frame()+1) % shield->frameCount() );
            shield->setPos( ship->x() - 9, ship->y() - 9 );

            QList<QGraphicsItem*> hits = shield->collidingItems(Qt::IntersectsItemBoundingRect);
            QList<QGraphicsItem*>::Iterator it;
            for ( it = hits.begin(); it != hits.end(); ++it )
            {
                if ( (*it)->type() >= ID_ROCK_LARGE &&
                     (*it)->type() <= ID_ROCK_SMALL )
                {
                    int factor;
                    switch ( (*it)->type() )
                    {
                        case ID_ROCK_LARGE:
                            factor = 3;
                            break;

                        case ID_ROCK_MEDIUM:
                            factor = 2;
                            break;

                        default:
                            factor = 1;
                    }

                    if ( factor > mShieldCount )
                    {
                        // shield not strong enough
                        shieldOn = false;
                        break;
                    }
                    rockHit( dynamic_cast<KRock*>(*it) );
                    // the more shields we have the less costly
                    reducePower( factor * (SHIELD_HIT_COST - mShieldCount*2) );
                }
            }
        }

        if ( !shieldOn )
        {
            shield->hide();
            QList<QGraphicsItem*> hits = ship->collidingItems();
            QList<QGraphicsItem*>::Iterator it;
            for ( it = hits.begin(); it != hits.end(); ++it )
            {
                if ( (*it)->type() >= ID_ROCK_LARGE &&
                     (*it)->type() <= ID_ROCK_SMALL )
                {
                    KBit *bit;
                    for ( int i = 0; i < 12; i++ )
                    {
                      bit = new KBit( mAnimation.value(ID_BIT), field );
                      bit->setPos( ship->x() + 5 - krandom.getDouble() * 10,
                                 ship->y() + 5 - krandom.getDouble() * 10);
                      bit->setFrame( krandom.getLong(bit->frameCount()) );
                      bit->setVelocity( 1-krandom.getDouble()*2,
                                        1-krandom.getDouble()*2 );
                      bit->setDeath( 60 + krandom.getLong(60) );
                      bit->show( );
                      bits.append( bit );
                    }
                    ship->hide();
                    shield->hide();
                    emit shipKilled();
                    break;
                }
            }
        }


        if ( rotateSlow )
            rotateSlow--;

        if ( rotateL )
        {
            shipAngle -= rotateSlow ? 1 : rotateRate;
            if ( shipAngle < 0 )
                shipAngle += SHIP_STEPS;
        }

        if ( rotateR )
        {
            shipAngle += rotateSlow ? 1 : rotateRate;
            if ( shipAngle >= SHIP_STEPS )
                shipAngle -= SHIP_STEPS;
        }

        double angle = shipAngle * PI_X_2 / SHIP_STEPS;
        double cosangle = cos( angle );
        double sinangle = sin( angle );

        if ( brakeShip )
        {
            thrustShip = false;
            rotateL = false;
            rotateR = false;
            rotateRate = ROTATE_RATE;
            if ( fabs(shipDx) < 2.5 && fabs(shipDy) < 2.5 )
            {
                shipDx = 0.0;
                shipDy = 0.0;
                ship->setVelocity( shipDx, shipDy );
                brakeShip = false;
            }
            else
            {
                double motionAngle = atan2( -shipDy, -shipDx );
                if ( angle > M_PI )
                    angle -= PI_X_2;
                double angleDiff = angle - motionAngle;
                if ( angleDiff > M_PI )
                    angleDiff = PI_X_2 - angleDiff;
                else if ( angleDiff < -M_PI )
                    angleDiff = PI_X_2 + angleDiff;
                double fdiff = fabs( angleDiff );
                if ( fdiff > 0.08 )
                {
                    if ( angleDiff > 0 )
                        rotateL = true;
                    else if ( angleDiff < 0 )
                        rotateR = true;
                    if ( fdiff > 0.6 )
                        rotateRate = mBrakeCount + 1;
                    else if ( fdiff > 0.4 )
                        rotateRate = 2;
                    else
                        rotateRate = 1;

                    if ( rotateRate > 5 )
                        rotateRate = 5;
                }
                else if ( fabs(shipDx) > 1 || fabs(shipDy) > 1 )
                {
                    thrustShip = true;
                    // we'll make braking a bit faster
                    shipDx += cosangle/6 * (mBrakeCount - 1);
                    shipDy += sinangle/6 * (mBrakeCount - 1);
                    reducePower( BRAKE_ON_COST );
                    addExhaust( ship->x() + 20 - cosangle*22,
                                ship->y() + 20 - sinangle*22,
                                shipDx-cosangle, shipDy-sinangle,
                                mBrakeCount+1 );
                }
            }
        }

        if ( thrustShip )
        {
	    // The ship has a terminal velocity, but trying to go faster
	    // still uses fuel (can go faster diagonally - don't care).
	    double thrustx = cosangle/4;
	    double thrusty = sinangle/4;
	    if ( fabs(shipDx + thrustx) < MAX_SHIP_SPEED )
		shipDx += thrustx;
	    if ( fabs(shipDy + thrusty) < MAX_SHIP_SPEED )
		shipDy += thrusty;
	    ship->setVelocity( shipDx, shipDy );
            reducePower( 1 );
            addExhaust( ship->x() + 20 - cosangle*20,
                        ship->y() + 20 - sinangle*20,
                        shipDx-cosangle, shipDy-sinangle, 3 );
        }

        ship->setFrame( shipAngle );

        if ( shootShip )
        {
            if ( !shootDelay && (int)missiles.count() < mShootCount + 2 )
            {
              KMissile *missile = new KMissile( mAnimation.value(ID_MISSILE), field );
              missile->setPos( 21+ship->x()+cosangle*10,
                             21+ship->y()+sinangle*10 );
              missile->setFrame(0);
              missile->setVelocity( shipDx + cosangle*MISSILE_SPEED,
                                   shipDy + sinangle*MISSILE_SPEED );
              missile->show( );
              missiles.append( missile );
              shotsFired++;
              reducePower( 1 );

              shootDelay = 5;
            }

            if ( shootDelay )
              shootDelay--;
        }

        if ( teleportShip )
        {
            int ra = rand() % 10;
            if( ra == 0 )
            ra += rand() % 20;
            int xra = ra * 60 + ( (rand() % 20) * (rand() % 20) );
            int yra = ra * 50 - ( (rand() % 20) * (rand() % 20) );
            ship->setPos( xra, yra );
        }

        vitalsChanged = true;
    }
}

// - - -

void KAsteroidsView::processPowerups()
{
    if ( !powerups.isEmpty() )
    {
        // if player gets the powerup remove it from the screen, if option
        // "Can destroy powerups" is enabled and a missile hits the powerup
        // destroy it

        KPowerup *pup;
        QLinkedList<KPowerup*>::iterator it = powerups.begin();

        while( it != powerups.end() )
        {
            pup = *it;
            pup->growOlder();

            if( pup->expired() )
            {
                delete *it;
                it = powerups.erase( it );
                continue;
            }

            wrapSprite( pup );

            QList<QGraphicsItem*> hits = pup->collidingItems(Qt::IntersectsItemBoundingRect);
            QList<QGraphicsItem*>::Iterator hIter;
            bool pupErased = false;
            for ( hIter = hits.begin(); hIter != hits.end(); ++hIter )
            {
                if ( (*hIter) == ship || (*hIter) == shield )
                {
                    switch( pup->type() )
                    {
                      case ID_ENERGY_POWERUP:
                        shipPower += 150;
                        if ( shipPower > MAX_POWER_LEVEL )
                            shipPower = MAX_POWER_LEVEL;
                        break;
                      case ID_TELEPORT_POWERUP:
                        mTeleportCount++;
                        break;
                      case ID_BRAKE_POWERUP:
                        if ( mBrakeCount < MAX_BRAKES )
                            mBrakeCount++;
                        break;
                      case ID_SHIELD_POWERUP:
                        if ( mShieldCount < MAX_SHIELDS )
                            mShieldCount++;
                        break;
                      case ID_SHOOT_POWERUP:
                        if ( mShootCount < MAX_FIREPOWER )
                            mShootCount++;
                        break;
                    }


                    delete *it;
                    it = powerups.erase( it );
                    pupErased = true;
                    vitalsChanged = true;
                    break;
                } 
                else if ( (*hIter)->type() == ID_MISSILE )
                {
                    if ( Settings::canDestroyPowerups() )
                    {
                        delete *it;
                        it = powerups.erase( it );
                        pupErased = true;
                        break;
                    }
                }
            }
            // we need to increment only if erase haven't done this for us already
            if( !pupErased )
                ++it; // QLinkedList<KPowerup*>::iterator
        }
    }         // -- if( powerups.isEmpty() )
}

// - - -

void KAsteroidsView::hideShield()
{
    shield->hide();
    mShieldCount = 0;
    shieldOn = false;
}


// - - -

