/***************************************************************************
 *   Copyright (C) 2004 by Leonid Zeitlin                                  *
 *   lz@europe.com                                                         *
 *                                                                         *
 *   QCA2 port Copyright (C) 2006 by Brad Hards <bradh@kde.org>            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.         *
 ***************************************************************************/

#include "kfile_cert.h"

#include <kgenericfactory.h>
#include <qfile.h>

typedef KGenericFactory<CertPlugin> CertFactory;

K_EXPORT_COMPONENT_FACTORY(kfile_cert, CertFactory("kfile-cert"))

CertPlugin::CertPlugin(QObject *parent, const QStringList &args)
  : KFilePlugin(parent, args)
{
    KFileMimeTypeInfo* info = addMimeTypeInfo( "application/x-x509-ca-cert" );

    // our new group
    KFileMimeTypeInfo::GroupInfo* group = 0L;
    KFileMimeTypeInfo::ItemInfo* item;

    group = addGroupInfo(info, "certInfo", i18n("Certificate Information"));
    item = addItemInfo(group, "ValidFrom", i18n("Valid From"), QVariant::DateTime);
    item = addItemInfo(group, "ValidUntil", i18n("Valid Until"), QVariant::DateTime);
    item = addItemInfo(group, "State", i18n("State"), QVariant::String);
    item = addItemInfo(group, "SerialNo", i18n("Serial Number"), QVariant::String);

    group = addGroupInfo(info, "certSubjectInfo", i18n("Subject"));
    item = addItemInfo(group, "O", i18n("Organization"), QVariant::String);
    item = addItemInfo(group, "OU", i18n("Organizational Unit"), QVariant::String);
    item = addItemInfo(group, "L", i18n("Locality"), QVariant::String);
    item = addItemInfo(group, "S", i18n("State"), QVariant::String);
    item = addItemInfo(group, "C", i18n("Country"), QVariant::String);
    item = addItemInfo(group, "CN", i18n("Common Name"), QVariant::String);
    item = addItemInfo(group, "E", i18n("Email"), QVariant::String);
    item = addItemInfo(group, "DNS", i18n("DNS"), QVariant::String);
    item = addItemInfo(group, "XMPP", i18n("XMPP"), QVariant::String);

    group = addGroupInfo(info, "certIssuerInfo", i18n("Issuer"));
    item = addItemInfo(group, "O", i18n("Organization"), QVariant::String);
    item = addItemInfo(group, "OU", i18n("Organizational Unit"), QVariant::String);
    item = addItemInfo(group, "L", i18n("Locality"), QVariant::String);
    item = addItemInfo(group, "S", i18n("State"), QVariant::String);
    item = addItemInfo(group, "C", i18n("Country"), QVariant::String);
    item = addItemInfo(group, "CN", i18n("Common Name"), QVariant::String);
    item = addItemInfo(group, "E", i18n("Email"), QVariant::String);
    item = addItemInfo(group, "DNS", i18n("DNS"), QVariant::String);
    item = addItemInfo(group, "XMPP", i18n("XMPP"), QVariant::String);
}

void CertPlugin::appendCertInfoItem( KFileMetaInfoGroup &group,
                                     const char* key,
                                     const QList<QString> values )
{
    const QStringList elements( values );
    if ( ! elements.isEmpty() )
    {
        appendItem( group, key,  elements.join( "," ) );
    }
}

void CertPlugin::appendCertificateInfo(KFileMetaInfoGroup &group,
                                       const QCA::CertificateInfo &info)
{
    appendCertInfoItem( group,  "O", info.values( QCA::Organization ) );
    appendCertInfoItem( group,  "OU", info.values( QCA::OrganizationalUnit ) );
    appendCertInfoItem( group,  "L", info.values( QCA::Locality ) );
    appendCertInfoItem( group,  "S", info.values( QCA::State ) );
    appendCertInfoItem( group,  "C", info.values( QCA::Country ) );
    appendCertInfoItem( group,  "CN", info.values( QCA::CommonName ) );
    appendCertInfoItem( group,  "E", info.values( QCA::Email ) );
    appendCertInfoItem( group,  "DNS", info.values( QCA::DNS ) );
    appendCertInfoItem( group,  "XMPP", info.values( QCA::XMPP ) );
}

static bool stringFromFile(const QString &fileName, QString *s)
{
    QFile f(fileName);
    if(!f.open(QFile::ReadOnly))
        return false;
    QTextStream ts(&f);
    *s = ts.readAll();
    f.close();
    return true;
}

static bool arrayFromFile(const QString &fileName, QByteArray *ba)
{
    QFile f(fileName);
    if(!f.open(QFile::ReadOnly))
        return false;
    *ba = f.readAll();
    f.close();
    return true;
}

static QString validityToString( const QCA::Validity validity )
{
    switch( validity )
    {
    case QCA::ValidityGood:
        return i18n( "Valid" );
        break;
    case QCA::ErrorRejected:
        return i18n( "Purpose rejected" );
        break;
    case QCA::ErrorUntrusted:
        return i18n( "Untrusted" );
        break;
    case QCA::ErrorSignatureFailed:
        return i18n( "Bad signature" );
        break;
    case QCA::ErrorInvalidCA:
        return i18n( "Invalid CA" );
        break;
    case QCA::ErrorInvalidPurpose:
        // not sure that can happen here, but cover it anyway
        return i18n( "Invalid Purpose" );
        break;
    case QCA::ErrorSelfSigned:
        return i18n( "Untrusted self-signed" );
        break;
    case QCA::ErrorRevoked:
        return i18n( "Revoked" );
        break;
    case QCA::ErrorPathLengthExceeded:
        return i18n( "Path length exceeded" );
        break;
    case QCA::ErrorExpired:
        return i18n( "Expired" ); // could maybe not yet be valid too
        break;
    case QCA::ErrorExpiredCA:
        return i18n( "CA Expired" );
        break;
    case QCA::ErrorValidityUnknown:
    default:
        return i18n( "Unknown" );
    }
}

void CertPlugin::dumpCertInfo( KFileMetaInfo &info, const QCA::Certificate &cert )
{
    KFileMetaInfoGroup group = appendGroup(info, "certInfo");
    appendItem( group,  "ValidFrom", cert.notValidBefore() );
    appendItem( group,  "ValidUntil", cert.notValidAfter() );
    appendItem( group,  "State",  validityToString( cert.validate( QCA::systemStore(), QCA::CertificateCollection() ) ) );
    appendItem( group,  "SerialNo", cert.serialNumber().toString() );

    group = appendGroup(info, "certIssuerInfo");
    appendCertificateInfo( group,  cert.issuerInfo() );

    group = appendGroup(info, "certSubjectInfo");
    appendCertificateInfo( group,  cert.subjectInfo() );
}

bool CertPlugin::readInfo(KFileMetaInfo& info, uint /*what*/)
{
    QCA::Initializer init;

    QString certString;
    if ( ! stringFromFile( info.path(), &certString ) )
    {
        // could not read the file
        return false;
    }

    if ( QCA::isSupported( "cert" ) )
    {
        QCA::ConvertResult importResult;
        QCA::Certificate cert = QCA::Certificate::fromPEM( certString, &importResult );

        if ( ( QCA::ConvertGood == importResult) && ( ! cert.isNull() ) )
        {
            // this is a valid certificate
            dumpCertInfo( info, cert );
            return true;
        }
    }

    // might it be a Certificate Signing Request?
    if ( QCA::isSupported( "csr" ) )
    {
        QCA::ConvertResult importResult;
        QCA::CertificateRequest csr = QCA::CertificateRequest::fromPEM( certString, &importResult );
        if ( ( QCA::ConvertGood == importResult ) && ( !csr.isNull() ) )
        {
            // we have a valid CertificateRequest
            KFileMetaInfoGroup  group = appendGroup(info, "certSubjectInfo");
            appendCertificateInfo( group,  csr.subjectInfo() );

            return true;
        }
    }

    // might it be a Certificate Revocation List?
    if ( QCA::isSupported( "crl" ) )
    {
        QCA::ConvertResult importResult;
        QCA::CRL crl = QCA::CRL::fromPEM( certString,  &importResult );
        if ( ( QCA::ConvertGood == importResult ) && ( !crl.isNull() ) )
        {
            // we have a good CRL;
            KFileMetaInfoGroup group = appendGroup(info, "certInfo");
            // This is stretching it a bit, but metadata is like that....
            appendItem( group,  "ValidFrom", crl.thisUpdate() );
            appendItem( group,  "ValidUntil", crl.nextUpdate() );
            appendItem( group,  "SerialNo", QString( crl.number() ) );

            group = appendGroup(info, "certIssuerInfo");
            appendCertificateInfo( group,  crl.issuerInfo() );

            return true;
        }
    }

    // maybe it is DER format?
    QByteArray derArray;
    if ( ! arrayFromFile( info.path(), &derArray ) )
    {
        // could not read the file
        return false;
    }

    if ( QCA::isSupported( "cert" ) )
    {
        QCA::ConvertResult importResult;
        QCA::Certificate cert = QCA::Certificate::fromDER( derArray, &importResult );

        if ( ( QCA::ConvertGood == importResult) && ( ! cert.isNull() ) )
        {
            // this is a valid certificate
            dumpCertInfo( info, cert );
            return true;
        }
    }

    // We could try a CRL or CSR in DER format, but I've never seen one.

    return false;
}

#include "kfile_cert.moc"
