/***************************************************************************
 *   Copyright (C) 2004, 2005 by Johnathan Burchill                              *
 *   jkerrb@users.sourceforge.net                                          *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "openArchiveThread.h"

#include "kdar.h"
#include "kdarEnums.h"
#include "controller.h"
#include "archiveHandle.h"
#include "kdarConfig.h"
#include "kdardeclarations.h"
#include "KDarInteraction.h"
#include "kdarListViewItem.h"
#include "kdarPostListViewItemEvent.h"
#include "kdarSetCursorEvent.h"
#include "kdarStatusBarEvent.h"

#include <kcursor.h>
#include <kdebug.h>
#include <klocale.h>

#include <qapplication.h>
#include <qstring.h>

#include <dar/libdar.hpp>
#include <dar/erreurs.hpp>

#include <string>

extern bool libdarWorking;
extern libdar::archive *theArchive;
extern kdarListViewItem *archiveItem;

openArchiveThread::openArchiveThread( controller * aController, KDarConfig * kdarconfig, bool updateListView )
    : KDarThread( aController, kdarconfig ),
      m_updateListView( updateListView )
{}

openArchiveThread::~openArchiveThread()
{
}

void openArchiveThread::run()
{
    //Every class that is derived from KDarThread must set this member variable:
    m_threadID = pthread_self();
    QString finishedMessage ( i18n( "Opening archive...done.\n" ) );
    libdarWorking = true;
    setWaitCursor();
    enableCancelButton();
    statusMessage( i18n( "Opening archive...\n" ), FOREVER );

    //Need some way to interact with user:
    KDarInteraction userInteractionDialog( m_controller );

    //check whether we have a valid archiveName:
    archiveHandle archiveHndl( m_kdc->archiveName() );
    //Check the encryption type:
    libdar::crypto_algo cryptoAlgorithm = libdar::crypto_none;
    if ( m_controller->handlesStrongCrypto() && m_kdc->cryptoAlgorithm() != KDar::CRYPTO_NONE )
    {
        //This should be performed by libdar. It's in the libdar TODO...
        kdDebug() << "openArchiveThread::run(): querying crypto algorithm." << endl;
        cryptoAlgorithm = archiveHndl.queryCryptoAlgorithm();
    }
    try
    {
        kdDebug() << "openArchiveThread::run(): opening archive..." << endl;
        theArchive = new libdar::archive( userInteractionDialog, archiveHndl.getArchiveLibdarPath(), kdar::toStdString( archiveHndl.getArchiveBasename() ), EXTENSION, cryptoAlgorithm, "", ( libdar::U_32 ) m_kdc->cryptoBlockSize(), kdar::toStdString( m_kdc->inputPipe() ), kdar::toStdString( m_kdc->outputPipe() ), kdar::toStdString( m_kdc->executeRef() ), m_kdc->verbose() );
        if ( m_updateListView )
        {
            kdarPostListViewItemEvent *postListViewEvent = new kdarPostListViewItemEvent( archiveHndl.getArchiveBasename(), true );
            QApplication::postEvent( m_controller, postListViewEvent );
        }
        kdDebug() << "openArchiveThread::run(): finished opening archive." << endl;
    }
    catch( libdar::Euser_abort &e )
    {
        kdDebug() << "openArchiveThread::run(): user cancelled operation." << endl;
        finishedMessage = i18n( "Opening archive...cancelled by user." );
        logMessage( finishedMessage );
        if ( theArchive )
        {
            delete theArchive;
            theArchive = 0;
        }
    }
    catch( libdar::Egeneric &e )
    {
        kdDebug() << "openArchiveThread::run(): caught generic exception." << endl;
        userInteractionDialog.warning( e.get_message() );
        if ( theArchive )
        {
            delete theArchive;
            theArchive = 0;
        }
        finishedMessage = i18n( "Error: unable to open the archive.\n" );
        logMessage( finishedMessage );
    }
    setNormalCursor();
    disableCancelButton();
    statusMessage( finishedMessage );
    libdarWorking = false;
    deleteMe();
}

