/***************************************************************************
 *   Copyright (C) 2004, 2005 by Johnathan Burchill                              *
 *   jkerrb@users.sourceforge.net                                          *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/
#ifndef _KDARTHREAD_H_
#define _KDARTHREAD_H_

#include "kdardeclarations.h"

#include <qdatetime.h>
#include <qstring.h>
#include <qthread.h>

#include <dar/libdar.hpp>
#include <pthread.h>

class KDarConfig;
class controller;

class KListView;

/**
 * This class serves as the base class for KDar threads. All libdar operations
 * are run as KDarThreads. This maintains GUI responsiveness.
 *
 * @short Base class for KDar threads.
 * @author Johnathan K. Burchill, jkerrb@users.sourceforge.net
 * @version 1.3.0
 */
class KDarThread : public QThread
{

public:
    /**
     * Constructor
     * @param parent the controller responsible for deleting the thread.
     * @param config the configuration object (main, create or restore) for this
     * thread.
     */
    KDarThread( controller * parent, KDarConfig * config, KListView * archiveFileView = 0 );
    /**
     * Destructor
     */
    virtual ~KDarThread();

    /**
     * Cancel a libdar operation.
     * @return returns whether the libdar operation was successfully cancelled.
     */
    bool cancelOperation();

protected:
    /**
     * Allow the user to cancel libdar operations.
     */
    void enableCancelButton();
    /**
     * Disable the cancel button. The button should be disabled when there
     * is no operation being performed.
     */
    void disableCancelButton();
    /**
     * Set the cursor to indicate "waiting".
     */
    void setWaitCursor();
    /**
     * Set the cursor back to normal.
     */
    void setNormalCursor();
    /**
     * Display a message in the status bar.
     *
     * @param message the message to display.
     * @param duration the amount of time to display @ref message, in milliseconds.
     * Set duration to "FOREVER" ( = 0 ) to display the message indefinitely.
     */
    void statusMessage( const QString & message, const int duration = DURATION );
    /**
     * Post a message to the Message window or a log file.
     *
     * @param message the message to log.
     */
    void logMessage( const QString & message, bool logFileOnly = false );
    /**
     * Opens the logFile for appending.
     */
    void openLogFile() const;
    /**
     * Closes the logFile.
     */
    void closeLogFile() const;
    /**
     * Disable sorting in the archive browser listView. This speeds up
     * listing and populating directories in the browser.
     * Enable sorting again with @ref enableBrowser.
     */
    void disableBrowser();
    /**
     * Enable sorting in the archive browser listView.
     *
     * @see disableBrowser.
     */
    void enableBrowser();
    /**
     * Start updating the progressbar.
     */
    void startProgressBarUpdates();
    /**
     * Stop updating the progressbar.
     */
    void stopProgressBarUpdates();
    /**
     * Start updating the slice size.
     */
    void startCheckFileSizeUpdates();
    /**
     * Stop updating the slize size.
     */
    void stopCheckFileSizeUpdates();
    /**
     * Returns a string of the time between startTime and endTime,
     * in terms of days,
     * hours, minutes and seconds.
     */
    QString timeDifference( const QDateTime & startTime, const QDateTime & endTime );
    /**
     * Sends a signal to the controller to delete this thread when run() is
     * finished.
     */
    void deleteMe();
    /**
     * Adds any selected listView items to the directory mask.
     *
     * @param extractionPrefix is prepended to each listView item's path.
     * For example, pass m_kdc->extractArchiveDirectory() when restoring
     * specific archive items.
     */
    libdar::ou_mask buildDirectoryMaskFromListView( QString extractionPrefix );

    controller *m_controller;
    KDarConfig *m_kdc;
    KListView *m_archiveFileView;

    QString m_endMessage;

    pthread_t m_threadID;
};

#endif // _KDARTHREAD_H_
