/***************************************************************************
 *   Copyright (C) 2004, 2005 by Johnathan Burchill,,,                           *
 *   jkerrb@users.sourceforge.net                                                        *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/
#ifndef _ARCHIVEHANDLE_H_
#define _ARCHIVEHANDLE_H_

#include <kurl.h>

#include <qstring.h>

#include <string>

#include <dar/libdar.hpp>

class KDarConfig;

/**
 * A "handle" for archive names. Provides methods associated with archive names.
 *
 * @short A "handle" for methods associated with archive names.
 * @author Johnathan K. Burchill, jkerrb@users.sourceforge.net
 * @version 2.0.0
 */

class archiveHandle
{
public:
    /**
     * Constructor
     *
     * @param archiveURL a KURL of the archive name. Can include the ".*.dar"
     * extension.
     */
    archiveHandle( KURL archiveURL );
    /**
     * Destructor
     */
    virtual ~archiveHandle();

    /**
     * Returns the path to the archive as a QString.
     */
    QString getArchivePath() const;
    /**
     * Returns the archive filename, including the extension if it exists.
     */
    QString getArchiveFilename() const;
    /**
     * Returns the archive basename. Does not include the path or the extension.
     */
    QString getArchiveBasename() const;
    /**
     * Returns the path to the archive as a libdar::path().
     */
    libdar::path getArchiveLibdarPath() const;
    /**
     * Returns the archive basename as a std::string. Useful when creating
     * libdar::masks or libdar::paths.
     */
    std::string getArchiveBasenameString() const;
    /**
     * This method checks the filesystem for the last slice of an archive.
     * It does this by looking for the largest slice number. It does not check
     * whether the last slice and the first slice belong to the same archive.
     *
     * @return Returns the number of the last slice, or 0 if there are no slices.
     */
    long lastArchiveSliceNumber() const;
    /**
     * Converts all archive name macros into their actual values.
     *
     * @param rawString the input archive name. The following macros will
     * have their values substituted:
     *
     *  - %hostname
     *  - %date
     *  - %time
     *  - %year
     *  - %month
     *  - %day
     *  - %hour
     *  - %minute
     *  - %second
     *  - %full
     *  - %diff
     *  - %refname
     */
    static QString parseStringSubstitutions( const QString & rawString, KDarConfig * kdarconfig = 0 );

    /**
     * Queries an existing archive for the encryption type. Actually,
     * this will be taken care of by libdar in a future version. In the
     * meantime, the archive header only stores whether there is encryption
     * or not, so KDar will currently only be able to open encrypted archives
     * based on blowfish.
     */
    libdar::crypto_algo queryCryptoAlgorithm();
private:
    /**
     * Removes the archive extension if the @ref archiveURL has one. This is
     * used in the constructor.
     */
    void trimFilename();

    /**
     * The archive path.
     */
    QString m_archivePath;
    /**
     * The archive filename, including the extension if it exists.
     */
    QString m_archiveFilename;
    /**
     * The archive basename, not including the path or extension.
     */
    QString m_archiveBasename;
};

#endif // _ARCHIVEHANDLE_H_
