/***************************************************************************
                          kcovermultistring.cpp  -  description
                             -------------------
    begin                : Thu Jun 29 2000
    copyright            : (C) 2000 by Pascal 'PeP' Panneels
    email                : pepouille@skynet.be
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <qfontmetrics.h>
#include <qpainter.h>
#include "kcovermultistring.h"
#include "id.h"

KCoverMultiString::KCoverMultiString()
{
	Font = new KCoverFont((char *)("Helvetica"));
	info_valid = false;
	backcolor = new QColor( Qt::white );
	textcolor = new QColor( Qt::black );
	Pixel_Separator = SEPARATOR_PIXEL_STRING;
	transparentbackground = true;
	alignoption = ALIGN_LEFT;
}

KCoverMultiString::KCoverMultiString( std::vector<QString> &text, int x, int y, int order ) : KCoverObject( x, y, order )
{
	Text = text;	
	Font = new KCoverFont((char *)("Helvetica"));
	backcolor = new QColor( Qt::white );
	textcolor = new QColor( Qt::black );
	ComputeLongestString();
	info_valid = false;
	Pixel_Separator = SEPARATOR_PIXEL_STRING;
	alignoption = ALIGN_LEFT;
	transparentbackground = true;
}

KCoverMultiString::KCoverMultiString(const KCoverMultiString &p)
: KCoverObject( p )
{
	Text = p.Text;	
	Font = new KCoverFont(*p.Font);
	ComputeLongestString();
	info_valid = p.info_valid;
	Pixel_Separator = p.Pixel_Separator;
	backcolor = new QColor( *p.backcolor );
	textcolor = new QColor( *p.textcolor );
	alignoption = p.alignoption;
	transparentbackground = p.transparentbackground;
}

KCoverMultiString::~KCoverMultiString()
{
	delete Font;
	delete backcolor;
	delete textcolor;
}

int KCoverMultiString::operator == (const KCoverMultiString &p) const
{
	if ( p.X==X && p.Y==Y && p.Order==Order && Text==p.Text )
		return 1;
	else
		return 0;
}

/** return true if clicked is on a text */
bool KCoverMultiString::IsPointInObject(QPoint &pos)
{
	QFontMetrics fm(*Font->Font);

	if ( pos.x()>=X && pos.x()<=X+Width() && pos.y()>=Y-fm.ascent() && pos.y()<=Y+Height() )
		return true;
	else
		return false;
}

// return the longest string in the vector
int KCoverMultiString::Width()
{
	if ( !info_valid )
		ComputeLongestString();
	return longest_width;
}

// return the height of all string together
int KCoverMultiString::Height()
{
	QFontMetrics fm(*Font->Font);
	return (fm.height()+Pixel_Separator)*Text.size();
}

int KCoverMultiString::RealX()
{
	return X;
}

int KCoverMultiString::RealY()
{
	return Y;
}

void KCoverMultiString::Paint( QPainter *paint,  QPoint orig)
{
	QPoint pos=QPoint(X,Y)+orig;
	QFontMetrics fm(*Font->Font);
	int H = fm.height();
	
	// compute the longest
	if ( !info_valid )
		ComputeLongestString();

	paint->setFont(*Font->Font);
	paint->setPen(*textcolor);
	paint->setBackgroundMode(transparentbackground ? Qt::TransparentMode : Qt::OpaqueMode );
	paint->setBackgroundColor(*backcolor);    		
	
	int px = pos.x();
	int py = pos.y();
	int CX = px;
	int CY = py;

	for ( unsigned int i=0; i<Text.size(); i++ )
		{
		int CW = fm.width( Text[i].latin1() );
		
		switch (alignoption)
			{
			case ALIGN_LEFT :
				CX = px;
				break;
			case ALIGN_CENTER :
				CX = px + (longest_width-CW) / 2;
				break;
			case ALIGN_RIGHT :								
				CX = px + (longest_width-CW);
				break;
			}
		paint->drawText( CX, CY, Text[i].latin1() );
		CY += Pixel_Separator + H ;
		}	
}

// compute the widthest string in pixels of the text
void KCoverMultiString::ComputeLongestString()
{
	QFontMetrics fm(*Font->Font);
	longest_string_index = 0;
	longest_width = 0;
	int t;
	
	for ( unsigned int i=0; i<Text.size();i++ )
		{
		QRect r = fm.boundingRect(Text[i].latin1());
		if ( (t=r.width()) > longest_width )
			{
			longest_width = t;
			longest_string_index = i;
			}
		}
	info_valid=true;
}

/** save an object */
void KCoverMultiString::Save(QFile &file)
{
	char tmp[256];
	int r1, g1, b1, r2, g2, b2;
	
	// tag TAG_MULTI_STRING
	tmp[0] = TAG_MULTI_STRING;
	file.writeBlock( tmp, 1 );
	// call the base class to save the coordinates
	KCoverObject::Save(file);
	// save colors, alignement, transparent info
	backcolor->rgb( &r1, &g1, &b1 );
	textcolor->rgb( &r2, &g2, &b2 );
	sprintf( tmp,"%c%c%c%c%c%c%c%c", alignoption, transparentbackground, r1, g1, b1, r2, g2, b2 );
	file.writeBlock( tmp,8 );
	Font->Save(file);
	// pixel separator info
	file.writeBlock( (char *)&Pixel_Separator, sizeof(int));
	// save the number of text lines
	r1 = Text.size();
	file.writeBlock( (char *)&r1, sizeof(int) );
	for ( int i=0; i<r1; i++ )
		{
		r2 = Text[i].length();
		file.writeBlock( (char *)&r2, sizeof(int) );
		file.writeBlock( Text[i].latin1(), r2 );
		}
}

/** load an object */
void KCoverMultiString::Load(QFile &file, int version)
{
	char tmp[512];
	int r1, r2 ;
	int err;
	
	err = file.readBlock( tmp, 1 );
	if ( tmp[0]!= TAG_MULTI_STRING )
		return; // error : not a string
	KCoverObject::Load(file, version);
	// lot of informations...
	err = file.readBlock( tmp, 8 );
	alignoption = tmp[0];
	transparentbackground = tmp[1];
	backcolor = new QColor( (unsigned char)tmp[2], (unsigned char)tmp[3], (unsigned char)tmp[4] );
	textcolor = new QColor( (unsigned char)tmp[5], (unsigned char)tmp[6], (unsigned char)tmp[7] );
	// font
	Font = new KCoverFont((char *)("Helvetica"));
	Font->Load( file, version );
	// pixel separator
	err = file.readBlock( (char *)&Pixel_Separator, sizeof(int) );
	// texts
	err = file.readBlock( (char *)&r2, sizeof(int) );	// number of lines
	for ( int i=0;i<r2;i++ )
		{
		err = file.readBlock( (char *)&r1, sizeof(int) );
		err = file.readBlock( tmp, r1 );
		tmp[r1] = 0;
		Text.push_back( QString( tmp ) );
		}
}

const int KCoverMultiString::GetTopAlignDelta()
{
	return GetBottomAlignDelta();
}

const int KCoverMultiString::GetBottomAlignDelta()
{
	ComputeLongestString();
	QFontMetrics fm(*Font->Font);
	QRect r = fm.boundingRect(Text[longest_string_index].latin1());
	return r.height();
}

const int KCoverMultiString::GetMiddleAlignDelta()
{
	return GetBottomAlignDelta()/2;
}
