/***************************************************************************
 *   Copyright (C) 2004 by Roberto Virga                                   *
 *   rvirga@users.sf.net                                                   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef SETIDATA_H
#define SETIDATA_H

#include <qdatetime.h>
#include <qdom.h>
#include <qstring.h>
#include <qvaluelist.h>

QString formatRA(double ra);
QString formatDec(double dec, bool sign=true);

struct SETITapeInfo
{
  QString name;
  QDateTime start_time;
  double start_time_jd;
  QDateTime last_block_time;
  double last_block_time_jd;
  unsigned last_block_done,
           missed,
           tape_quality;
  
  bool parse(const QDomElement &node);
};
 
struct SETICoordinateT
{
  QDateTime time;
  double ra,
         dec;
  
  bool parse(const QDomElement &node);
};

struct SETICoords
{
  QValueList<SETICoordinateT> coordinate_t;
  
  bool parse(const QDomElement &node);
};

struct SETIDataDesc
{
  struct {
    double ra,
           dec;
  } start, end;
  double true_angle_range;
  QDateTime time_recorded;
  double time_recorded_jd;
  unsigned nsamples;
  SETICoords coords;
  
  bool parse(const QDomElement &node);
  
  static double teraFLOPs(double angle_range);
  double teraFLOPs() const;
};

struct SETIReceiverCfg
{
  unsigned s4_id;
  QString name;
  double beam_width,
         center_freq,
         latitude,
         longitude,
         elevation,
         diameter;
  struct {
    double orientation;
    QValueList<double> corr_coeff;
  } az;
  QValueList<double> zen_corr_coeff;
  
  bool parse(const QDomElement &node);
};

struct SETIRecorderCfg
{
  QString name;
  unsigned bits_per_sample,
           sample_rate,
           beams;
  QString version;
  
  bool parse(const QDomElement &node);
};

struct SETISplitterCfg
{
  QString version,
          data_type;
  unsigned fft_len,
           ifft_len;
  QString filter,
          window;
  
  bool parse(const QDomElement &node);
};

struct SETIChirpParameterT
{
  unsigned chirp_limit,
           fft_len_flags;
  
  bool parse(const QDomElement &node);
};

struct SETIChirps
{
  QValueList<SETIChirpParameterT> chirp_parameter_t;
  
  bool parse(const QDomElement &node);
};

struct SETIAnalysisCfg
{
  double spike_thresh;
  unsigned spikes_per_spectrum;
  struct {
    double null_chi_sq_thresh,
           chi_sq_thresh,
           power_thresh,
           peak_power_thresh;
    unsigned pot_length;
  } gauss;
  struct {
    double thresh,
           display_thresh;
    unsigned max,
             min,
             fft_max,
             pot_length;
  } pulse;
  struct {
    double thresh;
    unsigned max,
             min,
             pot_length;
  } triplet;
  struct {
    double overlap_factor;
    unsigned t_offset;
    double min_slew,
           max_slew;
  } pot;
  double chirp_resolution;
  unsigned analysis_fft_lengths;
  struct {
    unsigned boxcar_length,
             chunk_size;
  } bsmooth;
  SETIChirps chirps;
  unsigned pulse_beams,
           max_signals;
  
  bool parse(const QDomElement &node);
};

struct SETIGroupInfo
{
  QString name;
  SETITapeInfo tape_info;
  SETIDataDesc data_desc;
  SETIReceiverCfg receiver_cfg;
  SETIRecorderCfg recorder_cfg;
  SETISplitterCfg splitter_cfg;
  SETIAnalysisCfg analysis_cfg;
  
  bool parse(const QDomElement &node);
};

struct SETISubbandDesc
{
  unsigned number;
  double center,
         base,
         sample_rate;
  
  bool parse(const QDomElement &node);
};

struct SETIWorkunitHeader
{
  QString name;
  SETIGroupInfo group_info;
  SETISubbandDesc subband_desc;
  
  bool parse(const QDomElement &node);
};

struct SETIRFI
{
  unsigned checked,
           found;
};

struct SETISpike
{
  double peak_power,
         mean_power;
  QDateTime time;
  double time_jd,
         ra,
         decl;
  unsigned q_pix;
  double freq,
         detection_freq,
         barycentric_freq;
  unsigned fft_len;
  double chirp_rate;
  SETIRFI rfi;
  unsigned reserved;  
  
  static const QString type;
  bool parse(const QDomElement &node);
  double score() const;
  double resolution() const;
  double signal_ratio() const;
};

struct SETIGaussian
{
  double peak_power,
         mean_power;
  QDateTime time;
  double time_jd,
         ra,
         decl;
  unsigned q_pix;
  double freq,
         detection_freq,
         barycentric_freq;
  unsigned fft_len;
  double chirp_rate;
  SETIRFI rfi;
  unsigned reserved;  
  double sigma,
         chisqr,
         null_chisqr;
  double max_power;
  QValueList<unsigned> pot;
  
  static const QString type;
  
  bool parse(const QDomElement &node);
  
  double score() const;
  bool interesting() const;
  double resolution() const;
  double signal_ratio() const;
};

struct SETIPulse
{
  double peak_power,
         mean_power;
  QDateTime time;
  double time_jd,
         ra,
         decl;
  unsigned q_pix;
  double freq,
        detection_freq,
        barycentric_freq;
  unsigned fft_len;
  double chirp_rate;
  SETIRFI rfi;
  unsigned reserved;
  double period,
         snr,
         thresh;
  unsigned len_prof;
  QValueList<unsigned> pot;
  
  static const QString type;
  
  bool parse(const QDomElement &node);
  
  double score() const;
  double resolution() const;
  double signal_ratio() const;
};

struct SETITriplet
{
  double peak_power,
         mean_power;
  QDateTime time;
  double time_jd,
         ra,
         decl;
  unsigned q_pix;
  double freq,
         detection_freq,
         barycentric_freq;
  unsigned fft_len;
  double chirp_rate;
  SETIRFI rfi;
  unsigned reserved;
  double period;

  static const QString type; 
   
  bool parse(const QDomElement &node);
  
  double score() const;
  double resolution() const;
  double signal_ratio() const;
};

struct SETIBestSpike
{
  SETISpike spike;
  struct {
    double score;
    unsigned bin;
    double fft_ind;
  } bs; 
  
  bool parse(const QDomElement &node);
};

struct SETIBestGaussian
{
  SETIGaussian gaussian;
  struct {
    double score,
           display_power_thresh;
    unsigned bin,
             fft_ind;
  } bg; 
  
  bool parse(const QDomElement &node);
};

struct SETIBestPulse
{
  SETIPulse pulse;
  struct {
    double score;
    unsigned freq_bin;
    double time_bin;
  } bp; 
  
  bool parse(const QDomElement &node);
};

struct SETIBestTriplet
{
  SETITriplet triplet;
  struct {
    double score,
           bperiod;
    unsigned tpotind[3][2],
             freq_bin;
    double time_bin,
           scale;
  } bt; 
  
  bool parse(const QDomElement &node);
};

struct SETIState
{
  unsigned ncfft;
  double cr;
  unsigned fl;
  double prog;
  int potfreq;
  unsigned potactivity,
           signal_count;
  SETIBestSpike best_spike;
  SETIBestGaussian best_gaussian;
  SETIBestPulse best_pulse;
  SETIBestTriplet best_triplet;
  
  bool parse(const QDomElement &node);
};

struct SETIResult
{
  SETIWorkunitHeader workunit_header;
  SETIState state;
  QValueList<SETISpike> spike;
  QValueList<SETIGaussian> gaussian;
  QValueList<SETIPulse> pulse;
  QValueList<SETITriplet> triplet;
  
  bool parse(const QDomElement &node);
  
  int bestSpike(double *score=NULL) const;
  int bestGaussian(double *score=NULL) const;
  int bestPulse(double *score=NULL) const;
  int bestTriplet(double *score=NULL) const;
};

#endif
