/***************************************************************************
 *   Copyright (C) 2005 by Roberto Virga                                   *
 *   rvirga@users.sf.net                                                   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <qgl.h>
#include <qimage.h>

#include <kapplication.h>
#include <kio/netaccess.h>
#include <ktempfile.h>

#include "kbspredictormoleculelog.h"

KBSPredictorMoleculeLog *KBSPredictorMoleculeLog::s_self = NULL;

KBSPredictorMoleculeLog *KBSPredictorMoleculeLog::self()
{
  if(NULL == s_self) s_self = new KBSPredictorMoleculeLog();
  
  return s_self;
}

KBSPredictorMoleculeLog::KBSPredictorMoleculeLog(QObject *parent, const char *name)
                       : QObject(parent, name)
{
}

const KBSPredictorMoleculeLogPreferences &KBSPredictorMoleculeLog::preferences(PredictorAppType type) const
{
  return m_preferences[type];
}

void KBSPredictorMoleculeLog::setPreferences(PredictorAppType type,
                                             const KBSPredictorMoleculeLogPreferences &preferences)
{
  m_preferences[type] = preferences;
}

void KBSPredictorMoleculeLog::logWorkunit(const QString &name, const PredictorResult *data)
{
  if(!QGLFormat::hasOpenGL()) return;
  
  PredictorAppType app_type = data->app_type;
  KBSPredictorMoleculeLogPreferences preferences = this->preferences(app_type);
  
  if(!preferences.url.isValid()) return;
  
  KBSPredictorMoleculeView *view = new KBSPredictorMoleculeView();
  if(app_type == MFOLD) {
    view->setChain(data->mfold.monsster.init_chain);
    view->setSeq(data->mfold.monsster.seq);
  } else
    view->setPDB(data->charmm.protein.pdb);
  view->setStyle(preferences.style);
  view->setColoring(preferences.coloring);
  view->resize(640, 480);
  
  const QString pictureFileName = name + "." + preferences.picture.format.lower();
  KURL pictureFileURL(preferences.url, pictureFileName);
  
  if(preferences.picture.filter & KBSPredictorMoleculeLogPreferences::Workunit
       && pictureFileURL.isValid()
       && !KIO::NetAccess::exists(pictureFileURL, false, kapp->mainWidget()))
    if(pictureFileURL.isLocalFile())
      view->pixmap().save(pictureFileURL.path(-1), preferences.picture.format);
    else
    {
      KTempFile fileTemp;
      fileTemp.setAutoDelete(true);
      
      view->pixmap().save(fileTemp.name(), preferences.picture.format);
      KIO::NetAccess::upload(fileTemp.name(), pictureFileURL, kapp->mainWidget());
    }
  
  const QString modelFileName = name + "." + preferences.model.format.lower();
  KURL modelFileURL(preferences.url, modelFileName);
  
  if(preferences.model.filter & KBSPredictorMoleculeLogPreferences::Workunit
       && modelFileURL.isValid()
       && !KIO::NetAccess::exists(modelFileURL, false, kapp->mainWidget()))
    if(modelFileURL.isLocalFile())
      if("WRL" == preferences.model.format)
        view->exportVRML(modelFileURL.path(-1));
      else
        view->exportX3D(modelFileURL.path(-1));
    else
    {
      KTempFile fileTemp;
      fileTemp.setAutoDelete(true);
      
      if("WRL" == preferences.model.format)
        view->exportVRML(fileTemp.name());
      else
        view->exportX3D(fileTemp.name());
      KIO::NetAccess::upload(fileTemp.name(), modelFileURL, kapp->mainWidget());
    } 
  
  delete view; 
}

void KBSPredictorMoleculeLog::logResult(const QString &name, const PredictorResult *data)
{
  if(!QGLFormat::hasOpenGL()) return;
  
  PredictorAppType app_type = data->app_type;
  KBSPredictorMoleculeLogPreferences preferences = this->preferences(app_type);
  
  if(!preferences.url.isValid()) return;
  
  KBSPredictorMoleculeView *view = new KBSPredictorMoleculeView();
  if(app_type == MFOLD) {
    view->setChain(data->mfold.monsster.final.chain);
    view->setSeq(data->mfold.monsster.seq);
  } else
    view->setPDB(data->charmm.protein.final_pdb);
  view->setStyle(preferences.style);
  view->setColoring(preferences.coloring);
  view->resize(640, 480);
  
  const QString pictureFileName = name + "." + preferences.picture.format.lower();
  KURL pictureFileURL(preferences.url, pictureFileName);
  
  if(preferences.picture.filter & KBSPredictorMoleculeLogPreferences::Result
       && pictureFileURL.isValid()
       && !KIO::NetAccess::exists(pictureFileURL, false, kapp->mainWidget()))
    if(pictureFileURL.isLocalFile())
      view->grabFrameBuffer().save(pictureFileURL.path(-1), preferences.picture.format);
    else
    {
      KTempFile fileTemp;
      fileTemp.setAutoDelete(true);
      
      view->grabFrameBuffer().save(fileTemp.name(), preferences.picture.format);
      KIO::NetAccess::upload(fileTemp.name(), pictureFileURL, kapp->mainWidget());
    }
  
  const QString modelFileName = name + "." + preferences.model.format.lower();
  KURL modelFileURL(preferences.url, modelFileName);
  
  if(preferences.model.filter & KBSPredictorMoleculeLogPreferences::Result
       && modelFileURL.isValid()
       && !KIO::NetAccess::exists(modelFileURL, false, kapp->mainWidget()))
    if(modelFileURL.isLocalFile())
      if("WRL" == preferences.model.format)
        view->exportVRML(modelFileURL.path(-1));
      else
        view->exportX3D(modelFileURL.path(-1));
    else
    {
      KTempFile fileTemp;
      fileTemp.setAutoDelete(true);
    
      if("WRL" == preferences.model.format)
        view->exportVRML(fileTemp.name());
      else
        view->exportX3D(fileTemp.name());
      KIO::NetAccess::upload(fileTemp.name(), modelFileURL, kapp->mainWidget());
    }  
    
    delete view; 
}

#include "kbspredictormoleculelog.moc"
