/***************************************************************************

   Copyright (C) 2005-2007 by Christian Weilbach <christian_weilbach@web.de>
   Copyright (C) 2007-2008 Antonio Aloisio <gnuton@gnuton.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the
   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
 ***************************************************************************/

#include "media/uploadmediadialog.h"

#include <QFileInfo>
#include <QLabel>
#include <QLayout>
#include <QImage>
#include <QBuffer>


#include <kfiledialog.h>
#include <klineedit.h>
#include <kpushbutton.h>
#include <kmessagebox.h>
#include <klocale.h>
#include <kdebug.h>
#include <kurlrequester.h>
#include <kvbox.h>
#include <kio/job.h>

#include "kbloggerconfig.h"
//#include "media/imagemanager.h"
#include "backend/backend.h"
#include "media/media.h"
#include "composer/blogchooser.h"
#include "itemsmanager.h"
#include "composer/composereditor.h"

const int MAX_FILE_SIZE_WARNING = 1024; // 1Mb

namespace KBlogger
{
/** UploadMediaDialog Class **/
UploadMediaDialog::UploadMediaDialog( const QString& filename,
                                      const QString& blogname,
                                      QWidget *parent):KDialog(parent){
    kDebug();
    //prepare the composer for media insertion
    ComposerEditor *composer = qobject_cast<ComposerEditor*> (parent);
    Q_ASSERT(composer);

    mUploadMediaWidget= new UploadMediaWidget(filename, blogname, composer, this);
    
    setCaption( "Add Media" );
    setButtons( KDialog::Ok | KDialog::Cancel );
    setMainWidget( mUploadMediaWidget );

    connect ( this, SIGNAL (  okClicked() ),
              mUploadMediaWidget, SLOT( slotEnqueue() ));
    connect ( mUploadMediaWidget, SIGNAL( enableButtonOK(bool) ),
              this, SLOT ( slotEnableButtonOk(bool) ) );

    enableButtonOk( false );
    show();
}

UploadMediaDialog::~UploadMediaDialog(){
    kDebug();
    delete mUploadMediaWidget;
}

void UploadMediaDialog::slotEnableButtonOk(bool state){
    kDebug();
    enableButtonOk(state);
}

/** UploadMediaWidget Class **/
UploadMediaWidget::UploadMediaWidget( const QString& filename,
                                      const QString& blogname,
                                      ComposerEditor *composer,
                                      QWidget *parent)
        : QWidget(parent),
        mMedia(0),
        mComposer(composer),
        mBlogname(blogname),
        fileJob(0),
        mBlogChooser(0)
        //mImageManager(0)
{
    kDebug();
    setupUi(this);
//     setAttribute( Qt::WA_DeleteOnClose ); //Destroy on close

    //Add BlogServer Chooser
    if (blogname.isEmpty()) {
        mBlogChooser = new BlogChooser(blogGroupBox);
        blogGroupBox->layout()->addWidget(mBlogChooser);
    } else {
        /*QLabel *blognameLabel = new QLabel(mBlogname, blogGroupBox);
        blogGroupBox->layout()->addWidget(blognameLabel);*/
        blogGroupBox->setHidden(true);
    }

    connect ( FilekURLRequester, SIGNAL(  textChanged(const QString&)),
              this, SLOT( updateServerSideName(const QString&)) );
    connect ( FilekURLRequester, SIGNAL(  textChanged(const QString&)),
              this, SLOT( updateFilesize(const QString&)) );
//     connect ( this, SIGNAL (  okClicked() ),
//               this, SLOT( slotEnqueue() ));

    // initialize if called with filename
    if ( !filename.isEmpty() ) {
        FilekURLRequester->setUrl( filename );
    }
    
}

UploadMediaWidget::~UploadMediaWidget()
{
    kDebug();
}

void UploadMediaWidget::slotEnqueue()
{
    kDebug();
    if ( mBlogChooser ) {
        if ( mBlogChooser->blogCombo->currentIndex() == 0 ) {
            KMessageBox::sorry ( this, i18n ( "Please select a blog profile first."));
            return;
        }
    }

    emit enableButtonOK(false);

    fileJob = KIO::storedGet( FilekURLRequester->url() );

    connect( fileJob, SIGNAL( result( KJob* ) ),
             this, SLOT( handleFileJobResult( KJob* ) ) );
}

void UploadMediaWidget::handleFileJobResult( KJob* job )
{
    kDebug();
    if ( job->error() ) {
        KMessageBox::error ( this,  job->errorString() );
        return;
    }

    //Get required data: Media data, MimeType, blogName
    QByteArray mediaData( qobject_cast<KIO::StoredTransferJob*>(job)->data() );
    QString mediaMimetype( qobject_cast<KIO::TransferJob*>(job)->mimetype() );
    if ( mBlogname.isEmpty() ){
        mBlogname = mBlogChooser->blogCombo->currentText();
    }

    //TODO check for mediatypes, since most blogs only support image + a view else
    
    //Create Media
    //NOTE: Data will be saved in the kblogger cache automatically.
     if ( mediaMimetype.indexOf( "image" ) != -1 ){
         QString msg = i18n( "KBlogger is unable to create Thumbnails. This feature will be added soon." );
         KMessageBox::sorry ( this, msg );
     }
     mMedia=new Media ( mediaData, 
                        NameLineEdit->text(), //FileName
                        mediaMimetype,
                        mBlogname ); 
     ItemsManager::self()->addMedia(mMedia);
     mComposer->insertMedia(mMedia);
     

    /*
    // create a thumbnail for images
    // TODO check the mimetype of images better
    if( mediaMimetype.indexOf( "image" ) != -1 ){
        kDebug() << "do QImage resizing";
        QImage imageToResize( mediaData.data() );
        if( imageToResize.isNull() ) kDebug() << "could not initialize the image"; 
        imageToResize = imageToResize.scaledToWidth( 240 ); // TODO add a dialog
        QByteArray thumbnailData;
        QBuffer buffer(&thumbnailData);
        buffer.open(QIODevice::WriteOnly);
        imageToResize.save( &buffer, "PNG" );
        mThumbnail = new Media( thumbnailData,
                                "thumbnail_"+NameLineEdit->text(),
                                mediaMimetype,
                                mBlogname
                              );

        mMedia = new Media( mediaData,
                                NameLineEdit->text(),
                                mediaMimetype,
                                mBlogname
                              );

        if ( mMedia->isCached() && mThumbnail->isCached() ) {
            ItemsManager::self()->addMedia(mMedia);
            ItemsManager::self()->addMedia(mThumbnail);
            if(composer) composer->insertThumbnail(mThumbnail,mMedia);
        }
        else {
            QString msg = i18n("Impossible to copy the file %1 in the cache",
                               FilekURLRequester->url().url() );
            KMessageBox::sorry ( this, msg );
        }
        close();
        return;
    }

     // upload the original file otherwise
    mMedia = new Media( mediaData,
                              NameLineEdit->text(),
                              mediaMimetype,
                              mBlogname
                         );
*/  
//     if ( mMedia->isCached() ) {
//          ItemsManager::self()->addMedia(mMedia);
//          if(composer) composer->insertMedia(mMedia);
//     }
//     else {
//         QString msg = i18n("Impossible to copy the file %1 in the cache",
//                            FilekURLRequester->url().url() );
//         KMessageBox::sorry ( this, msg );
//     }
    close();
}

void UploadMediaWidget::updateServerSideName(const QString& filename)
{
    kDebug();
    QFileInfo fileToUpload(filename);
    NameLineEdit->setText(fileToUpload.fileName());
}

void UploadMediaWidget::updateFilesize(const QString& filename)
{
    kDebug();
    double filesize;
    QFileInfo fileToUpload(filename);
    filesize = fileToUpload.size();
    if ( !filesize ) {
        filesizeLabel->setText(i18n("<font color='red'>This file is not valid</font>"));
    } else {
        filesize = filesize / 1024; //Convert byte in KByte
        if ( filesize > MAX_FILE_SIZE_WARNING ) {
            filesizeLabel->setText(i18n ("File size: ") + "<font color='red'>"+QString::number(filesize) +"</font>"+ i18n(" Kb"));
            KMessageBox::sorry ( this, i18n ( "This file is too big to be uploaded." ) );
            KDialog *mParentDialog= qobject_cast<KDialog*>(parent());
            mParentDialog->close();
        } else {
            filesizeLabel->setText(i18n ("File size: ") + "<font color='green'>" + QString::number(filesize) + "</font>" + i18n(" Kb"));
        }
        /*if (mImageManager)
            mImageManager->setImage(filename);*/
    }
    enableButtonOK(true);
}

} //namespace

#include "uploadmediadialog.moc"
