/***************************************************************************

   Copyright (C) 2007-2008 Antonio Aloisio <gnuton@gnuton.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the
   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
 ***************************************************************************/
#include "mainwidget.h"

#include <QStackedWidget>

#include <kstandarddirs.h>
#include <kiconloader.h>
#include <kactioncollection.h>
#include <kmessagebox.h>
#include <ktextbrowser.h>

#include "profileconfig.h"
#include "backend/backend.h"
#include "itemlist.h"
#include "post/postslist.h"
#include "media/medialistwidget.h"
#include "mainwindow.h"
#include "post/post.h"
#include "media/media.h"
#include "itemsmanager.h"
#include "backend/blogserver.h"
#include "composer/visualeditor.h"

#define HTML_CODE "<html><body bgcolor=white >"
namespace KBlogger
{

MainWidget *MainWidget::s_self = 0;

MainWidget *MainWidget::self(KActionCollection *actionCollection, QWidget* parent )
{
    if ( !s_self ) {
        s_self = new MainWidget(actionCollection, parent);
        s_self->initialize();
    }
    return s_self;
}

MainWidget::MainWidget( KActionCollection *actionCollection, QWidget* parent ):
        QWidget(parent),
        mMainWindow( qobject_cast<MainWindow*> ( parent )),
        mActionCollection(actionCollection)
{
    kDebug();
    setupUi(this);
    setAttribute(Qt::WA_DeleteOnClose); //Destroy on close
}

void MainWidget::initialize()
{
    kDebug();
    //Create StackedWidget
    stackedWidget = new QStackedWidget(PostsListWidget);
    PostsListWidget->layout()->addWidget(stackedWidget);

    //Create PreviewBrowser Widget
    previewBrowser = new VisualEditor(previewWidget);
    previewBrowser->setReadOnly(true);
    previewWidget->layout()->addWidget(previewBrowser);

    populateAccountList();

    connect( blogListTree, SIGNAL( itemClicked( QTreeWidgetItem *, int) ),
             this, SLOT(changeShownEntriesList(QTreeWidgetItem *)));
    connect( stackedWidget, SIGNAL( currentChanged(int) ), SLOT(updateKActions(int)));

    readConfig();
}

MainWidget::~MainWidget()
{
    kDebug();
    saveStatus();
    s_self = 0;
}

void MainWidget::viewEntry(Post* post)
{
    kDebug();
    if (!post) return;
    QString text;
    text += HTML_CODE;
    text += "<div><H3><FONT COLOR=#a0b0ff>" + post->title() + "</font></H3></div>";
    text += "<div><i>&nbsp;" + post->categories().join(", ") + "</i></div>" ;
//  text+=i18n("<i>Creation Date<i>")+
    text += "<div>&nbsp;" + post->content() + "</div>";
    text += "</html>";
    previewBrowser->setText(text);
}

void MainWidget::viewEntry(Media *media)
{
    kDebug();
    if (!media) return;
    QString text;
    text = HTML_CODE;
    text += "<div><b>Name:</b> " + media->name() + "</div>";
    text += "<div><b>Url:</b> " + media->url().url() + "</div>" ;
    text += "<div align='center'><img src='" + media->cachedFilename() + "' /></div>";
    text += "</html>";

    previewBrowser->setText(text);
}

void MainWidget::openSelectedMedia()
{
    MediaList *mediaWidget;
    mediaWidget = qobject_cast<MediaList*>( stackedWidget->currentWidget() );
    if ( !mediaWidget) return;
    mediaWidget->openSelectedMedia();
}

KActionCollection* MainWidget::actionCollection() const
{
    return mMainWindow->actionCollection();
}

void MainWidget::reloadSettings()
{
    kDebug();

    //Scan Profiles file and fill blogList
    Backend::self()->populateBlogsList();
    //Creates QTreeWidgetItem with the new blogList
    populateAccountList();
    //Save Posts, Media and kblogger configuration
    saveStatus();
}

void MainWidget::saveStatus()
{
    kDebug();
    PostsList *postsWidget;
    MediaList *mediaWidget;

    for (int i = stackedWidget->count() - 1; i > 0; --i) {
        if ( ( postsWidget = qobject_cast<PostsList*> ( stackedWidget->widget(i) ) ) )
            postsWidget->save();
        else
            if ( (mediaWidget = qobject_cast<MediaList*> ( stackedWidget->widget(i) )) )
                mediaWidget->save();

    }
    saveConfig();
}

void MainWidget::updateAllSentPostList()
{
    kDebug();
    QString blogname = currentBlogname();
    if ( blogname.isEmpty() ) return;
    ItemsManager::self()->updatePostsSentList( blogname );
}

void MainWidget::trashPost()
{
    kDebug();
    PostsList *postsWidget;
    postsWidget = qobject_cast<PostsList*>(stackedWidget->currentWidget() );
    if ( !postsWidget) {
        //KMessageBox::sorry( this, i18n("No Post selected."));
        return;
    }
    postsWidget->trashPost();

}

void MainWidget::deletePost()
{
    kDebug();
    PostsList *postsWidget;
    postsWidget = qobject_cast<PostsList*>(stackedWidget->currentWidget() );
    if ( !postsWidget) {
        //KMessageBox::sorry( this, i18n("No Post selected."));
        return;
    }
    postsWidget->deletePost();
}

QString MainWidget::currentBlogname()
{
    kDebug();
    ItemList *itemWidget;
    itemWidget = qobject_cast<ItemList*>( stackedWidget->currentWidget() );
    if ( itemWidget)
        return itemWidget->blogname();
    else
        return QString();
}

void MainWidget::modifyPost()
{
    kDebug();
    PostsList *postsWidget;
    postsWidget = qobject_cast<PostsList*>(stackedWidget->currentWidget() );
    if ( !postsWidget) {
        KMessageBox::sorry( this, i18n("No Post selected."));
        return;
    }
    postsWidget->modifyPost();
}

void MainWidget::removeLocalMedia()
{
    kDebug();
    MediaList *mediaWidget;
    mediaWidget = qobject_cast<MediaList*>( stackedWidget->currentWidget() );
    if ( !mediaWidget) {
        KMessageBox::sorry( this, i18n("No Media selected."));
        return;
    }
    mediaWidget->removeLocalItem();
}

void MainWidget::uploadCurrentMedia()
{
    kDebug();
    QString blogname = currentBlogname();
    if ( blogname.isEmpty() ) return;
    ItemsManager::self()->enqueueMediaToUpload( blogname );
}

void MainWidget::uploadCurrentPosts()
{
    kDebug();
    QString blogname = currentBlogname();
    if ( blogname.isEmpty() ) return;
    ItemsManager::self()->enqueuePostsToUpload( blogname );
}

void MainWidget::populateAccountList()
{
    kDebug();

    //A.Declare
    int index = 0;

    //B. Clean Map,QWidgetTree and QStackWidget
    mQTreeWidgetItemMap.clear(); //FIXME MEMORY LEAK IN TREEINDEX!!
    blogListTree->clear();
    //prevent a memory leak!
    for (int i = stackedWidget->count(); i >= 0; --i)
        delete stackedWidget->widget(i);

    //C. Add Welcome Screen
    KTextBrowser* welcomeBrowser;
    welcomeBrowser = new KTextBrowser(this);
    stackedWidget->addWidget(welcomeBrowser);
    QTreeWidgetItem *root;
    root = new QTreeWidgetItem(blogListTree);
    root->setText( 0, i18n("Blogs") );
    root->setIcon( 0, KIcon("kblogger"));
    root->setExpanded (true);
    TreeIndex welcomeIndex(index, static_cast<QWidget*> (welcomeBrowser));
    mQTreeWidgetItemMap.insert( root, welcomeIndex );
    populateWelcomeTextBrowser(welcomeBrowser);
    blogListTree->insertTopLevelItem( 0, root );

    //D. Populate
    QList<BlogServer> blogList;
    blogList = Backend::self()->blogServers();
    for ( int i = 0; i < blogList.size(); ++i) {
        QTreeWidgetItem *item;
        BlogServer blog;
        QString accountName;
        QStringList childrenList;
        QList<QTreeWidgetItem *> children;
        QList<KIcon> iconList;
        QList<QWidget*> widgetList;

        blog = blogList.at(i);

        //Add Parent Item
        item = new QTreeWidgetItem(root);
        item->setText( 0, blog.blogname() );
        item->setIcon( 0, blog.icon() );
        //blogListTree->insertTopLevelItem( 0, item );

        //Create & Add PostsList in the StackWidget.
        accountName = item->text(0);
        SentPostsList *mSentEntries;
        LocalDraftPostsList *mLocalEntries;
        TrashedPostsList *mTrashEntries;
        MediaList *mMediaEntries;

        mSentEntries = new SentPostsList(accountName, mMainWindow, this);
        mLocalEntries = new LocalDraftPostsList(accountName, mMainWindow, this);
        mTrashEntries = new TrashedPostsList(accountName, mMainWindow, this);
        mMediaEntries = new MediaList(accountName, mMainWindow, this);

        TreeIndex sentIndex(++index,
                              static_cast<QWidget*> (mSentEntries),
                              accountName
                             );
        mQTreeWidgetItemMap.insert( item, sentIndex );

        stackedWidget->addWidget(mSentEntries);
        stackedWidget->addWidget(mLocalEntries);
        stackedWidget->addWidget(mMediaEntries);
        stackedWidget->addWidget(mTrashEntries);

        //Add Children Items
        childrenList << i18n( "Local entries" )
        << i18n( "Media" )
        << i18n( "Trash" );

        iconList << KIcon( "mail-message-new" )
        << KIcon("image-x-generic")
        << KIcon("user-trash");

        widgetList << static_cast<QWidget*> ( mLocalEntries )
        << static_cast<QWidget*> ( mMediaEntries )
        << static_cast<QWidget*> ( mTrashEntries );

        for (int j = 0; j < childrenList.size() ; ++j) {
            QTreeWidgetItem* childItem;
            childItem = new QTreeWidgetItem( (QTreeWidget*)0,
                                                QStringList(childrenList.at(j)) );
            TreeIndex childIndex(++index,
                                 static_cast<QWidget*> (widgetList.at(j)),
                                 childrenList.at(j)
                                );
            mQTreeWidgetItemMap.insert(childItem, childIndex);
            children.append(childItem);
            children.at(j)->setIcon(0, iconList.at(j));
        }
        item->addChildren(children);
    }
   updateItemNumbers();
}

void MainWidget::changeShownEntriesList(QTreeWidgetItem *currentItem)
{
    kDebug();
    TreeIndex treeIndexFromMap( mQTreeWidgetItemMap.value(currentItem) );
    int row = treeIndexFromMap.pos();
    kDebug() << "MainWidget::changeShownEntriesList() selected row:" << row << endl;
    stackedWidget->setCurrentIndex(row);
}

void MainWidget::updateKActions(int i)
{
    kDebug();

    QString enabledActionGroup;

    switch (i) {
    case - 1: {
        mMainWindow->updateSearchLine();
        return;
    }
    break;

    case 0: { //Welcome Screen
        mMainWindow->updateSearchLine();
    }
    break;

    default: {
        //Sent, Trash & Modify entries Lists
        PostsList *entriesWidget;
        entriesWidget = qobject_cast<PostsList*>( stackedWidget->widget(i) );
        if ( entriesWidget ) {
            enabledActionGroup = "postActionGroup";
            entriesWidget->updateSearchLine();
        }
        //Media List
        MediaList *mediaWidget;
        mediaWidget = qobject_cast<MediaList*>( stackedWidget->widget(i) );
        if ( mediaWidget ) {
            enabledActionGroup = "mediaActionGroup";
            mediaWidget->updateSearchLine();
        }
    }
    }
    QList<QActionGroup *> actionGroupList;
    actionGroupList = mMainWindow->actionCollection()->actionGroups();

    for (int j = 0; j < actionGroupList.size(); ++j) {
        bool enable;

        QActionGroup *actionGroup;
        actionGroup = actionGroupList.at(j);

        if ( !actionGroup ) continue;

        if ( i == 0 ) //Welcome Screen - All KActions are disabled.
            enable = false;
        else if ( actionGroup->objectName() == enabledActionGroup )
            enable = true;
        else if ( actionGroup->objectName() == "kblogActionGroup" )
            enable = true;
        else
            enable = false;

        kDebug() << "groupName=" << actionGroup->objectName() << "Enabled=" << enable;

        actionGroup->setEnabled(enable);
    }
}

void MainWidget::populateWelcomeTextBrowser(QTextBrowser* textBrowser)
{
    kDebug();

    QString html = "<div align=center><img src='";
    html += KStandardDirs::locate("appdata", "pics/kbloggerWelcome.jpg");
    html += "' /></div>";

    textBrowser->setHtml(html);
}

void MainWidget::saveConfig()
{
    kDebug();
    KSharedConfigPtr config = KSharedConfig::openConfig();
    KConfigGroup qsplitterGroup( config, "MainWindow QSplittersSizes" );
    qsplitterGroup.writeEntry("mainHSplitterSizes", mainHSplitter->sizes() );
    qsplitterGroup.writeEntry("mainVSplitterSizes", mainVSplitter->sizes() );
    qsplitterGroup.config()->sync();

}

void MainWidget::readConfig()
{
    kDebug();
    KSharedConfigPtr config = KSharedConfig::openConfig();
    QList<int> splitterHsizes, splitterVsizes;
    KConfigGroup qsplitterGroup( config, "MainWindow QSplittersSizes" );
    splitterHsizes = qsplitterGroup.readEntry("mainHSplitterSizes", splitterHsizes );
    splitterVsizes = qsplitterGroup.readEntry("mainVSplitterSizes", splitterVsizes );

    if ( !splitterHsizes.isEmpty() )
        mainHSplitter->setSizes( splitterHsizes );
    if ( !splitterVsizes.isEmpty() )
        mainVSplitter->setSizes( splitterVsizes );
}

void MainWidget::updateItemNumbers()
{
    kDebug();

    QMapIterator<QTreeWidgetItem *, TreeIndex> i(mQTreeWidgetItemMap);
    while (i.hasNext()) {
        i.next();

        PostsList *entriesListWidget = 0;
        MediaList *MediaListWidget = 0;

        QTreeWidgetItem *item = i.key();
        TreeIndex treeIndexFromMap( i.value() );
        int count = -1;

        entriesListWidget = qobject_cast<PostsList*>( treeIndexFromMap.widget() );
        if (entriesListWidget) {
            count = entriesListWidget->count();
        } else {
            MediaListWidget = qobject_cast<MediaList*>( treeIndexFromMap.widget() );
            if (MediaListWidget) {
                count = MediaListWidget->count();
            }
        }
        if (count < 0) continue;
        item->setText(0, QString("%1 (%2)").arg(treeIndexFromMap.text()).arg(count));
    }
}

} //namespace

#include "mainwidget.moc"
