/***************************************************************************

   Copyright (C) 2005-2007 by Christian Weilbach <christian_weilbach@web.de>
   Copyright (C) 2007-2008 Antonio Aloisio <gnuton@gnuton.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the
   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
 ***************************************************************************/
#include "composer/composer.h"

#include <QCheckBox>
#include <QRegExp>
#include <QLabel>
#include <QLayout>
#include <QFileInfo>
#include <QValidator>
#include <Q3CString>
#include <QPixmap>
#include <QListWidget>
#include <QFile>

#include <klineedit.h>
#include <klocale.h>
#include <ktextedit.h>
#include <kcombobox.h>
#include <kpushbutton.h>
#include <kdatetimewidget.h>
#include <kdebug.h>
#include <kmessagebox.h>
#include <kstandarddirs.h>

#include "kbloggerconfig.h"
#include "backend/backend.h"
//#include "backendmanager.h" //FIXME Post
#include "post/post.h"
#include "post/postslist.h"
#include "composer/composereditor.h"
#include "itemsmanager.h"
#include "backend/blogserver.h"
#include "composer/blogchooser.h"
#include "composer/visualeditor.h"

#include "backend/bloglist.h"

#define MAGIC_KEY 0xA1B2C3D4
#define VERSION 001
#define DIR_TO_SAVE_CATEGORIES "categories/"

namespace KBlogger
{

Composer::Composer ( Post* post, QWidget* parent):
        QDialog(parent)
{
    kDebug();
    mPostToModify = post;
    initialize();
    loadPostSlot (post);
    connect ( buttonBox, SIGNAL( accepted() ), this, SLOT( enqueueSlot() ) );
    connect ( buttonBox, SIGNAL( rejected() ), this, SLOT( close() ) );
}

Composer::Composer (const QString& blogname, QWidget* parent):
        QDialog(parent)
{
    kDebug();
 
    // TODO make blog actions transparent to blog choser
    if (blogname.isEmpty()){
            KMessageBox::error(this,i18n("Please select a blog first."),
                                    i18n("Cannot create post without a blog."));
            return;
    }
    mPostToModify = 0;
    initialize();
    mBlogname = blogname;
    setWindowTitle( i18n("Blog post for blog: %1", blogname ) );
    timeStampKDateTime->setDateTime ( QDateTime::currentDateTime() );
    connect ( buttonBox, SIGNAL( accepted() ), this, SLOT( enqueueSlot() ) );
    connect ( buttonBox, SIGNAL( rejected() ), this, SLOT( close() ) );
}


Composer::~Composer()
{
    kDebug();
    //TODO Stop 'Fetch Categories' if it's running!
}

QString Composer::blogname()
{
    kDebug();
    return mBlogname;
}

void Composer::initialize ()
{
    kDebug();
    setupUi( this );
    mBackend = Backend::self();
    mItemsManager = ItemsManager::self();

    setAttribute( Qt::WA_DeleteOnClose ); //Destroy on close

    //Add Editor: NOTE Editor requires BlogChooser!
    mEditor = new ComposerEditor(this, editorFrame);
    editorFrame->layout()->addWidget(mEditor);

    //Connections
    connect ( refreshCategoriesButton, SIGNAL( clicked() ),
              this, SLOT( slotRefreshCateroriesFromServer() ) );
    connect( CategoryComboBox, SIGNAL( currentIndexChanged(const QString& ) ),
             this , SLOT( hideFirstCategory( const QString&)));

}

void Composer::loadPostSlot ( Post* post )
{
    kDebug();
    Q_ASSERT(post);

    //First of All we need to set the blogname,
    //In this way categories are loaded and some options
    //are enabled by updateBlogOptions();
    mBlogname = post->getBlogName();
    setWindowTitle( i18n("Blog post for blog: %1", mBlogname ) );
 
    //Set Title
    TitleLineEdit->setText ( post->title() );

    //Set Categories
    QStringList categories=post->categories();

    //Set First Category
    if(CategoryComboBox->count()){
        int index;
        QString category;
        category=categories.takeFirst();
        index=CategoryComboBox->findText(category);

        if (index != -1) //Category found
            CategoryComboBox->setCurrentIndex(index);
    }

    //Set Other Categories
    checkOtherCategories(categories);
    
    //Set Content
    mEditor->mVisualTextEditor->setHtml (
        mEditor->htmlToRichtext(post->content() ));
    
    //Set other post options
    timeStampKDateTime->setDateTime ( post->creationDateTime().dateTime () );
    commentCheckBox->setChecked( post->isCommentAllowed() );
    trackbackCheckBox->setChecked( post->isTrackBackAllowed());
}

bool Composer::loadCategories(const QString& blogname)
{
    kDebug();

    //blogname test
    if (blogname.isEmpty()) {
        kError() << "blogname is empty" << endl;
        return false;
    }

    //Setting up filename
    QString filename;
    filename = DIR_TO_SAVE_CATEGORIES + blogname;
    if (filename.isEmpty()) {
        kDebug() << "\tfilename is empty" << endl;
        return false;
    }
    QString fileToOpen = KStandardDirs::locateLocal("appdata", filename , true);
    kDebug() << "opening file: " << fileToOpen << endl;
    if (fileToOpen.isEmpty()) {
        return false;
    }
    QFile file( fileToOpen );
    file.open(QIODevice::ReadOnly);
    QDataStream in(&file);

    // Read and check the header
    quint32 magic;
    in >> magic;
    if (magic != MAGIC_KEY) {
        kDebug() << "BAD MAGIC KEY, Trying to retrieve categories from " << blogname << endl;
        return false;
    }

    // Read the version
    qint32 version;
    in >> version;
    if (version != VERSION) {
        kDebug() << "BAD VERSION" << endl;
        getCategories(blogname);
        return false;
    }

    //Read the number of categories
    qint32 categoriesNumber = -1;
    in >> categoriesNumber;

    if ( categoriesNumber == -1 ) {
        //file is empty
        file.close();
        return false;
    } else {
        //Read data
        QString name, description, htmlUrl, rssUrl;
        QListWidgetItem *categoriesListItem;
        for (int i = 0; i < categoriesNumber; ++i) {
            in >> name;
            in >> description;
            in >> htmlUrl;
            in >> rssUrl;
            categoriesListItem = new QListWidgetItem(name, categoriesList);
            categoriesListItem->setCheckState(Qt::Unchecked);

            CategoryComboBox->insertItem (0, name );
        }

        mBackend->slotStatusBarMessage(i18n("Loaded %1 categories...", categoriesNumber));
        file.close();
    }
    return true;
}

void Composer::getCategories(const QString& blogname)
{
    kDebug();
    mBackend->slotStatusBarMessage(i18n("Fetching List of Categories..."));
    connect (mBackend,
             SIGNAL(listedCategories( const QList<QMap<QString, QString> >&)),
             this,
             SLOT(slotCategoryInfoRetrieved( const QList<QMap<QString, QString> >&)));
    mBackend->listCategories(blogname, this);
}

void Composer::slotRefreshCateroriesFromServer()
{
    kDebug();
    getCategories(mBlogname);
}

void Composer::hideFirstCategory( const QString& categoryToHide )
{
    kDebug();
    QListWidgetItem *category;
    for (int row = 0; row < categoriesList->count(); ++row) {
        category = categoriesList->item(row);
        if ( categoryToHide == category->text() ) {
            category->setHidden(true);
            category->setCheckState(Qt::Unchecked);
        } else {
            category->setHidden(false);
        }
    }
}

void Composer::checkOtherCategories( QStringList& categories )
{
    //We assume that the first category is not in the gategories stringList.
    kDebug();
    QListWidgetItem *categoryItem;
    for (int row=0; row < categoriesList->count(); ++row) {
        categoryItem = categoriesList->item(row);
        QString category;
        for ( int i=0; i < categories.count(); i++){
            category=categories.at(i);
            if ( categoryItem->text() == category ){
                categoryItem->setCheckState(Qt::Checked);
                categories.removeAt(i);
                i--;
            }
        }
    }
}

void Composer::saveCategories(const QString& blogname, const QList< QMap<QString, QString> >* categories)
{
    kDebug();

    //blogname test
    if (blogname.isEmpty()) {
        kError() << "blogname is empty" << endl;
        return;
    }

    //Setting up filename
    QString filename;
    filename = DIR_TO_SAVE_CATEGORIES + blogname;
    if (filename.isEmpty()) {
        kDebug() << "\tfilename is empty" << endl;
        return;
    }
    QString fileToSave = KStandardDirs::locateLocal("appdata", filename , true);
    kDebug() << "\tsaving file: " << fileToSave << endl;

    //
    //TODO qApp->processEvents(QEventLoop::ExcludeUserInputEvents);

    //Open file to write and setting up Datastream
    QFile file( fileToSave );
    file.open(QIODevice::WriteOnly);
    QDataStream out(&file);   // we will serialize the data into the file

    // Write the header
    out << (quint32)MAGIC_KEY; //"magic number"
    out << (qint32)VERSION; //binary file version

    out.setVersion(QDataStream::Qt_4_0);

    //write the number of categories that will be stored;
    out << categories->count();

    //write categories data;
    const QMap<QString, QString> *category;
    for ( int i = 0; i < categories->count(); ++i) {
        category = &(categories->at(i));
        if (!category) {
            kError() << "category is NULL" << endl;
            break;
        }
        out << category->value("name");
        out << category->value("description"); //TODO UNUSED
        out << category->value("htmlUrl"); //TODO UNUSED
        out <<  category->value("rssUrl"); //TODO UNUSED
    }
    file.close();
}

void Composer::enqueueSlot()
{
    kDebug();

    //TEST - No Post Content // TODO make that a public function of ComposerEditor
    if ( mEditor->mVisualTextEditor->toPlainText().isEmpty() ) {
        KMessageBox::information(this, i18n("Your post is empty."));
        return;
    }

    //TEST - Post Title is Empty
    if ( !TitleLineEdit->isHidden() && TitleLineEdit->text().isEmpty() ) {
        KMessageBox::sorry ( this, i18n ( "Please fill the title field." ) );
        return;
    }

    //Give a title to untitled posts for the list view //TODO right?
    if (TitleLineEdit->isHidden()) {
        QString title = i18n("Untitled post of %1",
                             KDateTime::currentUtcDateTime().toString()) ;
        TitleLineEdit->setText(title);
    }


    mBackend->slotStatusBarMessage(i18n("Post enqueued in the Local Draft."));

    //Post content contains HTML code of HtmlEditor.
    mEditor->syncEditors(1); // 1 = HtmlEditor

    QStringList categories;
    categories << CategoryComboBox->currentText();
    for ( int i = 0; i < categoriesList->count(); ++i) {
        QListWidgetItem * mPost = categoriesList->item(i);
        if (mPost->checkState() == Qt::Checked)
            categories <<  mPost->text();
    }

    //Settings up & create post
    Post *post;
    post = new Post( mBlogname,
                             TitleLineEdit->text(),
                             mEditor->contentTextHtmlEditor->toPlainText(),
                             categories,
                             !publishCheckBox->isChecked(),
                             static_cast<KDateTime>(timeStampKDateTime->dateTime()) );
    if (mPostToModify) {
        post->setPostId( mPostToModify->postId() );
    }

    post->setCommentAllowed(  commentCheckBox->isChecked() );
    post->setTrackBackAllowed( trackbackCheckBox->isChecked() );
    post->setStatus(KBlog::BlogPost::New);

    //mEditor->ContentTextHtmlEditor->clear();
    //mEditor->visualTextEditor->clear();
    //TitleLineEdit->clear();

    if ( mPostToModify ) {
        //Modify Post
        PostsList *containerWidget = 0;
        containerWidget = qobject_cast<PostsList*> (mPostToModify->container());
        if ( containerWidget ) {
            containerWidget->removePost(mPostToModify);
        }
    }
    mItemsManager->createNewPost(post);
    done(0);
}

void Composer::slotCategoryInfoRetrieved( const QList< QMap<QString, QString> >& categories)
{
    kDebug();

    mBackend->disconnect (
        SIGNAL(categoryInfoRetrieved( const QList<QMap<QString, QString> >&))
    );

    fillCategoriesWidgets(&categories);

    mBackend->slotStatusBarMessage(i18n("Retrieved %1 categories from your blog...", categories.count()));

    //Saving categories;
    saveCategories(mBlogname, &categories);
}

void Composer::fillCategoriesWidgets(const QList< QMap<QString, QString> >* categories)
{
    kDebug();
    QString name, description, htmlUrl, rssUrl;
    const QMap<QString, QString> *category;
    QListWidgetItem *categoriesListItem;
    for ( int i = 0; i < categories->count(); ++i) {
        category = &(categories->at(i));

        name = category->value("name");
        description = category->value("description"); //TODO UNUSED
        htmlUrl = category->value("htmlUrl"); //TODO UNUSED
        rssUrl = category->value("rssUrl"); //TODO UNUSED

        categoriesListItem = new QListWidgetItem(name, categoriesList);
        categoriesListItem->setCheckState(Qt::Unchecked);

        CategoryComboBox->insertItem (0, name );

    }
}

void Composer::updateBlogOptions(const QString& blogname)
{
    kDebug();
    if (!mBackend) return; // TODO can/should this ever happen?

    CategoryComboBox->clear();
    categoriesList->clear();

    QList<QHash<QString,QVariant> > blogList = BlogList();
    QHash<QString,QVariant> blogHash = blogList[ mBackend->blogServer(blogname).type() ];
    kDebug() << "BLOGNAME" << blogname
    << "USED TYPE=" << mBackend->blogServer(blogname).type();

    if( blogHash["hasCategorySupport"].toBool() ){
        //Try to load saved categories.
        loadCategories(blogname);
    }

    if( blogHash["hasTitleSupport"].toBool() ){
        TitleLineEdit->show(); 
        TitleLabel->show();
    }
    else {
        TitleLineEdit->hide(); 
        TitleLabel->hide();
    }

    if( blogHash["hasPublishSupport"].toBool() ){
        publishCheckBox->show();
    }
    else {
        publishCheckBox->hide();
    }

    if( blogHash["hasCategorySupport"].toBool() ){
        showCategoryOptions(true);
    }
    else {
        showCategoryOptions(false);
        mBackend->slotStatusBarMessage(i18n("Categories are not supported on this blog."));
    }

    if( blogHash["hasAllowCommentSupport"].toBool() ){
         commentCheckBox->show();
    }
    else {
        commentCheckBox->hide();
    }

    if( blogHash["hasAllowTrackbackSupport"].toBool() ){
        trackbackCheckBox->show();
    }
    else {
        trackbackCheckBox->hide();
    }

    if( blogHash["hasMediaSupport"].toBool() ){
        mEditor->enableMedia(true);
    }
    else {
        mEditor->enableMedia(false);
    }

    if( blogHash["hasDateTimeCreatedSupport"].toBool() ){
        timeStampKDateTime->setEnabled ( true );
    }
    else {
        timeStampKDateTime->setEnabled ( false );
    }
}

void Composer::showCategoryOptions( bool show )
{
    kDebug();
    if( !show ){
        CategoriesLabel->hide();
        CategoryComboBox->hide();
        label_2->hide();
        categoriesList->hide();
        line_3->hide();
        refreshCategoriesButton->hide();
    }
    else {
        CategoriesLabel->show();
        CategoryComboBox->show();
        label_2->show();
        categoriesList->show();
        line_3->show();
        refreshCategoriesButton->show();
    }
}

} //namespace

#include "composer.moc"
