/***************************************************************************

   Copyright (C) 2007-2008 Antonio Aloisio <gnuton@gnuton.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the
   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
 ***************************************************************************/

#ifndef BACKEND_H
#define BACKEND_H

#include <kdatetime.h>

#include "backend/blogjobqueue.h"
#include "backend/blogjob.h"
#include "media/media.h"


namespace KBlogger
{
class MainWindow;
class Post;
class BlogServer;
class BackendJobQueue;
class WaitDialog;

/**
  @brief
  This class uses BackendJobQueue and feeds it BackendJob[s]. It is the way to talk to
  servers for KBlogger. It also handles showing the WaitDialog.
*/

class Backend : public QObject
{
//     friend class BlogJob;

    Q_OBJECT
public:
    //Singleton
    static Backend *self(QObject* parent = 0);

    ~Backend();

    QList<BlogServer> blogServers();

    BlogServer blogServer(const QString& blogname);

    void populateBlogsList();

    void runQueuedJobs();
    ///Blog's functions (jobs)
    //List recent posts on the serve, is run immediatly
    void listPosts(const QString& blogname, int downloadCount = 5, QWidget *caller = 0);
    // List the categories of the blog. @see getCategoriesSlot(), listCategoriesFinished()
    void listCategories(const QString& blogname, QWidget *caller = 0);
    void sendPost( Post* kbPost, QWidget *caller = 0 );
    void fetchPost(Post* kbPost, QWidget *caller = 0);
    //Remove a post from the server, is run immediatly.
    void removePost( Post *kbPost, QWidget *caller = 0 );
    void sendMedia( Media *kbMedia, QWidget *caller = 0 );

public Q_SLOTS:
    // synchronize the local changes with the blogs.
    // (Upload new or modified posts, medias and  delete posts)
    void sync();
    void slotStopQueue();
    void slotStatusBarMessage(const QString&);

private Q_SLOTS:
    void slotListedCategories( const QList< QMap<QString, QString> >& categories );
    void slotListedRecentPosts( const QList<KBlog::BlogPost>& posts );

    void slotJobsDone();
    void slotJobDone(KJob*);
    void slotJobsStopped();
    void slotCancelClicked();
    void slotError(const QString&);

private:
    //Default Costructor
    Backend( QObject* parent = 0);
    void initialize();
    BlogJob* createBlogJob( BlogServer&, BlogJob::BlogAction, QVariant& );

    //Vars
    static Backend *s_self;
    MainWindow *mMainWindow;
    QMap<QString, BlogServer> mBlogsMap;
    BlogJobQueue *mJobsQueue;
    WaitDialog* mWaitDialog;
    QMap<KJob*,QWidget*> mJobToWidget;
    QMap<KJob*,Post*> mMoveDraftToSent;
    QMap<KJob*,Post*> mRemoveFromTrash;
    QMap<KJob*,KBlog::BlogMedia*> mAddMedia;
    QMap<KJob*,QString> mJobToBlogname;

Q_SIGNALS:
    void listedCategories( const QList< QMap<QString, QString> >& categories );
    void statusBarMessage( const QString& message );
    void jobsFinished();
    void errorMessage( const QString& message );
};

}

#endif // _backend_H_
