/***************************************************************************

   Copyright (C) 2007 Antonio Aloisio <gnuton@gnuton.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the
   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
 ***************************************************************************/

#include "postslist.h"

#include <QTableWidget>
#include <QFile>
#include <QVariant>
#include <QTimer>
#include <QMenu>

#include <kdebug.h>
#include <kblog/blog.h>
#include <kblog/blogpost.h>
#include <kmessagebox.h>
#include <kstandarddirs.h>
#include <kinputdialog.h>
#include <kactioncollection.h>
#include <kapplication.h>

#include "backend.h"
#include "mainwidget.h"
#include "kblogger.h"
#include "composereditor.h"
#include "kbloggerpost.h"
#include "itemsmanager.h"

#define MAGIC_KEY 0xA1B2C3D4
#define VERSION 001
// TODO manage that globally
#define DIR_TO_SAVE_ENTRIES "entries/"

namespace KBlogger
{

/**
 *   PostsList
 */
PostsList::PostsList( const QString& blogName, EntriesType type,
                      Application* kbloggerParent, QWidget* parent ):
        ItemList(parent),
        mBackend(Backend::self()),
        mKblogger(kbloggerParent),
        mType(type)
{
    kDebug();
    mMainWidget = MainWidget::self();
    setupUi(this);
    setAttribute(Qt::WA_DeleteOnClose); //Destroy on close
    setBlogname(blogName);
    setFilename(blogName, type);

    //true content column
    listTree->setColumnHidden(6, true);

    //Connect
    connect(listTree, SIGNAL( itemSelectionChanged () ),
            SLOT( previewItem() ));
    connect(listTree, SIGNAL(  itemDoubleClicked(QTreeWidgetItem*, int ) ),
            SLOT( modifyPost() ));
    connect(listTree, SIGNAL( customContextMenuRequested(const QPoint&) ),
            SLOT( showPopup(const QPoint&) ));

    load();
}

PostsList::~PostsList()
{
    kDebug();
    QList<QTreeWidgetItem*> treeList = mPostsMap.keys();
    QList<QTreeWidgetItem*>::ConstIterator it = treeList.begin();
    QList<QTreeWidgetItem*>::ConstIterator end = treeList.end();
    for( ; it != end; it++ ){
        delete mPostsMap[ (*it) ];
        delete ( *it );
    }
}

void PostsList::appendPost(KBloggerPost* post)
{
    kDebug();
    Q_ASSERT(post);

    kDebug() << " Post=" << post << endl
    << " Id=" << post->postId() << endl
    << " Title=" << post->title() << endl
    << " Creation Date=" << post->creationDateTime() << endl
    << " Status=" << post->status();

    if ( post->status() == KBlog::BlogPost::Fetched ) {
        //NOTE Posts in the Sent list have the status set to Created.
        post->setStatus(KBlog::BlogPost::Created);
        // check if the fetched post is stored somewhere already
        if ( ItemsManager::self(this)->isStored(post) ){
            kDebug() << "post is not added" ;
            return;
        }
    }

    //Change container parameter
    post->setContainer( qobject_cast<QWidget*>(this) );
    //Insert Post in the PostsListWidget
    QStringList itemStringList;
    itemStringList
    << post->getBlogName() // column 0
    << post->postId()
    << post->title()
    << post->categories().join(" ")
    << post->creationDateTime().dateTime().toString("ddd, dd MMM yyyy, HH:mm")
    << post->content()
    << QString(); //        << post->publishedString();

    QTreeWidgetItem *item;
    item = new QTreeWidgetItem(itemStringList);
    if ( mType == local )
        item->setIcon( title, post->statusIcon() );
    else
        item->setIcon( id, post->statusIcon() );


    //Insert Post in the mPostsMap
    mPostsMap[item] = post;

    //Insert the item in the listTree
    listTree->insertTopLevelItem(0, item);

    //Update the number of items of the listTree
    mMainWidget->updateItemNumbers();
}

int PostsList::count()
{
    kDebug();
    return mPostsMap.count();
}

void PostsList::removePost( KBloggerPost* postToDelete, bool delPost)
{
    kDebug();
    Q_ASSERT(postToDelete);

    QMap<QTreeWidgetItem*, KBloggerPost*>::iterator it = mPostsMap.begin(), itEnd = mPostsMap.end();
    for ( ; it != itEnd; ++it ) {
        if ( (*it) == postToDelete ) { //if (postInTheWidget->content() == postToDelete->content() ) {
            delete it.key();
            mPostsMap.erase( it );
            break;
        }
    }
    mMainWidget->updateItemNumbers();
}

void PostsList::save()
{
    kDebug();
    Q_ASSERT(!mFilename.isEmpty());

    KApplication::kApplication()->processEvents(QEventLoop::ExcludeUserInputEvents);
    QString fileToSave = KStandardDirs::locateLocal("appdata", mFilename , true);

    kDebug() << "Save " <<  mPostsMap.count()
    << "entries in " << fileToSave ;

    QFile file( fileToSave );
    file.open(QIODevice::WriteOnly);
    QDataStream out(&file);   // we will serialize the data into the file

    // Write a header
    out << (quint32)MAGIC_KEY; //"magic number"
    out << (qint32)VERSION; //binary file version

    out.setVersion(QDataStream::Qt_4_0);

    //write mPostsMap data;
    out << mPostsMap.count();

    QMapIterator<QTreeWidgetItem*, KBloggerPost*> i(mPostsMap);
    while (i.hasNext()) {
        i.next();
        KBloggerPost *post = i.value();
        if (!post) {
            kError() << "post is NULL" ;
            return;
        }
        out << post->getBlogName();
        out << post->postId();
        out << post->title();
        out << post->categories().join(",");
        out << post->getDateTimeString();
        out << post->content();
        out << post->isPrivate();
    }
    file.close();
    KApplication::kApplication()->processEvents();
}

void PostsList::load()
{
    kDebug();
    Q_ASSERT( !mFilename.isEmpty() );

    mPostsMap.clear(); //FIXME Memory Leak?

    QString fileToOpen = KStandardDirs::locate("appdata", mFilename);
    if (fileToOpen.isEmpty()) {
        kDebug() <<  mFilename << " doesn't exist" ;
        return;
    }

    QFile file( fileToOpen );
    file.open(QIODevice::ReadOnly);
    QDataStream in(&file);

    // Read and check the header
    quint32 magic;
    in >> magic;
    if (magic != MAGIC_KEY) {
        kError() << "BAD MAGIC KEY" ;
        return;
    }

    // Read the version
    qint32 version;
    in >> version;
    if (version != VERSION) {
        kError() << "BAD VERSION" ;
        return;
    }

    // Read the number of entries
    qint32 entriesNumber;
    in >> entriesNumber;

    //Getting data from file, clear the listTree Widget and insert the readed items in the widget.
    QString blogName, postID, title, categories, dateTime, content;
    bool noPublished;
    KDateTime creationDate;
    listTree->clear();

    //while ( !in.atEnd() ){
    for ( int i = 0; i < entriesNumber; ++i ) {
        in >> blogName;
        in >> postID;
        in >> title;
        in >> categories; //CATEGORIES
        in >> dateTime;
        in >> content;
        in >> noPublished;

        KBloggerPost *post = new KBloggerPost(blogName, //FIXME BAD CODE:P
                                              title,
                                              content,
                                              categories.split(","),
                                              noPublished,
                                              creationDate,
                                              postID);

        if ( categories.isEmpty() ) { //FIXME
            QStringList categoriesList;
            categoriesList << QString();
            post->setCategories( categoriesList );
        }

        post->setDateTimeString(dateTime);
        if ( mType == sent )
            post->setStatus(KBlog::BlogPost::Created);
        if ( mType == local )
            post->setStatus(KBlog::BlogPost::New);
        /*        if ( mType == trashed )
                    post->setDeleted(true);
                else
                    post->setDeleted(false);
        */
        appendPost( post );
    }
    file.close();
}

QList<KBloggerPost*> PostsList::getEntriesList()
{
    kDebug();
    return mPostsMap.values();
}

void PostsList::previewItem()
{
    kDebug();
    KBloggerPost* mPost = getCurrentPost();
    if ( !mPost ) {
        kWarning() << "mPost is NULL; NOTE: It's not a problem if the TreeWidget is empty.";
        return;
    }
    mMainWidget->viewEntry(mPost);
}

void PostsList::showPopup(const QPoint& pos)
{
    kDebug();

    QTreeWidgetItem * item = 0 ;
    item = listTree->itemAt(pos);
    if (!item) return;

    QMenu menu(i18n("Post menu"), listTree);

    KActionCollection *aktionCollection = mMainWidget->actionCollection();
    const QList< QActionGroup * > actionGroups =  aktionCollection->actionGroups();
    QActionGroup* actionGroup;
    for (int i = 0; i < actionGroups.size(); ++i) {
        if ( actionGroups.at(i)->objectName() == "postActionGroup" ) {
            actionGroup = actionGroups.at(i);
            break;
        }
    }

    QList<QAction *> actions = actionGroup->actions();
    for (int i = 0; i < actions.size(); ++i) {
        if ( actions.at(i) )
            menu.addAction( actions.at(i) );
    }
    //menu.exec(listTree->mapToGlobal(pos));
    menu.exec(QCursor::pos());
}

void PostsList::setFilename( const QString& account, EntriesType type )
{
    kDebug() << "Account='" << account << "'" ;

    QString filename = QString(account).prepend(DIR_TO_SAVE_ENTRIES);
    switch (type) {
    case sent:
        mFilename = filename.append("-sent");
        break;
    case local:
        mFilename = filename.append("-local");
        break;
    case trashed:
        mFilename = filename.append("-trash");
        break;
    }
}

void PostsList::trashPost()
{
    kDebug() << "PostsList::trashPost()" ;

    //Create the item
    KBloggerPost* post = 0;
    post = getCurrentPost();

    //test
    if (!post) {
        kError() << "post is NULL";
        return;
    }

    switch ( mType ) {
    case trashed: {
        if ( !post->postId().isEmpty() ) {
            //Remove post from the server
            mBackend->removePost( post, MainWidget::self(0, 0) );
        } else {
            //Remove local post from the trash
            removePost(post, true);
        }

    }
    break;

    default:
        //Add post in the trash widget.
        ItemsManager::self(this)->trashPost(post);
        //Remove post from the current widget.
        removePost(post, false);
    }
}
void PostsList::deletePost()
{
    kDebug();

    KBloggerPost* post = 0;
    post = getCurrentPost();

    //test
    if (!post) {
        kError() << "post is NULL";
        return;
    }

    removePost(post, true);
}

void PostsList::modifyPost()
{
    kDebug() ;

    KBloggerPost *post = 0;
    post = mPostsMap[ listTree->currentItem() ];
    if (!post) return;
    mKblogger->modifyPost(post);
}

KBloggerPost* PostsList::getCurrentPost()
{
    kDebug();
    QTreeWidgetItem* currentQTreeItem = listTree->currentItem();
    if (!currentQTreeItem) return 0;
    return mPostsMap[ currentQTreeItem ];
}

void PostsList::updateSearchLine()
{
    kDebug();
    Q_ASSERT(mKblogger);
    mKblogger->updateSearchLine(listTree);
}
/*
PostsList::EntriesType PostsList::whereIsThisPost()
{
  kDebug();
  return mType;
}
*/
/**
 *  SentPostsList
 */

SentPostsList::SentPostsList( const QString& blogName,
                              Application* kbloggerParent, QWidget* parent )
        : PostsList( blogName, sent, kbloggerParent, parent)
{
    kDebug();

    ItemsManager::self(this)->addInTheWidgetsList(blogName, this);
    listTree->setColumnHidden(blog, true);
    listTree->setColumnHidden(content, true);
}

SentPostsList::~SentPostsList()
{
    kDebug();
    //mBackend->disconnect();
}

void SentPostsList::updatePostsSentList()
{
    kDebug();

//     mBackend->update Backend(blogname());
    int downloadCount = KInputDialog::getInteger(i18n("Update Sent List"),
                        i18n("How many entries would you like to update?"),
                        10, 1, 50, 1, 0, this);
    if ( downloadCount < 1 ) return;
    mBackend->listPosts(blogname(), downloadCount, MainWidget::self(0, 0) );
}


/**
 *   LocalDraftPostsList
 */
LocalDraftPostsList::LocalDraftPostsList( const QString& blogName,  Application* kbloggerParent, QWidget* parent ): PostsList(blogName, local, kbloggerParent, parent)
{
    kDebug();

    ItemsManager::self(this)->addInTheWidgetsList(blogName, this);
    listTree->setColumnHidden(blog, true);
    listTree->setColumnHidden(id, true);
    listTree->setColumnHidden(content, true);
}

LocalDraftPostsList::~LocalDraftPostsList()
{
    kDebug();
}

/**
 *   TrashedPostsList
 */
TrashedPostsList::TrashedPostsList( const QString& blogName,  Application* kbloggerParent, QWidget* parent ): PostsList(blogName, trashed, kbloggerParent, parent)
{
    kDebug();

    ItemsManager::self(this)->addInTheWidgetsList(blogName, this);
    listTree->setColumnHidden(blog, true);
    listTree->setColumnHidden(content, true);
}

TrashedPostsList::~TrashedPostsList()
{
    kDebug();
}

} //namespace
#include "postslist.moc"
