/***************************************************************************
                          labelprinter.cpp  -  description
                             -------------------
    begin                : Son Mai 5 2002
    copyright            : (C) 2002 by Dominik Seichter
    email                : domseichter@web.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "labelprinter.h"
#include "kbarcode.h"
#include "label.h"
#include "sqltables.h"
#include "printersettings.h"
#include "printlabeldlg.h"
#include "smalldialogs.h"

// Other includes
#include <stdlib.h>
#include <stdio.h>

// QT includes
#include <qclipboard.h>
#include <qcursor.h>
#include <qgroupbox.h>
#include <qlayout.h>
#include <qpainter.h>
#include <qpaintdevicemetrics.h>
#include <qprogressdialog.h>
#include <qsqlcursor.h>
#include <qvalidator.h>

// KDE includes
#include <kapplication.h>
#include <kcombobox.h>
#include <kfiledialog.h>
#include <kiconloader.h>
#include <kimageio.h>
#include <klistview.h>
#include <klineedit.h>
#include <klineeditdlg.h>
#include <klocale.h>
#include <kmenubar.h>
#include <kmessagebox.h>
#include <kpopupmenu.h>
#include <knuminput.h>
#include <kprinter.h>
#include <kurlrequester.h>

#define MNU_IMPORT_ID 9999

// a small helper function
QString removeQuote( QString text, QString quote )
{
    if( quote.isEmpty() )
        return text;
        
    text = text.stripWhiteSpace();

    if( text.left( quote.length() ) == quote )
        text = text.mid( quote.length(), text.length() - quote.length() );

    if( text.right( quote.length() ) == quote )
        text = text.left( text.length() - quote.length() );

    return text;
}

LabelPrinter::LabelPrinter( QString filename, QWidget *parent, const char *name, WFlags f )
    : DSMainWindow(parent, name, f)
{
    def = 0;
    
    QGroupBox* w = new QGroupBox( this );
    w->setColumnLayout(0, Qt::Vertical );
    pageLayout = new QVBoxLayout( w->layout() );
    setCentralWidget( w );

    buttonLayout = new QHBoxLayout( 0, 6, 6 );
    dataLayout = new QVBoxLayout( 0, 6, 6 );
    QSpacerItem* spacer = new QSpacerItem( 20, 20, QSizePolicy::Expanding, QSizePolicy::Expanding );

    list = new KListView( w );
    list->addColumn( i18n("Index") );
    list->addColumn( i18n("Number of Labels") );
    list->addColumn( i18n("Article Number") );
    list->addColumn( i18n("Group") );
    list->setAllColumnsShowFocus( true );
    connect( list, SIGNAL(clicked(QListViewItem*,const QPoint &,int)),
             this, SLOT(changeItem(QListViewItem*,const QPoint &,int)));

    QGridLayout* infoGrid = new QGridLayout();
        
    info1 = new QLabel( w );
    info2 = new QLabel( w );
    info3 = new QLabel( w );
    info4 = new QLabel( w );
    info5 = new QLabel( w );

    infoGrid->addWidget( info1, 0, 0 );
    infoGrid->addWidget( info2, 0, 1 );
    infoGrid->addWidget( info3, 0, 2 );
    infoGrid->addWidget( info4, 0, 3 );
    infoGrid->addMultiCellWidget( info5, 1, 1, 0, 3 );
    
    buttonPrint = new KPushButton( w );
    buttonPrint->setText( i18n("&Print...") );
    buttonPrint->setIconSet( SmallIconSet("fileprint") );
    buttonPrint->setEnabled( false );

    buttonImages = new KPushButton( w );
    buttonImages->setText( i18n("&Create Images...") );
    buttonImages->setIconSet( SmallIconSet("image") );
    buttonImages->setEnabled( false );

    gItem = new QGroupBox( i18n("Add Item"), w );
    gItem->setColumnLayout( 0, Qt::Vertical );
    gItem->layout()->setSpacing( 6 );
    gItem->layout()->setMargin( 11 );
    QGridLayout* grid = new QGridLayout( gItem->layout() );

    QLabel* label = new QLabel( gItem );
    label->setText( i18n( "Customer name and no.:" ) );
    grid->addWidget( label, 0, 0 );

    customerName = new KComboBox( false, gItem );
    grid->addWidget( customerName, 0, 1 );

    customerId = new KComboBox( false, gItem );
    grid->addWidget( customerId, 0, 2 );

    customerName->setEnabled( false );
    customerId->setEnabled( false );
    
    label = new QLabel( gItem );
    label->setText( i18n( "Label:" ) );
    grid->addWidget( label, 1, 0 );
    
    urlrequester = new KURLRequester( gItem );
    urlrequester->setMode( KFile::File | KFile::ExistingOnly );
    urlrequester->setFilter( "*.kbarcode" );
    grid->addMultiCellWidget( urlrequester, 1, 1, 1, 2 );

    label = new QLabel( gItem );
    label->setText( i18n("Group:") );
    grid->addWidget( label, 2, 0 );

    groupName = new KLineEdit( gItem );
    grid->addMultiCellWidget( groupName, 2, 2, 1, 2 );

    numberArticles = new KIntNumInput( 1, gItem );
    numberArticles->setLabel( i18n( "Number of labels:" ), KNumInput::AlignLeft | KNumInput::AlignVCenter );
    numberArticles->setRange( 1, 10000, 1, false );
    grid->addWidget( numberArticles, 3, 0 );

    label = new QLabel( gItem );
    label->setText( i18n( "Article number:" ) );
    grid->addWidget( label, 3, 1 );

    articleId = new KLineEdit( gItem );
    grid->addWidget( articleId, 3, 2 );

    buttonAdd = new KPushButton( gItem );
    buttonAdd->setText( i18n( "&Add" ) );

    buttonRemove = new KPushButton( gItem );
    buttonRemove->setText( i18n("&Remove" ) );

    buttonRemoveAll = new KPushButton( gItem );
    buttonRemoveAll->setText( i18n("R&emove All") );

    grid->addWidget( buttonAdd, 6, 0 );
    grid->addWidget( buttonRemove, 6, 1 );
    grid->addWidget( buttonRemoveAll, 6, 2 );
    
    dataLayout->addWidget( gItem );
    dataLayout->addWidget( list );
    dataLayout->setStretchFactor( list, 2 );
    dataLayout->addLayout( infoGrid );

    buttonLayout->addItem( spacer );
    buttonLayout->addWidget( buttonPrint );
    buttonLayout->addWidget( buttonImages );

    pageLayout->addLayout( dataLayout );
    pageLayout->addLayout( buttonLayout );
    pageLayout->setStretchFactor( dataLayout, 2 );
    pageLayout->addItem( spacer );

    description = "";
    measure = 0;

    DSMainWindow::setupActions();
    setupActions();

    setupSql();

    connect( buttonAdd, SIGNAL( clicked() ), this, SLOT( addItem() ) );
    connect( buttonRemove, SIGNAL( clicked() ), this, SLOT( removeItem() ) );
    connect( buttonRemoveAll, SIGNAL( clicked() ), list, SLOT( clear() ) );
    connect( buttonRemoveAll, SIGNAL( clicked() ), this, SLOT( updateInfo() ) );
    connect( buttonPrint, SIGNAL( clicked() ), this, SLOT( print() ) );
    connect( buttonImages, SIGNAL( clicked() ), this, SLOT( images() ) );
    connect( customerName, SIGNAL( activated(int) ), this, SLOT( customerNameChanged(int) ) );
    connect( customerId, SIGNAL( activated(int) ), this, SLOT( customerIdChanged(int) ) );
    connect( urlrequester, SIGNAL( textChanged( const QString & ) ), this, SLOT( changeDescription() ) );
    connect( this, SIGNAL( connectedSQL() ), this, SLOT( setupSql() ) );
    connect( SqlTables::getInstance(), SIGNAL( tablesChanged() ), this, SLOT( setupSql() ) );
    connect( SqlTables::getInstance(), SIGNAL( connectedSQL() ), this, SLOT( setupSql() ) );

    if( !filename.isEmpty() )
        loadFromFile( filename );
    
    updateInfo();
    show();
}

LabelPrinter::~LabelPrinter()
{
    DSMainWindow::saveConfig();
    if( measure )
        delete measure;
    if( def )
        delete def;
}

void LabelPrinter::setupActions()
{
    mnuImport = new KPopupMenu( this );
    mnuImport->insertItem( i18n("Import from File ..."), this, SLOT( loadFromFile() ) );
    mnuImport->insertItem( i18n("Import from Clipboard ..."), this, SLOT( loadFromClipboard() ) );
    mnuImport->insertItem( i18n("Import barcode_basic"), this, SLOT( addAllItems() ) );
    
    menuBar()->insertItem( i18n("&Import"), mnuImport, MNU_IMPORT_ID, 1 );
    menuBar()->setItemEnabled( MNU_IMPORT_ID, false );
}

void LabelPrinter::setupSql()
{
    SqlTables* tables = SqlTables::getInstance();
    if( !tables->isConnected() )
        return;
        
    QSqlCursor cur( TABLE_CUSTOMER );
    cur.select();
    customerId->clear();
    customerName->clear();
    while ( cur.next() ) {
        customerId->insertItem( cur.value("customer_no" ).toString() );
        customerName->insertItem( cur.value("customer_name" ).toString() );
    }

    KCompletion* comp = articleId->completionObject();
    comp->clear();
    QSqlQuery query( "select article_no from " TABLE_BASIC " order by article_no" );
    QStringList slist;
    while ( query.next() )
        slist.append( query.value(0).toString() );

    comp->setItems( slist );

    menuBar()->setItemEnabled( MNU_IMPORT_ID, true );    
    customerName->setEnabled( true );
    customerId->setEnabled( true );
}

void LabelPrinter::setCustomerId( QString id )
{
    for( int i = 0; i < customerId->count(); i++ )
        if( customerId->text( i ) == id ) {
            customerId->setCurrentItem( i );
            customerIdChanged( i );
            break;
        }
}

void LabelPrinter::setLabelURL( QString url )
{
    urlrequester->setURL( url );
}

bool LabelPrinter::existsArticle( QString article )
{
    if( article.isEmpty() )
        return false;
    
    QSqlQuery query( "select uid from barcode_basic where article_no='" + article + "'" );
    while ( query.next() )
        return true;

    return false;       
}

void LabelPrinter::addItem()
{
    if( !articleId->text().isEmpty() && !existsArticle( articleId->text() ) ) {
        KMessageBox::error( this, i18n("Please enter a valid article ID") );
        return;
    }

    QString temp;
    temp.sprintf("%0*i", 5, list->childCount() + 1 );
        
    KListViewItem* item = new KListViewItem( list, temp, QString( "%1" ).arg( numberArticles->value() ),
            articleId->text(), groupName->text() );
    list->insertItem( item );

    updateInfo();
        
    if( !groupName->text().isEmpty() ) {
          bool isinside = false;
          KCompletion* comp = groupName->completionObject();
          QStringList slist = comp->items();
          for( unsigned int i = 0; i < slist.count(); i++ )
              if( slist[i] == groupName->text() ) {
                  isinside = true;
                  break;
              }

          if(!isinside)                
              comp->addItem( groupName->text() );
    }

    articleId->setText("");
    groupName->setText("");
}

void LabelPrinter::addAllItems()
{
    DSSmallDialogs::AddAllDialog* dlg = new DSSmallDialogs::AddAllDialog( this, "dlg" );
    if( dlg->exec() != QDialog::Accepted )
        return;
        
    QString temp;
    QString group = dlg->groupName();
    int n = dlg->numberLabels();

    QSqlQuery query("SELECT article_no FROM " TABLE_BASIC );
    while( query.next() ) {
        temp.sprintf("%0*i", 5, list->childCount() + 1 );

        KListViewItem* item = new KListViewItem( list, temp, QString::number( n ),
                query.value( 0 ).toString(), group );
        list->insertItem( item );
    }
    
    updateInfo();
}

void LabelPrinter::removeItem()
{
    QListViewItem* item = list->firstChild();
    while( item ) {
        if( item->isSelected() ) {
            QListViewItem* it = item->nextSibling();
            delete item;
           
            while( it ) {
                int a = it->text( 0 ).toInt();
                QString temp;
                temp.sprintf("%0*i", 5, a - 1 );                
                it->setText( 0, temp );
                it = it->nextSibling();                
            }
            
            break;
        } else
            item = item->nextSibling();
    }

    updateInfo();    
}

void LabelPrinter::print( bool immediately, QString prn )
{
    PrintLabelDlg pld( this, "pld" );
    pld.setLabelsEnabled( false );
    if( pld.exec() != QDialog::Accepted )
        return;

    int move = pld.position();
    PrinterSettings::getInstance()->getData()->border = pld.border();


    KPrinter* printer = PrinterSettings::getInstance()->setupPrinter( this, immediately, prn );

    if( printer ) {
        QBuffer* buffer = getBuffer();

        if( !buffer ) {
            delete printer;
            return;
        }

        doPrint( printer, buffer, move );
        delete buffer;
        delete printer;
        KApplication::setOverrideCursor( QCursor(Qt::ArrowCursor), true );
    }
}

QBuffer* LabelPrinter::getBuffer()
{
    QFile f( urlrequester->url() );
    if ( !f.open( IO_ReadOnly ) )
        return 0;

    QByteArray byte = f.readAll();
    f.close();

    QBuffer* buffer = new QBuffer( byte );
    if( !buffer->open( IO_ReadOnly ) )
        return 0;
    
    return buffer;
}

void LabelPrinter::checkForNewPage( double ph, KPrinter* printer, Label* label )
{
    if( ph > pageh || c_w == measure->num_v ) {
        printer->newPage();
        label->setPage( label->page() + 1 );
        curh = measure->gap_top;
        curw = measure->gap_left;
        c_w = 0;
    }
}

void LabelPrinter::drawBorders( double x, double y, double w, double h, QPainter* painter )
{
    // Draw borders around labels
    if( PrinterSettings::getInstance()->getData()->border ) {
        painter->setPen( QPen( Qt::black, 1 ) );
        painter->drawRect( (int)x, (int)y, (int)w, (int)h );
    }
    
}

void LabelPrinter::changeLine( int num, double height )
{
    if( c_h >= num ) {
        c_h = 0;
        curw = measure->gap_left;
        curh += height + measure->gap_v;
        c_w++;
    }
}

void LabelPrinter::proccessEvents( int lb, KPrinter* printer, QPainter* p, QString value, Label* label )
{
    if( lb == NEW_PAGE ) {
        if( curh > measure->gap_top || curw > measure->gap_left) {
            printer->newPage();
            label->setPage( label->page() + 1 );
            curh = measure->gap_top;
            curw = measure->gap_left;
            c_h = 0;
            c_w = 0;
        }
    }

    if( lb == LINE_BREAK )
        changeLine( measure->num_h, measure->height );

    if( lb == ARTICLE_GROUP_NO ) {
        changeLine( measure->num_h, measure->height );
        checkForNewPage( curh + measure->height, printer, label );

        Label::getXLabel( curw, curh, measure->width, measure->height, p, lb, value );
        drawBorders( curw, curh, measure->width, measure->height, p );

        curw += measure->width + measure->gap_h;
        c_h++;
    }

    printXLabel( lb, printer, p, label );
}

void LabelPrinter::printXLabel( int lb, KPrinter* printer, QPainter* p, Label* label )
{
    if( lb == LABEL_X ) {
        changeLine( measure->num_h, measure->height );
        checkForNewPage( curh + measure->height, printer, label );

        Label::getXLabel( curw, curh, measure->width, measure->height, p, lb, "" );

        drawBorders( curw, curh, measure->width, measure->height, p );

        curw += measure->width + measure->gap_h;
        c_h++;
    }
}

void LabelPrinter::doPrint( KPrinter* printer, QBuffer* buffer, int move )
{
    QPainter p( printer );
    QPaintDeviceMetrics pdm( p.device() );
    // int pagew = pdm.width();    // in pixel
    pageh = pdm.height();   // in pixel

    curw = 0;
    curh = 0;

    QString labelname = urlrequester->url();

    QProgressDialog* progress = new QProgressDialog( i18n("Printing..."), i18n("&Cancel"), labels+1,this );
    progress->setMinimumDuration( 0 );
    progress->setProgress( 0 );

    // sort by group
    list->setSorting( 3, true );
    list->sort();
    progress->setProgress( 1 );
    
    QListViewItem* item = list->firstChild();
    c_h = c_w = 0;

    labelprinterdata* lpdata = PrinterSettings::getInstance()->getData();
    measure = def->getMeasurements( printer );

    bool moved = false;
    QString oldgroup = QString::null;
    unsigned int index = 0; // counter gets increased for every label, but not for 
                                         // labels like article_no etc. (those who are created
                                         // by events.
    while( item ) {
        Label* l = new Label( def, buffer, labelname, printer, customerId->currentText(),
                 item->text( 2 ), item->text( 3 ) );

        if( !moved ) {
            for( int mi = 0; mi < move-1; mi++ ) {
                if( !curw )
                    curw = measure->gap_left;

                if( !curh )
                    curh = measure->gap_top;

                changeLine( measure->num_h, measure->height );                    

                if( curh + measure->height > pageh || c_w == measure->num_v ) {
                    c_w = 0;
                    curh = measure->gap_top;
                    curw = measure->gap_left;
                }

                curw += measure->width + measure->gap_h;
                c_h++;
            }
            moved = true;
        }

                
        bool newgroup = false;
        if( oldgroup == QString::null )
            newgroup = true;
        else {
            if( item->text( 3 ) != oldgroup )
                newgroup = true;    
        }
        oldgroup = item->text( 3 );                        

        if( !curw )
            curw = measure->gap_left;

        if( !curh )
            curh = measure->gap_top;

        if( newgroup ) {
            proccessEvents( lpdata->groupEvent1, printer, &p, i18n("Group : ") + item->text( 3 ), l );
            proccessEvents( lpdata->groupEvent2, printer, &p, i18n("Group : ") + item->text( 3 ), l );
            proccessEvents( lpdata->groupEvent3, printer, &p, i18n("Group : ") + item->text( 3 ), l );
            proccessEvents( lpdata->groupEvent4, printer, &p, i18n("Group : ") + item->text( 3 ), l );
        }
        proccessEvents( lpdata->articleEvent1, printer, &p, i18n("Article No. : ") + item->text( 2 ), l );
        proccessEvents( lpdata->articleEvent2, printer, &p, i18n("Article No. : ") + item->text( 2 ), l );
        proccessEvents( lpdata->articleEvent3, printer, &p, i18n("Article No. : ") + item->text( 2 ), l );
        proccessEvents( lpdata->articleEvent4, printer, &p, i18n("Article No. : ") + item->text( 2 ), l );

        for( int i = 0; i < item->text( 1 ).toInt(); i++ ) {
            changeLine( measure->num_h, measure->height );
            checkForNewPage( curh + measure->height, printer, l );

            p.setClipRect( (int)curw, (int)curh, (int)measure->width, (int)measure->height );
            l->setIndex( index );
            p.drawPicture( (int)curw, (int)curh, *l->picture() );
            index++;
            p.setClipping( false );
            
            drawBorders( curw, curh, measure->width, measure->height, &p );
            
            curw += measure->width + measure->gap_h;
            c_h++;

            kapp->processEvents( 0 );
            progress->setProgress( progress->progress() + 1 );
            if( progress->wasCancelled() ) {
                printer->abort();
                delete l;
                delete progress;
                return;
            }
        }

        delete l;                
        item = item->nextSibling();
    }

    p.end();
    delete progress;
}

void LabelPrinter::customerIdChanged( int index )
{
    customerName->setCurrentItem( index );
    enableControls();
}

void LabelPrinter::customerNameChanged( int index )
{
    customerId->setCurrentItem( index );
}

void LabelPrinter::loadFromFile()
{
    QString f = KFileDialog::getOpenFileName( 0, 0, this );
    if( !f.isEmpty() )
        loadFromFile( f );
}

void LabelPrinter::loadFromClipboard()
{
    QClipboard *cb = KApplication::clipboard();
    QString text = cb->text();
    loadData( 0, text );
}

void LabelPrinter::loadFromFile( QString f )
{
    QFile file( f );

    if( !file.open( IO_ReadOnly ) ) {
        qDebug("Unable to open file: " + f );
        return;
    }

    loadData( &file );
    
    file.close();
}

void LabelPrinter::loadData( QFile* file, const QString & data )
{
    labelprinterdata* lpdata = PrinterSettings::getInstance()->getData();
    if( lpdata->separator.isEmpty() ) {
        KMessageBox::sorry( this, i18n("Separator is empty. Please set it to a value.") );
        return;
    }

    if( lpdata->comment.isEmpty() ) {
        KMessageBox::sorry( this, i18n("Comment is empty. Please set it to a value.") );
        return;
    }

    // new config entry!!!
    KConfig* config = kapp->config();
    config->setGroup("FileFormat");
    int pos[3] = { config->readNumEntry("Data0", 0 ),
                   config->readNumEntry("Data1", 1 ),
                   config->readNumEntry("Data2", 2 ) };

    bool custom_article_no = lpdata->useCustomNo;
    /*
     * read data here
     */
    QString s;
    QStringList dropped;
    QStringList datalist;
    
    if( !file ) {
        datalist = QStringList::split( "\n", data, false );
    }

    unsigned int i = 0;    
    while( 1 ) {
        if( file ) {
            if( file->readLine( s, 1000 ) == -1 )
                break;
        } else {
            if( i >= datalist.count() )
                break;
                
            s = datalist[i];
            i++;
        }
            
        s = s.stripWhiteSpace();
        if( s.left( lpdata->comment.length() ) == lpdata->comment && !lpdata->comment.isEmpty() )
            /* Line is a comment */
            continue;

        if( s.isEmpty() )
            continue;

        QString data[3];
        data[0] = s.section(lpdata->separator, 0, 0 );
        data[1] = s.section(lpdata->separator, 1, 1 );
        data[2] = s.section(lpdata->separator, 2, 2 );

        QString article = "";
        QString quantity = "";
        QString group = "";

        if( pos[0] == 0 )
            quantity = data[0];
        else if( pos[0] == 1 )
            article = data[0];
        else
            group = data[0];

        if( pos[1] == 0 )
            quantity = data[1];
        else if( pos[1] == 1 )
            article = data[1];
        else
            group = data[1];

        if( pos[2] == 0 )
            quantity = data[2];
        else if( pos[2] == 1 )
            article = data[2];
        else
            group = data[2];

        // data[0] == quantity
        // data[1] == article_no
        // data[2] == group

        quantity = removeQuote( quantity, lpdata->quote );
        article = removeQuote( article, lpdata->quote );
        group = removeQuote( group, lpdata->quote );

        bool qint = false;
        (void)quantity.toInt( &qint );

        if( qint && custom_article_no ) {
            qint = false;
            QSqlQuery query("SELECT article_no FROM customer_text WHERE article_no_customer='" + article + "'" );
            while( query.next() ) {
                article = query.value( 0 ).toString();
                qint = true;
                break;
            }
        }

        if( qint && existsArticle( article ) ) {
            QString index;
            index.sprintf("%0*i", 5, list->childCount()+1 );

            new KListViewItem( list, index, quantity, article, group );

            if( !group.isEmpty() ) {
                bool inside = false;
                KCompletion* comp = groupName->completionObject();
                QStringList slist = comp->items();
                for( unsigned int i = 0; i < slist.count(); i++ )
                    if( slist[i] == groupName->text() ) {
                        inside = true;
                        break;
                    }
                if( !inside )
                    comp->addItem( groupName->text() );
            }
        } else
            qint = false;

        if( !qint )
            dropped.append( quantity + lpdata->separator +  article + lpdata->separator + group );

    }

    updateInfo();

    if( !dropped.isEmpty() )
#if QT_VERSION >= 0x030100
        KMessageBox::informationList( this, i18n("<qt>The following items can not be added:" )+ "</qt>", dropped );
#else
        KMessageBox::questionYesNoList( this, i18n("<qt>The following items can not be added:" )+ "</qt>", dropped );
#endif            
}

void LabelPrinter::updateInfo()
{
    labels = 0;
    int articles = 0;
    int groups = 0;
    int pages = 0;
    
    QStringList slist;
    QStringList glist;
    
    QListViewItem* item = list->firstChild();
    while( item ) {
        labels += item->text( 1 ).toInt();
        if( !glist.contains( item->text( 3 ) ) ) {
            glist.append( item->text( 3 ) );
            groups++;
        }
                    
        if( !slist.contains( item->text( 2 ) ) ) {
            articles++;
            slist.append( item->text( 2 ) );
        }
        
        item = item->nextSibling();
    }

    // Calculate numer of pages
    // algorithm by Stonki!
    int num_h = 0;
    int num_v = 0;

    if( def ) {
        measurements* m = def->getMeasurements();
        num_h = m->num_h;
        num_v = m->num_v;
    }
    
    int number_page = num_h * num_v;
    int fake = 0;

    if( number_page ) {
        pages = int((fake + labels) / number_page);
        if( (fake + labels) % number_page > 0 && labels)
            pages++;
    }

    info1->setText( QString(i18n("Labels to print: %1")).arg( labels ) );
    info2->setText( QString(i18n("Different articles: %1")).arg( articles ) );
    info3->setText( QString(i18n("Different groups: %1")).arg( groups ) );
    info4->setText( QString(i18n("Pages: %1")).arg( pages ) );
    info5->setText( i18n("Label description: ") + description );

    enableControls();
}

void LabelPrinter::changeDescription()
{
    QFile f( urlrequester->url() );
    if ( !f.open( IO_ReadOnly ) )
        return;

    QDomDocument* doc = new QDomDocument( "KBarcodeLabel" );
    if ( !doc->setContent( &f ) ) {
        f.close();
        delete doc;
        return;
    }
    
    f.close();
    QDomNode n = doc->documentElement().firstChild();
    while( !n.isNull() ) {
        QDomElement e = n.toElement(); // try to convert the node to an element.
        if( !e.isNull() )
            if( e.tagName() == "label" ) {
                QDomNode node = e.firstChild();
                while( !node.isNull() ) {
                    QDomElement e = node.toElement(); // try to convert the node to an element.
                    if( !e.isNull() )
                        if( e.tagName() == "description" )
                            description = e.text();
                        else if( e.tagName() == "id" )
                            def = readDefinition( &e, this );
                            
                    node = node.nextSibling();
                }
                n = n.nextSibling();
            }

        n = n.nextSibling();
    }

    delete doc;

    updateInfo();
}

void LabelPrinter::changeItem( QListViewItem* item, const QPoint &, int col )
{
    if(!item)
        return;
        
    bool ok = false;
    QString text = item->text( col );
    QIntValidator v( 1, 100000, this );
    QRegExpValidator vr( QRegExp("*", true, true ), 0 );
    switch( col ) {
        case 1:
            text = KLineEditDlg::getText( "", item->text( col ),
                   &ok, this, &v );
            break;
        case 2:
            text = KLineEditDlg::getText( "", item->text( col ),
                   &ok, this );

            if( !text.isEmpty() )
                if( !existsArticle( text ) ) {
                    KMessageBox::error( this, i18n("Please enter a valid article ID") );
                    ok = false;
                }
            break;
        case 3:
            text = KLineEditDlg::getText( "", item->text( col ),
                   &ok, this, &vr );
            break;
        default:
            break;
    };

    if( ok ) {
        item->setText( col, text );
        updateInfo();
    }
}

void LabelPrinter::enableControls()
{
    bool e = (!urlrequester->url().isEmpty() && def && list->childCount() && def->getId() > -1 ); 

    buttonPrint->setEnabled( e );
    buttonImages->setEnabled( e );
    
    if( list->childCount() ) {
        buttonRemove->setEnabled( true );
        buttonRemoveAll->setEnabled( true );
    } else {
        buttonRemove->setEnabled( false );
        buttonRemoveAll->setEnabled( false );
    }
}

void LabelPrinter::images()
{
    QString dirname = KFileDialog::getExistingDirectory ( QString::null, this, i18n("Select output directory") );
    if( dirname.isEmpty() )
        return;
        
    QString labelname = urlrequester->url();
    QBuffer* buffer = getBuffer();
    if( !buffer )
        return;
    
    QListViewItem* item = list->firstChild();
    
    QProgressDialog* progress = new QProgressDialog( i18n("Creating Images..."), i18n("&Cancel"), labels+1,this );
    progress->setMinimumDuration( 0 );
    progress->setProgress( 0 );

    measure = def->getMeasurements( this );

    KConfig* conf = kapp->config();
    conf->setGroup("ImageCreation");
    bool filenamearticle = conf->readBoolEntry( "FilenameArticle", "true" );
    QString format = conf->readEntry( "FileFormat", "PNG" );
   
    QPainter p;
    QPixmap pixmap( (int)measure->width, (int)measure->height );
    while( item ) {
        Label* l = new Label( def, buffer, labelname, this, customerId->currentText(),
                 item->text( 2 ), item->text( 3 ) );

        QPicture* pic = l->picture();
        pixmap.fill( Qt::white );
        p.begin( &pixmap );
        p.drawPicture( 0, 0, *pic );
        p.end();
        for( int i = 0; i < item->text( 1 ).toInt(); i++ ) {
            QString filename = dirname + "/" +
                               ( filenamearticle ? item->text( 2 ) : l->barcodeNo() )
                               + QString("_%1.").arg( i ) + KImageIO::suffix( format );

            unsigned int c = 0;
            while( QFile::exists( filename ) ) {
                filename += "." + QString::number( c );
                c++;
            }
                        
            pixmap.save( filename, format );

            kapp->processEvents( 0 );
            progress->setProgress( progress->progress() + 1 );
            if( progress->wasCancelled() ) {
                delete l;
                delete progress;
                return;
            }
        }

        delete l;
        item = item->nextSibling();
    }
    delete buffer;
    delete measure;
    delete progress;

    KMessageBox::information( this, i18n("Images creation finished.") );
}



