/***************************************************************************
                          labeleditor.cpp  -  description
                             -------------------
    begin                : Die Apr 23 2002
    copyright            : (C) 2002 by Dominik Seichter
    email                : domseichter@web.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "labeleditor.h"
#include "label.h"
#include "commands.h"
#include "mycanvasitem.h"
#include "mycanvasview.h"
#include "kbarcode.h"
#include "newlabel.h"
#include "mybarcode.h"
#include "previewdialog.h"
#include "configdialog.h"
#include "databasebrowser.h"
#include "labelprinter.h"
#include "sqltables.h"
#include "multilineeditdlg.h"
#include "printersettings.h"
#include "barcodecombo.h"
#include "barcodedialog.h"
#include "rectsettingsdlg.h"
#include "printlabeldlg.h"
#include "mimesources.h"
#include "zplutils.h"

// QT includes
#include <qbuffer.h>
#include <qcanvas.h>
#include <qcheckbox.h>
#include <qclipboard.h>
#include <qdockarea.h>
#include <qdom.h>
#include <qdragobject.h> 
#include <qgroupbox.h>
#include <qimage.h>
#include <qinputdialog.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qmap.h>
#include <qmime.h>
#include <qpainter.h>
#include <qpaintdevicemetrics.h>
#include <qpicture.h>
#include <qpoint.h>
#include <qprogressdialog.h>
#include <qsqlquery.h>
#include <qtextbrowser.h>
#include <qtooltip.h>
#include <qvalidator.h>
#include <qxml.h>
#if QT_VERSION <= 0x030100
    #include <qregexp.h>
#endif

// KDE includes
#include <kabc/address.h>
#include <kabc/addressee.h>
#include <kabc/addresseedialog.h>
#include <kabc/phonenumber.h>
#include <kabc/stdaddressbook.h>
#include <kaction.h>
#include <kapplication.h>
#include <kcolordialog.h>
#include <kcommand.h>
#include <kcombobox.h>
#include <kfiledialog.h>
#include <kiconloader.h>
#include <kimageio.h>
#include <klineedit.h>
#include <klistbox.h>
#include <klocale.h>
#include <kmenubar.h>
#include <kmessagebox.h>
#include <knuminput.h>
#include <kpopupmenu.h>
#include <kpushbutton.h>  
#include <kprinter.h>
#include <krun.h>
#include <kspell.h>
#include <kstatusbar.h>
#include <kstandarddirs.h>
#include <ktempfile.h>

#define STATUS_ID_SIZE 100
#define STATUS_ID_TEMPLATE 101
#define STATUS_ID_MOUSE 102

#define ID_ITEM_GROUP 6000
#define ID_ITEM_FILENAME 6001
#define ID_ITEM_DATE 6002
#define ID_ITEM_ARTICLE_NO 6003
#define ID_ITEM_ARTICLE_DESCRIPTION 6004
#define ID_ITEM_BARCODE_NO 6005
#define ID_ITEM_FIELD_0 6006
#define ID_ITEM_FIELD_1 6007
#define ID_ITEM_FIELD_2 6008
#define ID_ITEM_FIELD_3 6009
#define ID_ITEM_FIELD_4 6010
#define ID_ITEM_FIELD_5 6011
#define ID_ITEM_FIELD_6 6012
#define ID_ITEM_FIELD_7 6013
#define ID_ITEM_FIELD_8 6014
#define ID_ITEM_FIELD_9 6015
#define ID_ITEM_CUSTOMER_NAME 6016
#define ID_ITEM_CUSTOMER_NO 6017
#define ID_ITEM_LINE_0 6018
#define ID_ITEM_LINE_1 6019
#define ID_ITEM_LINE_2 6020
#define ID_ITEM_LINE_3 6021
#define ID_ITEM_LINE_4 6022
#define ID_ITEM_LINE_5 6023
#define ID_ITEM_LINE_6 6024
#define ID_ITEM_LINE_7 6025
#define ID_ITEM_LINE_8 6026
#define ID_ITEM_LINE_9 6027
#define ID_ITEM_ENCODING_TYPE 6028
#define ID_RESOLUTION 6029

#define ID_LOCK_ITEM 6050;

using namespace KABC;

LabelEditor::LabelEditor( QWidget *parent, QString _filename, const char *name, WFlags f )
    : DSMainWindow( parent, name, f )
{
    description = "";
    d = new Definition();
    history = new KCommandHistory( actionCollection(), false );

    statusBar()->insertItem( "", STATUS_ID_TEMPLATE, 0, true );
    statusBar()->insertItem( "", STATUS_ID_SIZE, 0, true );
    statusBar()->insertItem( "", STATUS_ID_MOUSE, 2, true );
    statusBar()->setSizeGripEnabled( true );
    statusBar()->show();
    
    c = new MyCanvas( this );
    c->setDoubleBuffering( true );
    c->setUpdatePeriod( 50 );

    cv = new MyCanvasView( d, c, this );
    cv->setHistory( history );
    cv->setPosLabel( statusBar(), STATUS_ID_MOUSE );
    setCentralWidget( cv );

    setupActions();
    setAutoSaveSettings( QString("Window") + name, true );

    loadConfig();
    clearLabel();
    show();

//    if( isFirstStart() )
//        moveDockWindow( tools, Qt::DockLeft );

    connect( cv, SIGNAL( doubleClickedItem(QCanvasItem*) ), this, SLOT( doubleClickedItem(QCanvasItem*) ) );
    connect( cv, SIGNAL( showContextMenu(QCanvasItem*, QPoint) ), this, SLOT( showContextMenu(QCanvasItem*,QPoint) ) );
    connect( cv, SIGNAL( movedSomething() ), this, SLOT( setEdited() ) );
    connect( kapp, SIGNAL( aboutToQuit() ), this, SLOT( saveConfig() ) );

    connect( history, SIGNAL( commandExecuted() ), cv, SLOT( updateGUI() ) );
    connect( history, SIGNAL( commandExecuted() ), this, SLOT( setEdited() ) );

    if( !_filename.isEmpty() )
        load( _filename );
    else {
        if( newdlg )
            startupDlg( CreateNewLabel );
    }
}

LabelEditor::~LabelEditor()
{
    delete d;
}

void LabelEditor::loadConfig()
{
    KConfig* config = kapp->config();
    recentAct->loadEntries( config, "RecentFiles" );

    config->setGroup("LabelEditor");
    c->setGrid( gridsize );
    c->setGrid( gridcolor );
    history->setUndoLimit( undolimit );
    history->setRedoLimit( undolimit );

    gridAct->setChecked( config->readBoolEntry("gridenabled", false ) );
    toggleGrid();
}

void LabelEditor::saveConfig()
{
    KConfig* config = kapp->config();

    recentAct->saveEntries( config, "RecentFiles" );

    config->setGroup("LabelEditor");
    config->writeEntry("gridenabled", gridAct->isChecked() );

    config->sync();

    DSMainWindow::saveConfig();
}

void LabelEditor::clearLabel()
{
    QCanvasItemList list = c->allItems();
    // First delete all items:
    for( unsigned int i = 0; i < list.count(); i++ )
        delete list[i];

    description = "";
    history->clear();
    m_edited = false;

    updateInfo();
    c->update();
    cv->repaintContents();
}

bool LabelEditor::save()
{
    bool ret;
    if( filename.isEmpty() )
        ret = saveas();
    else
        ret = save( filename );

    KURL url;
    url.setPath( filename );
    recentAct->addURL( url );
        
    updateInfo();

    return ret;
}

bool LabelEditor::saveas()
{
    QString name = KFileDialog::getSaveFileName ( NULL, "*.kbarcode", this );
    if(name.isEmpty())
        return false;

    if( name.right(9).lower() != ".kbarcode" )
        name += ".kbarcode";

    return save( name );
}

bool LabelEditor::save( QString name )
{
    if( QFile::exists( name ) )
        QFile::remove( name );
        
    QFile f( name );
    if ( !f.open( IO_ReadWrite ) )
        return false;

    QDomDocument doc( "KBarcodeLabel" );
    QDomElement root = doc.createElement( "kbarcode" );
    doc.appendChild( root );

    QDomElement data = doc.createElement( "label" );

    if( !description.isEmpty() ) {
        QDomElement labeldescription = doc.createElement( "description" );
        labeldescription.appendChild( doc.createTextNode( description ) );
        data.appendChild( labeldescription );
    }

    QDomElement labelid = doc.createElement( "id" );
    writeDefinition( &labelid, d );
    labelid.appendChild( doc.createTextNode( QString( "%1" ).arg(d->getId()) ) );

    data.appendChild( labelid );
    root.appendChild( data );

    QCanvasItemList list = c->allItems();
    for( unsigned int i = 0; i < list.count(); i++ )
        if( list[i]->rtti() == BARCODE_RTTI ) {
            // Barcode
            CanvasBarcode* bcode = (CanvasBarcode*)list[i];
            barcodeData data = bcode->data();
            QDomElement tag = doc.createElement( "barcode" );
            writeBarcode( &tag, &data, bcode, cv );

            root.appendChild( tag );
        } else if( list[i]->rtti() == PIC_RTTI ) {
            // Picture
            PictureRectangle* rect = (PictureRectangle*)list[i];
            QDomElement tag = doc.createElement( "picture" );
            writeXMLRect( &tag, QRect( (int)rect->x(), (int)rect->y(), rect->width(), rect->height() ), cv );

            tag.setAttribute( "z", list[i]->z() );
            tag.setAttribute( "rotation", rect->getRotation() );

            QBuffer buf;
            QImageIO* iio = new QImageIO( &buf, "XPM" );
            iio->setImage( rect->getImage() );

            if(!buf.open( IO_WriteOnly ))
                qDebug("Can't open buffer!");
            
            (void)iio->write();

            buf.close();
            delete iio;
            tag.appendChild( doc.createCDATASection( QString( buf.buffer() ) ) );
            root.appendChild( tag );
        } else if( list[i]->rtti() == TEXT_RTTI ) {
            // Text 
            CanvasText* mytext = (CanvasText*)list[i];
            QDomElement tag = doc.createElement( "textfield" );
            writeXMLRect( &tag, QRect( (int)mytext->x(), (int)mytext->y(), mytext->width(), mytext->height() ), cv );

            tag.setAttribute( "z", mytext->z() );
            tag.setAttribute( "rotation", mytext->getRotation() );

            QDomElement texttag = doc.createElement( "text" );
            texttag.appendChild( doc.createTextNode( mytext->text() ) );

            tag.appendChild( texttag );
            root.appendChild( tag );
        } else if( list[i]->rtti() == RECT_RTTI ) {
            DrawingRect* rc = (DrawingRect*)list[i];
            QDomElement tag = doc.createElement( "rect" );
            writeXMLRect( &tag, QRect( (int)rc->x(), (int)rc->y(), rc->width(), rc->height() ), cv );
            
            tag.setAttribute( "z", rc->z() );
            tag.setAttribute( "circle", rc->isCircle() );
            tag.setAttribute( "borderwidth", rc->borderWidth() );
            tag.setAttribute( "borderstyle", rc->borderStyle() );
            writeXMLColor( &tag, "color", rc->color() );
            writeXMLColor( &tag, "bordercolor", rc->borderColor() );

            root.appendChild( tag );        
        } else if( list[i]->rtti() == LINE_RTTI ) {
            MyCanvasLine* cl = (MyCanvasLine*)list[i];
            QDomElement tag = doc.createElement( "line" );

            QPoint translation = cv->getTranslation();
            QPoint s = cl->startPoint() - translation;
            QPoint e = cl->endPoint() - translation;
                                                
            tag.setAttribute( "x1", s.x() );
            tag.setAttribute( "y1", s.y() );
            tag.setAttribute( "x2", e.x() );
            tag.setAttribute( "y2", e.y() );

            tag.setAttribute( "z", cl->z() );
            tag.setAttribute( "width", cl->pen().width() );
            tag.setAttribute( "style", cl->pen().style() ); 
            writeXMLColor( &tag, "color", cl->pen().color() );

            root.appendChild( tag );            
        }

    QCString xml = doc.toCString();
    f.writeBlock( xml, xml.length() );
    f.close();

    filename = name;
    history->documentSaved();
    m_edited = false;

    enableActions();    
    setCaption( filename, false );
        
    return true;
}

bool LabelEditor::load()
{
    QString name = KFileDialog::getOpenFileName ( NULL, "*.kbarcode", this, i18n("Select Label") );
    if(name.isEmpty()) return false;

    return load( name );    
}

bool LabelEditor::load( QString name )
{
    if( name.isEmpty() ) {
        return load();
    }

    filename = name;
    setCaption( filename, false );
    
    QFile f( name );
    if ( !f.open( IO_ReadOnly ) )
        return false;

    clearLabel();

    QDomDocument doc( "KBarcodeLabel" );
    if ( !doc.setContent( &f ) ) {
        f.close();
        return false;
    }
    f.close();

    QDomNode n = doc.documentElement().firstChild();
    // this should not be neccessary,
    // but <label><id> needs to be processed first
    while( !n.isNull() ) {
        QDomElement e = n.toElement(); // try to convert the node to an element.
        if( !e.isNull() )
            // label has to be the first xml element!
            if( e.tagName() == "label" ) {
                QDomNode n = e.firstChild();
                while( !n.isNull() ) {
                    QDomElement e = n.toElement(); // try to convert the node to an element.
                    if( !e.isNull() )                                                  
                        if( e.tagName() == "description" )
                            description = e.text();
                        else if( e.tagName() == "id" ) {
                            delete d;
                            d = readDefinition( &e, this );
                            cv->setDefinition( d );
                        }

                    n = n.nextSibling();
                }
            }
        n = n.nextSibling();
    }
                
    n = doc.documentElement().firstChild();
    while( !n.isNull() ) {
        QDomElement e = n.toElement(); // try to convert the node to an element.
        if( !e.isNull() )
            if( e.tagName() == "barcode" ) {
                barcodeData data = readBarcode( &e, cv );
                CanvasBarcode* bcode = new CanvasBarcode( c, &data );
                bcode->move( data.xml.x, data.xml.y );
                bcode->updateBarcode();
                bcode->setZ( e.attribute( "z", "0" ).toInt() );
            } else if( e.tagName() == "textfield" ) {
                CanvasText* t = new CanvasText( c, this );
                QRect r = readXMLRect( &e, cv );
                
                t->move( r.x(), r.y() );
                t->setRot( e.attribute("rotation", "0" ).toDouble() );
                t->setZ( e.attribute( "z", "0" ).toInt() );

                QString etext;                
                QDomNode n = e.firstChild();
                while( !n.isNull() ) {
                    QDomElement e = n.toElement(); // try to convert the node to an element.
                    if( !e.isNull() )
                        if( e.tagName() == "text" )
                            etext = e.text();
                    n = n.nextSibling();
                }

                t->setSize( r.width(), r.height() );
                t->setText( etext );
                t->show();
            } else if( e.tagName() == "picture" ) {
                PictureRectangle* r = new PictureRectangle( c );
                r->setRot( e.attribute("rotation", "0" ).toDouble() );

                QRect rect = readXMLRect( &e, cv );
                r->move( rect.x(), rect.y() );
                                    
                QImage* p = new QImage();
                p->loadFromData( e.text().utf8(), "XPM" );
                r->setImage( p );

                r->setZ( e.attribute( "z", "0" ).toInt() );
                if( rect.width() > 0 && rect.height() > 0 )
                    r->setSize( rect.width(), rect.height() );
                                    
                r->show();
                r->redraw();
            } else if( e.tagName() == "rect" ) {
                DrawingRect* r = new DrawingRect( c );
                QRect rect = readXMLRect( &e, cv );
                r->move( rect.x(), rect.y() );
                r->setZ( e.attribute( "z", "0" ).toInt() );
                r->setCircle( e.attribute( "circle", "0" ).toInt() );
                r->setSize( rect.width(), rect.height() );
                r->setBorderWidth( e.attribute( "borderwidth", "1" ).toInt() );
                r->setBorderStyle( e.attribute( "borderstyle", "1" ).toInt() );
                r->setColor( readXMLColor( &e, "color", Qt::black ) );
                r->setBorderColor( readXMLColor( &e, "bordercolor", Qt::black ) );
            } else if( e.tagName() == "line" ) {
                MyCanvasLine* cl = new MyCanvasLine( c );
                QPoint translation = cv->getTranslation();
                cl->setPoints( e.attribute( "x1", "0" ).toInt() + translation.x(),
                               e.attribute( "y1", "0" ).toInt() + translation.y(),
                               e.attribute( "x2", "0" ).toInt() + translation.x(),
                               e.attribute( "y2", "0" ).toInt()  + translation.y() );

                cl->setZ( e.attribute( "z", "0" ).toInt() );
                cl->setPen( QPen( readXMLColor( &e, "color", Qt::black ),
                            e.attribute( "width", "0" ).toInt(), (QPen::PenStyle)e.attribute( "style", "0" ).toInt() ) );
                cl->show();
            }

        n = n.nextSibling();
    }

    KURL url;
    url.setPath( filename );
    recentAct->addURL( url );

    enableActions();
    cv->repaintContents( true );

    return true;
}

bool LabelEditor::newLabel()
{
    NewLabel* nl = new NewLabel( this );
    if( nl->exec() != QDialog::Accepted ) {
        delete nl;
        return false;
    }

    d->setId( nl->labelId() );
    clearLabel();
    cv->setDefinition( d );
    delete nl;

    filename = "";
    setCaption( filename, false );
    enableActions();

    return true;
}

void LabelEditor::setupActions()
{
    KAction* newAct = KStdAction::openNew( this, SLOT(startEditor()), actionCollection() );
    KAction* loadAct = KStdAction::open( this, SLOT(startLoadEditor()), actionCollection() );
    KAction* quitAct = KStdAction::quit(kapp, SLOT(quit()), actionCollection());
    KAction* closeAct = KStdAction::close( this, SLOT( close() ), actionCollection(), "close" );
    closeLabelAct = new KAction( i18n("Close &Label" ), 0, 0, this, SLOT( closeLabel() ), actionCollection() );

    recentAct = new KRecentFilesAction( i18n("&Recent Files"), 0, this, SLOT( loadRecentEditor( const KURL& ) ) );

    KAction* importPrintFileAct = KStdAction::print( this, SLOT(batchPrint()), actionCollection() );
    importPrintFileAct->setText(i18n("&Import and Print Batch File..."));

    saveAct = KStdAction::save( this, SLOT( save() ), actionCollection(), "save" );
    saveAsAct = KStdAction::saveAs( this, SLOT( saveas() ), actionCollection(), "saveas" );
    exportAct = new KAction( i18n("...as &Image"), 0, 0, this, SLOT(exportLabel() ), actionCollection() );
    exportZPLAct = new KAction( i18n("...as &IPL or ZPL"), 0, 0, this, SLOT(exportZPL() ), actionCollection() );
    descriptionAct = new KAction( i18n("&Change description..."), 0, 0, this, SLOT(changeDes()), actionCollection() );
    deleteAct = new KAction( i18n("&Delete Object"), QIconSet( BarIcon("editdelete") ), Key_Delete, cv, SLOT( deleteCurrent() ), actionCollection() );
    printAct = KStdAction::print( this, SLOT( print() ), actionCollection(), "print" );
    changeSizeAct = new KAction( i18n("&Change Label..."), 0, 0, this, SLOT( changeSize() ), actionCollection() );
    barcodeAct = new KAction( i18n("Insert &Barcode"), QIconSet( BarIcon("barcode") ), 0, this, SLOT( insertBarcode() ), actionCollection() );
    barcodeAct->setEnabled( BarCode::haveBarcode() );
    
    pictureAct = new KAction( i18n("Insert &Picture"), QIconSet( BarIcon("inline_image") ), 0, this, SLOT( insertPicture() ), actionCollection() );
    textAct = new KAction( i18n("Insert &Text"), QIconSet( BarIcon("text") ), 0, this, SLOT( insertText() ), actionCollection() );
    lineAct = new KAction( i18n("Insert &Line"), QIconSet( BarIcon("kbarcodelinetool") ), 0, this, SLOT( insertLine() ), actionCollection() );
    rectAct = new KAction( i18n("Insert &Rectangle"), QIconSet( BarIcon("kbarcoderect") ), 0, this, SLOT( insertRect() ), actionCollection() );
    circleAct = new KAction( i18n("Insert &Ellipse"), QIconSet( BarIcon("kbarcodeellipse") ), 0, this, SLOT( insertCircle() ), actionCollection() );
    addressAct = new KAction( i18n("Insert &Address..."), QIconSet( BarIcon("kaddressbook") ), 0, this, SLOT( insertAddress() ), actionCollection() );
    spellAct = KStdAction::spelling( this, SLOT(spellCheck()), actionCollection(), "spell" );
    gridAct = new KToggleAction( i18n("&Grid"), QIconSet( BarIcon("kbarcodegrid") ), 0, this, SLOT( toggleGrid() ), actionCollection() );
    previewAct = new KAction( i18n("&Preview..."), 0, 0, this, SLOT( preview() ), actionCollection() );
    sep = new KActionSeparator( this );
    cutAct = KStdAction::cut( this, SLOT( cut() ), actionCollection(), "cut" );
    copyAct = KStdAction::copy( this, SLOT( copy() ), actionCollection(), "copy" );
    pasteAct = KStdAction::paste( this, SLOT( paste() ), actionCollection(), "paste" );
    undoAct = (KAction*)actionCollection()->action("edit_undo");
    redoAct = (KAction*)actionCollection()->action("edit_redo");
    addressBookAct = new KAction( i18n("Address&book"), QIconSet( BarIcon("kaddressbook") ), 0, this, SLOT( launchAddressBook() ), actionCollection() );
    KAction* singleBarcodeAct = new KAction(i18n("&Create Single Barcode..."), "",
                                0, this, SLOT(startBarcodeGen()),
                                actionCollection(), "create" );
    singleBarcodeAct->setEnabled( BarCode::haveBarcode() );

    newAct->plug( toolBar() );
    loadAct->plug( toolBar() );
    saveAct->plug( toolBar() );
    printAct->plug( toolBar() );
    sep->plug( toolBar() );
    undoAct->plug( toolBar() );
    redoAct->plug( toolBar() );
    cutAct->plug( toolBar() );
    copyAct->plug( toolBar() );
    pasteAct->plug( toolBar() );

    menuFields = new KActionMenu( i18n("Insert &Database Field"), QIconSet( BarIcon("text") ), this, "menuFields" );
    menuFields->setDelayed( false );

    KPopupMenu* mnuBasic = new KPopupMenu( this );
    mnuBasic->insertItem( i18n("Article Number"), ID_ITEM_ARTICLE_NO );
    mnuBasic->insertItem( i18n("Article Description"), ID_ITEM_ARTICLE_DESCRIPTION );
    mnuBasic->insertSeparator();
    mnuBasic->insertItem( i18n("Field 0"), ID_ITEM_FIELD_0 );
    mnuBasic->insertItem( i18n("Field 1"), ID_ITEM_FIELD_1 );
    mnuBasic->insertItem( i18n("Field 2"), ID_ITEM_FIELD_2 );
    mnuBasic->insertItem( i18n("Field 3"), ID_ITEM_FIELD_3 );
    mnuBasic->insertItem( i18n("Field 4"), ID_ITEM_FIELD_4 );
    mnuBasic->insertItem( i18n("Field 5"), ID_ITEM_FIELD_5 );
    mnuBasic->insertItem( i18n("Field 6"), ID_ITEM_FIELD_6 );
    mnuBasic->insertItem( i18n("Field 7"), ID_ITEM_FIELD_7 );
    mnuBasic->insertItem( i18n("Field 8"), ID_ITEM_FIELD_8 );
    mnuBasic->insertItem( i18n("Field 9"), ID_ITEM_FIELD_9 );
    
    KPopupMenu* mnuCustomer = new KPopupMenu( this );
    mnuCustomer->insertItem( i18n("Customer Name"), ID_ITEM_CUSTOMER_NAME );
    mnuCustomer->insertItem( i18n("Customer Number"), ID_ITEM_CUSTOMER_NO );
    
    KPopupMenu* mnuCustomerText = new KPopupMenu( this );
    mnuCustomerText->insertItem( i18n("Line 0"), ID_ITEM_LINE_0 );
    mnuCustomerText->insertItem( i18n("Line 1"), ID_ITEM_LINE_1 );
    mnuCustomerText->insertItem( i18n("Line 2"), ID_ITEM_LINE_2 );
    mnuCustomerText->insertItem( i18n("Line 3"), ID_ITEM_LINE_3 );
    mnuCustomerText->insertItem( i18n("Line 4"), ID_ITEM_LINE_4 );
    mnuCustomerText->insertItem( i18n("Line 5"), ID_ITEM_LINE_5 );
    mnuCustomerText->insertItem( i18n("Line 6"), ID_ITEM_LINE_6 );
    mnuCustomerText->insertItem( i18n("Line 7"), ID_ITEM_LINE_7 );
    mnuCustomerText->insertItem( i18n("Line 8"), ID_ITEM_LINE_8 );
    mnuCustomerText->insertItem( i18n("Line 9"), ID_ITEM_LINE_9 );

    menuFields->popupMenu()->insertItem( i18n("barcode_basic"), mnuBasic );
    menuFields->popupMenu()->insertItem( i18n("customer"), mnuCustomer );
    menuFields->popupMenu()->insertItem( i18n("customer_text"), mnuCustomerText );
    menuFields->popupMenu()->insertSeparator();
    menuFields->popupMenu()->insertItem( i18n("Barcode Number"), ID_ITEM_BARCODE_NO );
    menuFields->popupMenu()->insertItem( i18n("Encoding Type"), ID_ITEM_ENCODING_TYPE );
    menuFields->popupMenu()->insertSeparator();
    menuFields->popupMenu()->insertItem( i18n("Group"), ID_ITEM_GROUP );
    menuFields->popupMenu()->insertItem( i18n("Filename"), ID_ITEM_FILENAME );
    menuFields->popupMenu()->insertItem( i18n("Date"), ID_ITEM_DATE );
    menuFields->popupMenu()->insertItem( i18n("Resolution"), ID_RESOLUTION );
    
    connect( menuFields->popupMenu(), SIGNAL(activated(int)), this, SLOT(insertDataText(int)));
    connect( mnuBasic, SIGNAL(activated(int)), this, SLOT(insertDataText(int)));
    connect( mnuCustomer, SIGNAL(activated(int)), this, SLOT(insertDataText(int)));
    connect( mnuCustomerText, SIGNAL(activated(int)), this, SLOT(insertDataText(int)));

    tools = new KToolBar( this, this->leftDock(), true, "tools" );
   
    barcodeAct->plug( tools );
    pictureAct->plug( tools );
    textAct->plug( tools );
    menuFields->plug( tools );
    lineAct->plug( tools );
    rectAct->plug( tools );
    circleAct->plug( tools );
    addressAct->plug( tools );
    (new KActionSeparator( this ))->plug( tools );
    deleteAct->plug( tools );
//    spellAct->plug( tools );  // KDE 3.2
    gridAct->plug( tools );
            
    DSMainWindow::setupActions();
    connect( recentAct, SIGNAL( urlSelected( const KURL& ) ), this, SLOT( startLoadRecentEditor( const KURL& ) ) );

    KPopupMenu* fileMenu = new KPopupMenu( this );
    KPopupMenu* editMenu = new KPopupMenu( this );
    KPopupMenu* viewMenu = new KPopupMenu( this );
    KPopupMenu* insMenu = new KPopupMenu( this );
    KPopupMenu* toolMenu = new KPopupMenu( this );
    KPopupMenu* barMenu = new KPopupMenu( this );
    exportMenu = new KPopupMenu( this );
    
    menuBar()->removeItemAt( 0 );
    menuBar()->insertItem( i18n("&File"), fileMenu, -1, 0 );
    menuBar()->insertItem( i18n("&Edit"), editMenu, -1, 1 );
    menuBar()->insertItem( i18n("&Insert"), insMenu, -1, 2 );
    menuBar()->insertItem( i18n("&View"), viewMenu, -1, 3 );
    menuBar()->insertItem( i18n("T&ools"), toolMenu, -1, 4 );
    menuBar()->insertItem( i18n("&Barcode"), barMenu, -1, 5 );

    // Menubar
    newAct->plug( fileMenu );
    loadAct->plug( fileMenu );
    recentAct->plug( fileMenu );
    saveAct->plug( fileMenu );
    saveAsAct->plug( fileMenu );
    fileMenu->insertItem( QIconSet( BarIcon("fileexport") ), i18n("&Export..."), exportMenu );
    sep->plug( fileMenu );
    printAct->plug( fileMenu );
    sep->plug( fileMenu );
    closeLabelAct->plug( fileMenu );
    closeAct->plug( fileMenu );
    quitAct->plug( fileMenu );

    exportAct->plug( exportMenu );
    exportZPLAct->plug( exportMenu );

    undoAct->plug( editMenu );
    redoAct->plug( editMenu );
    sep->plug( editMenu );
    cutAct->plug( editMenu );
    copyAct->plug( editMenu );
    pasteAct->plug( editMenu );
    sep->plug( editMenu );
    descriptionAct->plug( editMenu );
    changeSizeAct->plug( editMenu );
    sep->plug( editMenu );
    deleteAct->plug( editMenu );

    barcodeAct->plug( insMenu );
    pictureAct->plug( insMenu );
    textAct->plug( insMenu );
    menuFields->plug( insMenu );
    lineAct->plug( insMenu );
    rectAct->plug( insMenu );
    circleAct->plug( insMenu );
    addressAct->plug( insMenu );

//    spellAct->plug( toolMenu ); // KDE 3.2
    toolMenu->insertSeparator();
    addressBookAct->plug( toolMenu );
    
    gridAct->plug( viewMenu );
    previewAct->plug( viewMenu );
    
    singleBarcodeAct->plug( barMenu );
    importPrintFileAct->plug( barMenu );

    enableActions();
}

void LabelEditor::insertBarcode()
{
    NewBarcodeCommand* bc = new NewBarcodeCommand( cv );
    bc->execute();

    CanvasBarcode* bcode = static_cast<CanvasBarcode*>(bc->createdItem());
    if( !bcode )
        return;

    setupBarcode( bcode );

    if( bcode->getPixmap().isNull() ) {
        bc->unexecute();
        delete bc;
    } else
        history->addCommand( bc, false );
}

void LabelEditor::insertPicture()
{
    QString filename = KFileDialog::getOpenFileName( QString::null, KImageIO::pattern( KImageIO::Reading ), this );
    if( !filename.isEmpty() ) {
        QImage* img = new QImage( filename );
        if( img->isNull() ) {
            KMessageBox::information( this, i18n("Unsupported image format") );
            delete img;
            return;
        }
        insertPicture( img );
    }
}

void LabelEditor::insertPicture( QImage* img )
{
    NewPictureCommand* pc = new NewPictureCommand( img, cv );
    history->addCommand( pc, true );
}

void LabelEditor::insertText()
{
    insertText( "<nobr>Some Text</nobr>" );
}

void LabelEditor::insertDataText( int index )
{
    QString data;
    switch( index ) {
        case ID_ITEM_GROUP:
            data = "[group]";
            break;
        case ID_ITEM_FILENAME:
            data = "[filename]";
            break;
        case ID_ITEM_DATE:
            data = "[date]";
            break;
        case ID_ITEM_ARTICLE_NO:
            data = "[article_no]";
            break;
        case ID_ITEM_ARTICLE_DESCRIPTION:
            data = "[article_desc]";
            break;
        case ID_ITEM_BARCODE_NO:
            data = "[barcode_no]";
            break;
        case ID_ITEM_FIELD_0:
            data = "[field0]";
            break;
        case ID_ITEM_FIELD_1:
            data = "[field1]";
            break;
        case ID_ITEM_FIELD_2:
            data = "[field2]";
            break;
        case ID_ITEM_FIELD_3:
            data = "[field3]";
            break;
        case ID_ITEM_FIELD_4:
            data = "[field4]";
            break;
        case ID_ITEM_FIELD_5:
            data = "[field5]";
            break;
        case ID_ITEM_FIELD_6:
            data = "[field6]";
            break;
        case ID_ITEM_FIELD_7:
            data = "[field7]";
            break;
        case ID_ITEM_FIELD_8:
            data = "[field8]";
            break;
        case ID_ITEM_FIELD_9:
            data = "[field9]";
            break;
        case ID_ITEM_CUSTOMER_NAME:
            data = "[customer_name]";
            break;
        case ID_ITEM_CUSTOMER_NO:
            data = "[customer_no]";
            break;
        case ID_ITEM_LINE_0:
            data = "[line0]";
            break;
        case ID_ITEM_LINE_1:
            data = "[line1]";
            break;
        case ID_ITEM_LINE_2:
            data = "[line2]";
            break;
        case ID_ITEM_LINE_3:
            data = "[line3]";
            break;
        case ID_ITEM_LINE_4:
            data = "[line4]";
            break;
        case ID_ITEM_LINE_5:
            data = "[line5]";
            break;
        case ID_ITEM_LINE_6:
            data = "[line6]";
            break;
        case ID_ITEM_LINE_7:
            data = "[line7]";
            break;
        case ID_ITEM_LINE_8:
            data = "[line8]";
             break;
        case ID_ITEM_LINE_9:
            data = "[line9]";
            break;
        case ID_ITEM_ENCODING_TYPE:
            data = "[encoding_type]";
            break;
        case ID_RESOLUTION:
            data = "[resolution]";
            break;
        default:
            break;
    };

    if( !data.isEmpty() )
        insertText( data );
}

void LabelEditor::insertText( QString caption )
{
    NewTextCommand* tc = new NewTextCommand( caption, cv, this );
    history->addCommand( tc, true );
}

void LabelEditor::insertRect()
{
    NewRectCommand* rc = new NewRectCommand( cv );
    history->addCommand( rc, true );
}

void LabelEditor::insertCircle()
{
    NewRectCommand* rc = new NewRectCommand( cv, true );
    history->addCommand( rc, true );    
}

void LabelEditor::insertLine()
{
    NewLineCommand* lc = new NewLineCommand( cv );
    history->addCommand( lc, true );
}

void LabelEditor::insertAddress()
{
    Addressee add = AddresseeDialog::getAddressee( this );
    if( !add.isEmpty() ) {
#if KDE_VERSION < 0x030200
        insertText( "<p>REQUIRES KDE 3.2 AT THE MOMENT</p>" );
#else
        insertText( "<p>" + add.address( 1 ).formattedAddress() + "</p>" );
#endif
    }
}

void LabelEditor::exportLabel()
{
    KFileDialog fd( ":save_image", KImageIO::pattern( KImageIO::Writing ), this, "fd", true );
    fd.setMode( KFile::File );
    fd.setOperationMode( KFileDialog::Saving );    
    if( fd.exec() == QDialog::Accepted ) {
        KPrinter printer( true, (enum QPrinter::PrinterMode)PrinterSettings::getInstance()->getQuality() );

        QString path = fd.selectedURL().path();
        QString filter = fd.currentFilter();
        QString extension = filter.right( filter.length() - filter.findRev(".") -1 ).upper();
        path = path + "." + extension.lower();

        QPixmap pixmap( c->width(), c->height() );
        QPainter p( &pixmap );
        c->drawArea( c->rect(), &p );
        
        if(!pixmap.save( path, extension, 0 ))
            KMessageBox::error( this, i18n("An error occurred during saving the image") );
    }
}

void LabelEditor::exportZPL()
{
    KMessageBox::information( this, "The export code is still in development and its output might not be correct!" );

    KFileDialog fd( QString::null, "*.zpl|Zebra Printer Language (*.zpl)\n*.ipl|Intermec Printer Language (*.ipl)", this, "fd", true );
    if( fd.exec() != QDialog::Accepted )
        return;

    QFile file( fd.selectedFile() );
    if( !file.open( IO_WriteOnly ) )
        return;

    QTextStream t( &file );
    ExportUtils* z;
    if( fd.currentFilter() == "*.zpl" )
        z = new ZPLUtils( &t, cv );
    else if( fd.currentFilter() == "*.ipl" )
        z = new IPLUtils( &t, cv );
    else {
        qDebug("invalid filter");
        return;
    }
    
    QCanvasItemList list = c->allItems();
    for( unsigned int i = 0; i < list.count(); i++ )
        if( list[i]->rtti() == TEXT_RTTI ) {
            CanvasText* text = static_cast<CanvasText*>(list[i]);

            QString tt = text->text();
            z->setTextField( (int)text->x() - cv->getTranslation().x(), (int)text->y() - cv->getTranslation().y(), text->text() );
        } else if( list[i]->rtti() == BARCODE_RTTI ) {
            CanvasBarcode* barcode = static_cast<CanvasBarcode*>(list[i]);
            z->setBarcode( (int)barcode->x() - cv->getTranslation().x(), (int)barcode->y() - cv->getTranslation().y(),
                                      barcode->pixmap().height(),
                                      barcode->data().value, barcode->data().type );
            
        } else if( list[i]->rtti() == RECT_RTTI ) {
            DrawingRect* rect = static_cast<DrawingRect*>(list[i]);
            QSize size( rect->width(), rect->height() );
            z->setRect( (int)rect->x() - cv->getTranslation().x(), (int)rect->y() - cv->getTranslation().y(), size, rect->isCircle(), rect->borderWidth() );
        } else if( list[i]->rtti() == PIC_RTTI ) {
            PictureRectangle* rect = static_cast<PictureRectangle*>(list[i]);
            QImage img = rect->getTransformedPixmap().convertToImage();
            z->setImage( (int)rect->x() - cv->getTranslation().x(), (int)rect->y() - cv->getTranslation().y(), &img );
        }
        

    z->close();
    file.close();
}

void LabelEditor::changeDes()
{
    QString tmp = QInputDialog::getText( i18n("Label Description"),
            i18n("Please enter a description:"), QLineEdit::Normal, description );
    if( !tmp.isEmpty() )
        description = tmp;
}

void LabelEditor::changeSize()
{
    NewLabel* nl = new NewLabel( this, "nl", true, true );
    nl->setLabelId( d->getId() );
    if( nl->exec() == QDialog::Rejected )
        return;

    d->setId( nl->labelId() );
    cv->setDefinition( d );
    
    updateInfo();
    enableActions();
    c->update();
    cv->repaint();  
}

void LabelEditor::updateInfo()
{
    measurements* measure = d->getMeasurements();
    statusBar()->changeItem( i18n("Size: ") + QString("%1mm x %2mm").arg(
                 measure->width ).arg( measure->height  ), STATUS_ID_SIZE );
    statusBar()->changeItem( i18n("Label Template: ") + d->getProducer() + " - " + d->getType(), STATUS_ID_TEMPLATE );
}

void LabelEditor::doubleClickedItem( QCanvasItem* item )
{
    if( item->rtti() == TEXT_RTTI ) {
        changeText( item );
    } else if( item->rtti() == BARCODE_RTTI ) {
        setupBarcode( (CanvasBarcode*)item );
    } else if( item->rtti() == PIC_RTTI ) {
        changePictureSettings( item );
    } else if( item->rtti() == RECT_RTTI ) {
        changeRectSettings( item );
    } else if( item->rtti() == LINE_RTTI ) {
        changeLineSettings( item );
    }

    c->update();
    cv->repaintContents();
}

void LabelEditor::changeText( QCanvasItem* item )
{
    if( !item ){
        item = cv->getActive();
        if( !item )
            return;
    }

    CanvasText* mytext = (CanvasText*)item;

    bool ok = false;
    QString text = MultiLineEditDlg::getText( mytext->text(), &ok, this );
    if( ok ) {
        TextChangeCommand* tc = new TextChangeCommand( mytext, text );
        history->addCommand( tc, true );
    }
}

void LabelEditor::changePictureSettings( QCanvasItem* item )
{                                   
    if( !item ){
        item = cv->getActive();
        if( !item )
            return;
    }

    MyCanvasRectangle* r = (MyCanvasRectangle*)item;
    PictureSettings* ps = new PictureSettings( this );
    ps->spinRotation->setValue( (int)r->getRotation() );

    if( ps->exec() == QDialog::Accepted ) {
        PictureCommand* pc = new PictureCommand( (double)ps->spinRotation->value(), r->getRotation(), r );

        history->addCommand( pc, true );
    }
}

void LabelEditor::changeRectSettings( QCanvasItem* item )
{
    if( !item ){
        item = cv->getActive();
        if( !item )
            return;
    }

    DrawingRect* r = (DrawingRect*)item;
    
    RectSettingsDlg* rsd = new RectSettingsDlg( this );
    rsd->setFillColor( r->color() );
    rsd->setBorderColor( r->borderColor() );
    rsd->setBorderWidth( r->borderWidth() );
    rsd->setPenStyle( r->borderStyle() );
    if( rsd->exec() == QDialog::Accepted ) {
        RectCommand* rc = new RectCommand( rsd->fillColor(), rsd->borderColor(),
                              rsd->borderWidth(), rsd->penStyle(), r );

        history->addCommand( rc, true );        
    }
}

void LabelEditor::changeLineSettings( QCanvasItem* item )
{
    if( !item ){
        item = cv->getActive();
        if( !item || item->rtti() != LINE_RTTI )
            return;
    }

    MyCanvasLine* l = (MyCanvasLine*)item;

    LineSettingsDlg* lsd = new LineSettingsDlg( this ); // what a cool name for a variable :-D
    lsd->setPen( l->pen() );
    if( lsd->exec() == QDialog::Accepted ) {
        LineCommand* lc = new LineCommand( lsd->pen(), l );
        history->addCommand( lc, true );
    }
}

void LabelEditor::doubleClickedCurrent()
{
    if( cv->getActive() )
        doubleClickedItem( cv->getActive() );
}

void LabelEditor::showContextMenu( QCanvasItem* item, QPoint pos )
{
    KPopupMenu* menu = new KPopupMenu( this );

    KPopupMenu* orderMenu = new KPopupMenu( menu );
    orderMenu->insertItem( i18n("&On Top"), this, SLOT( onTopCurrent() ) );
    orderMenu->insertItem( i18n("&Raise"), this, SLOT( raiseCurrent() ) );
    orderMenu->insertItem( i18n("&Lower"), this, SLOT( lowerCurrent() ) );
    orderMenu->insertItem( i18n("&To Background"), this, SLOT( backCurrent() ) );    
    menu->insertItem( i18n("&Order"), orderMenu );
    
    KPopupMenu* centerMenu = new KPopupMenu( menu );
    centerMenu->insertItem( i18n("Center &Horizontally"), this, SLOT( centerHorizontal() ) );
    centerMenu->insertItem( i18n("Center &Vertically"), this, SLOT( centerVertical() ) );
    menu->insertItem( i18n("&Center"), centerMenu );

    menu->insertSeparator();
    menu->insertItem( SmallIcon("editdelete"), i18n("&Delete"), cv, SLOT( deleteCurrent() ) );
    menu->insertItem( i18n("&Move by Value"), this, SLOT( moveByValue() ) );
//    menu->insertItem( SmallIcon("lock"), i18n("&Lock Position"), this, SLOT( lockItem() ), 0, ID_LOCK_ITEM )
    menu->insertSeparator();

    if( item->rtti() == TEXT_RTTI ){
        menu->insertItem( i18n("&Edit"), this, SLOT( changeText() ) );
        menu->insertSeparator();
        menu->insertItem( i18n("&Properties"), this, SLOT( changePictureSettings() ) );
    } else {
        menu->insertItem( i18n("&Properties"), this, SLOT( doubleClickedCurrent() ) );
    }
    
    menu->popup( pos );
}

void LabelEditor::setupBarcode( CanvasBarcode* bcode )
{
    BarcodeSettingsDlg* b = new BarcodeSettingsDlg( this );
    b->barcode->setData( bcode->data() );
    for( int i = 0; i < b->comboComplex->count(); i++ )
        if( b->comboComplex->text( i ) == bcode->data().xml.caption )
            b->comboComplex->setCurrentItem( i );

    b->changedCombo();
    
    if( b->exec() == QDialog::Accepted ) {
        barcodeData* d = new barcodeData;
        *d = b->barcode->getData();
                
        if( b->comboComplex->currentItem() != b->comboComplex->count() - 1 ) {
            d->type = getTypeFromCaption( b->comboComplex->currentText() );
            d->value = BarCode::getMaxLength( d->type );
        }
        d->xml.caption = b->comboComplex->currentText();
        
        BarcodeCommand* bc = new BarcodeCommand( bcode, d );
        history->addCommand( bc, true );

        if( !bcode->validBarcode() )
            setupBarcode( bcode );
    }
}

void LabelEditor::print()
{
    PrintLabelDlg pld( this, "pld" );
    if( pld.exec() != QDialog::Accepted )
        return;

    int quantity = pld.labels();
    int move = pld.position();
    PrinterSettings::getInstance()->getData()->border = pld.border();
    move--;

    bool border = pld.border();

    KPrinter* printer = PrinterSettings::getInstance()->setupPrinter( this );
    if( !printer )
        return;

    QString oldname = filename;
    KTempFile tmp( QString::null, ".kbarcode" );

    save( tmp.name() );
    filename = oldname;
    setCaption( filename, isChanged() );
    tmp.file()->close();

    Label label( d, tmp.file(), tmp.name(), printer, QString::null, QString::null, QString::null );
    tmp.unlink();

    measurements* printmeasure = d->getMeasurements( printer );
               
    int c_h = 0, c_w = 0;
    double curw = printmeasure->gap_left, curh = printmeasure->gap_top;

    QProgressDialog* progress = new QProgressDialog( i18n("Printing..."), i18n("&Cancel"), quantity,this );
    progress->setMinimumDuration( 0 );
    progress->setProgress( 0 );
    QPainter painter( printer );
    
    QPaintDeviceMetrics pdm( painter.device() );
    int pageh = pdm.height();   // in pixel

    for( int i = 0; i < quantity + move; i++ ) {

        if( c_h >= printmeasure->num_h ) {
            c_h = 0;
            curw = printmeasure->gap_left;
            curh += printmeasure->height + printmeasure->gap_v;
            c_w++;
        }

        if( curh + printmeasure->height > pageh || c_w == printmeasure->num_v ) {
            label.setPage( label.page() + 1 );
            printer->newPage();
                        
            curh = printmeasure->gap_top;
            curw = printmeasure->gap_left;
            c_w = 0;
        }

        if( i >= move ) {
            painter.setClipRect( (int)curw, (int)curh, (int)printmeasure->width, (int)printmeasure->height );
            label.setIndex( i  - move );
            painter.drawPicture( QPoint( (int)curw, (int)curh ), *label.picture() );
            painter.setClipping( false );
            if( border ) {
                // Draw borders around labels
                painter.setPen( QPen( Qt::black, 1 ) );
                painter.drawRect( (int)curw, (int)curh, (int)printmeasure->width, (int)printmeasure->height );
            }
        }

        curw += printmeasure->width + printmeasure->gap_h;
        c_h++;

        kapp->processEvents( 0 );
        progress->setProgress( i );
        if( progress->wasCancelled() ) {                        
            printer->abort();
            break;
        }
    }

    painter.end();
    delete printmeasure;
    delete progress;
    delete printer;
}

void LabelEditor::spellCheck()
{
    KMacroCommand* sc = new KMacroCommand( i18n("Spellchecking") );
    QCanvasItemList list = c->allItems();
    for( unsigned int i = 0; i < list.count(); i++ )
        if( list[i]->rtti() == TEXT_RTTI ) {
            CanvasText* mytext = (CanvasText*)list[i];
            QString text = mytext->text();
            bool nocheck = false;
//            for( int z = 0; z < comboText->count(); z++ )
//                if( text == "[" + comboText->text(z) + "]" ) {
//                    nocheck = true;
//                    break;
//                }

            if( !nocheck ) {                                        
                QString textbefore = text;
                KSpell::modalCheck( text );
                if( text != textbefore ) {
                    TextChangeCommand* tc = new TextChangeCommand( mytext, text );
                    tc->execute();
                    sc->addCommand( tc );
                }
            }
        }

    history->addCommand( sc, false );
}

void LabelEditor::centerHorizontal()
{
    if( !cv->getActive() )
        return;

    MoveCommand* mv = new MoveCommand(
        int(((c->width() - cv->getActive()->boundingRect().width()) / 2) - cv->getActive()->x()),
        0, cv->getActive() );
    history->addCommand( mv, true );
}

void LabelEditor::centerVertical()
{
    if( !cv->getActive() )
        return;
    MoveCommand* mv = new MoveCommand( 0,
        int(((c->height() - cv->getActive()->boundingRect().height()) / 2) - cv->getActive()->y()),
        cv->getActive() );
    history->addCommand( mv, true );
}

void LabelEditor::raiseCurrent()
{
    if( !cv->getActive() )
        return;

    cv->getActive()->setZ( cv->getActive()->z() + 1 );
}

void LabelEditor::lowerCurrent()
{
    if( !cv->getActive() )
        return;

    cv->getActive()->setZ( cv->getActive()->z() - 1 );
}

void LabelEditor::onTopCurrent()
{
    if( !cv->getActive() )
        return;

    int z = 0;    

    QCanvasItemList list = c->allItems();
    for( unsigned int i = 0; i < list.count(); i++ )
        if( list[i]->z() > z )
            z = (int)list[i]->z();

    cv->getActive()->setZ( z + 1 );
}

void LabelEditor::backCurrent()
{
    if( !cv->getActive() )
        return;

    int z = 0;    

    QCanvasItemList list = c->allItems();
    for( unsigned int i = 0; i < list.count(); i++ )
        if( list[i]->z() < z )
            z = (int)list[i]->z();

    cv->getActive()->setZ( z - 1 );
}

const QString LabelEditor::fileName() const
{
    return filename.right( filename.length() - filename.findRev( "/" ) - 1 );
}

void LabelEditor::preview()
{
    QString oldname = filename;
    KTempFile tmp( QString::null, ".kbarcode" );

    save( tmp.name() );
    filename = oldname;    
    setCaption( filename, isChanged() );
                
    PreviewDialog* pd = new PreviewDialog( tmp.file(), d, fileName(), this );
    pd->exec();

    tmp.file()->close();
    tmp.unlink();
}

void LabelEditor::toggleGrid()
{
    c->setGrid( gridAct->isChecked() );
    cv->repaintContents();
}

void LabelEditor::moveByValue()
{
    if( !cv->getActive() )
        return;
        
    PositionDialog* pd = new PositionDialog( this );
    pd->spinX->setValue( (int)pixelToMm( cv->getActive()->x() - cv->getTranslation().x(), cv, DpiX ) );
    pd->spinY->setValue( (int)pixelToMm( cv->getActive()->y() - cv->getTranslation().y(), cv, DpiY ) );

    if( pd->exec() == QDialog::Accepted ) {
        MoveCommand* mv = new MoveCommand(
            int( mmToPixel( pd->spinX->value(), cv, DpiX ) + cv->getTranslation().x() - cv->getActive()->x()),
            int( mmToPixel( pd->spinY->value(), cv, DpiY ) + cv->getTranslation().y() - cv->getActive()->y() ),
            cv->getActive() );
        history->addCommand( mv, true );
    }
}

void LabelEditor::cut()
{
    copy();
    cv->deleteCurrent();
}

void LabelEditor::copy()
{
    QCanvasItem* item = cv->getActive();
    if( !item )
        return;

    if( item->rtti() == TEXT_RTTI ) {
        DSTextDrag* drag = new DSTextDrag();
        QString text = (static_cast<CanvasText*>(item))->text();
        drag->setKBarcode( text.local8Bit() );
        drag->setPlain( text.replace( QRegExp("<[^>]*>"), "" ) );
#if QT_VERSION >= 0x030100
        kapp->clipboard()->setData( drag, QClipboard::Clipboard );
#else
        kapp->clipboard()->setData( drag );
#endif
    } else if( item->rtti() == PIC_RTTI ){
        PictureRectangle* r = static_cast<PictureRectangle*>(item);
#if QT_VERSION >= 0x030100
        kapp->clipboard()->setData( new QImageDrag( r->getImage() ), QClipboard::Clipboard );
#else
        kapp->clipboard()->setData( new QImageDrag( r->getImage() ) );
#endif
    } else if( item->rtti() == RECT_RTTI ) {
        RectDrag* drag = new RectDrag();
        drag->setRect( static_cast<DrawingRect*>(item) );
#if QT_VERSION >= 0x030100
        kapp->clipboard()->setData( drag, QClipboard::Clipboard );
#else
        kapp->clipboard()->setData( drag );
#endif
    } else if( item->rtti() == LINE_RTTI ) {
        LineDrag* drag = new LineDrag();
        drag->setLine( static_cast<MyCanvasLine*>(item) );
#if QT_VERSION >= 0x030100
        kapp->clipboard()->setData( drag, QClipboard::Clipboard );
#else
        kapp->clipboard()->setData( drag );
#endif
    } else if( item->rtti() == BARCODE_RTTI ) {
        BarcodeDrag* drag = new BarcodeDrag();
        barcodeData data =  (static_cast<CanvasBarcode*>(item))->data();
        drag->setBarcode( &data );
#if QT_VERSION >= 0x030100
        kapp->clipboard()->setData( drag, QClipboard::Clipboard );
#else
        kapp->clipboard()->setData( drag );
#endif
    }
}

void LabelEditor::paste()
{
    QMimeSource* data = QApplication::clipboard()->data();
    if ( data->provides( DSTextDrag::selectionMimeType() ) ) {
        // KBarcode formated text
        QByteArray arr = data->encodedData( DSTextDrag::selectionMimeType() );
        if ( arr.size() )
            insertText( QString( arr ) );
    } else if( BarcodeDrag::canDecode( data, "application/x-kbarcode-barcode" ) ) {
        NewBarcodeCommand* nbc = new NewBarcodeCommand( cv );
        nbc->execute();
        if( BarcodeDrag::decode( data, static_cast<CanvasBarcode*>(nbc->createdItem()) ) ) {
            history->addCommand( nbc, false );
        } else {
            delete nbc->createdItem();
            delete nbc;
        }
    } else if( QImageDrag::canDecode( data ) ) {
        QImage img;
        if( QImageDrag::decode( data, img  ) ) 
            insertPicture( &img );
    } else if( RectDrag::canDecode( data ) ) {
        NewRectCommand* nrc = new NewRectCommand( cv );
        nrc->execute();
        if( RectDrag::decode( data, static_cast<DrawingRect*>(nrc->createdItem()) ) ) {
            history->addCommand( nrc, false );
        } else {
            delete nrc->createdItem();
            delete nrc;
        }
    } else if( LineDrag::canDecode( data ) ) {
        NewLineCommand* nlc = new NewLineCommand( cv );
        nlc->execute();
        if( LineDrag::decode( data, static_cast<MyCanvasLine*>(nlc->createdItem()) ) ) {
            history->addCommand( nlc, false );
        } else {
            delete nlc->createdItem();
            delete nlc;
        }
    } else {
        QString text;
#if QT_VERSION >= 0x030100
        text = kapp->clipboard()->text( QClipboard::Clipboard );
#else
        text = kapp->clipboard()->text();
#endif
        // support for pasting multi line texts
        if( !text.isEmpty() )
#if QT_VERSION >= 0x030100
            insertText( text.replace( QChar(0x0a) ,"<br>" ) );
#else
            insertText( text.replace( QRegExp( QChar(0x0a) ), "<br>" ) );
#endif
    }
}

void LabelEditor::startEditor()
{
    if( isChanged() ) {
        LabelEditor* lb = new LabelEditor();
        lb->startupDlg( CreateNewLabel, "" );
    } else
        newLabel();
}

void LabelEditor::startBarcodeGen()
{
    new BarCodeDialog();
}

void LabelEditor::startLoadRecentEditor( const KURL& url )
{
    if( !QFile::exists( url.path() ) ) {
        KMessageBox::information( this, i18n("The file %1 does not exist.").arg( url.path() ) );
        recentAct->removeURL( url );
        return;
    }
    
    if( isChanged() )
        new LabelEditor( 0, url.path() );
    else
        load( url.path() );
}

void LabelEditor::startLoadEditor()
{
    if( isChanged() ) {
        LabelEditor* lb = new LabelEditor();
        lb->startupDlg( LoadLabel, "" );
    } else
        load();
}

void LabelEditor::batchPrint()
{
    new LabelPrinter( "" );
}

void LabelEditor::closeEvent( QCloseEvent* e )
{
    if( !isChanged() ) {
        saveConfig();
        e->accept();
        delete this;
        return;
    }

    int m = KMessageBox::warningYesNoCancel( this,
        i18n("<qt>The document has been modified.<br><br>Do you want to save it ?</qt>") );

    if( m == KMessageBox::Cancel )
        e->ignore();
    else if( m == KMessageBox::No ) {
        saveConfig();
        e->accept();
        delete this;
    } else if( m == KMessageBox::Yes ) {
        if( save() ) {
            saveConfig();
            e->accept();
            delete this;
        }
    }
}

bool LabelEditor::isChanged()
{
    if( !c->width() && !c->height() )
        return false;
        
    if( m_edited )
        return true;

    return false;
}

bool LabelEditor::startupDlg( int mode, QString f )
{
    if( mode == CreateNewLabel ) {
        if(!newLabel()) {
            close();
            return false;
        }
    } else {
        if(!load(f)) {
            close();
            return false;
        }
    }

    return true;
}

void LabelEditor::closeLabel()
{
    delete d;
    d = new Definition();

    m_edited = false;
       
    clearLabel();
    enableActions();

    cv->setDefinition( d );
                
    filename = "";
    setCaption( filename, false );
}

void LabelEditor::setEdited()
{
    setCaption( filename, true );
    m_edited = true;
}

void LabelEditor::enableActions()
{
    if( d->getId() == -1 ){
        // label closed
        deleteAct->setEnabled( false );
        barcodeAct->setEnabled( false );
        pictureAct->setEnabled( false );
        textAct->setEnabled( false );
        rectAct->setEnabled( false );
        circleAct->setEnabled( false );
        lineAct->setEnabled( false );
        addressAct->setEnabled( false );
        spellAct->setEnabled( false );
        gridAct->setEnabled( false );
        menuFields->setEnabled( false );

        saveAct->setEnabled( false );
        saveAsAct->setEnabled( false );
        printAct->setEnabled( false );
        exportMenu->setEnabled( false );

        previewAct->setEnabled( false );        
        closeLabelAct->setEnabled( false );
        descriptionAct->setEnabled( false );

        cutAct->setEnabled( false );
        copyAct->setEnabled( false );
        pasteAct->setEnabled( false );
    } else {
        deleteAct->setEnabled( true );
        barcodeAct->setEnabled( BarCode::haveBarcode() );
        pictureAct->setEnabled( true );
        textAct->setEnabled( true );
        rectAct->setEnabled( true );
        circleAct->setEnabled( true );
        lineAct->setEnabled( true );
        addressAct->setEnabled( true );        
        spellAct->setEnabled( true );
        gridAct->setEnabled( true );
        menuFields->setEnabled( true );

        saveAct->setEnabled( true );
        saveAsAct->setEnabled( true );
        printAct->setEnabled( true );
        exportMenu->setEnabled( true );
        descriptionAct->setEnabled( true );

        previewAct->setEnabled( true );        
        closeLabelAct->setEnabled( true );

        cutAct->setEnabled( true );
        copyAct->setEnabled( true );
        pasteAct->setEnabled( true );
    }
}

void LabelEditor::launchAddressBook()
{
    KRun::runCommand( "kaddressbook" );
}

/*********************************************************/

BarcodeSettingsDlg::BarcodeSettingsDlg( QWidget* parent,  const char* name )
    : KDialogBase( KDialogBase::Plain, "Barcode",
      KDialogBase::Ok | KDialogBase::Cancel | KDialogBase::Default, KDialogBase::Ok, parent,name)
{
    setCaption( i18n( "Barcode Settings" ) );
    BarcodeSettingsLayout = new QVBoxLayout( plainPage(), 11, 6, "BarcodeSettingsLayout");

    QHBoxLayout* layout = new QHBoxLayout( 0, 6, 6 );
    
    TextLabel1 = new QLabel( plainPage(), "TextLabel1" );
    TextLabel1->setText( i18n( "Barcode Settings:" ) );
            
    comboComplex = new KComboBox( FALSE, plainPage() );
    QSqlQuery* query = new QSqlQuery("select encoding_type from barcode_basic group by encoding_type");
    while( query->next() )
        comboComplex->insertItem( "Main:" + query->value( 0 ).toString() );

    delete query;
    query = new QSqlQuery("select customer_no, encoding_type from customer_text group by customer_no, encoding_type");
    while( query->next() )
        comboComplex->insertItem( query->value( 0 ).toString() + ":" + query->value( 1 ).toString() );

    delete query;
    comboComplex->insertItem( i18n("Static") );

    barcode = new BarcodeWidget( plainPage(), "barcode" );
    
    QSpacerItem* spacer = new QSpacerItem( 0, 0, QSizePolicy::Expanding, QSizePolicy::Minimum );

    connect( this, SIGNAL( defaultClicked() ), this, SLOT( defaults() ) );
    connect( comboComplex, SIGNAL( activated(int) ), this, SLOT(changedCombo() ) );
    
    layout->addWidget( TextLabel1 );
    layout->addWidget( comboComplex );

    BarcodeSettingsLayout->addLayout( layout );
    BarcodeSettingsLayout->addWidget( barcode );    
    BarcodeSettingsLayout->addItem( spacer );
    
    changedCombo();
}

BarcodeSettingsDlg::~BarcodeSettingsDlg()
{
}

void BarcodeSettingsDlg::defaults()
{
    comboComplex->setCurrentItem( 0 );
    barcode->defaults();
}

void BarcodeSettingsDlg::changedCombo()
{
    bool v = false;
    if( comboComplex->currentItem() == comboComplex->count() - 1 )
        v = true;

    barcode->setDataEnabled( v );
    barcode->setStandardEnabled( v );
}

/*********************************************************/

PictureSettings::PictureSettings( QWidget* parent,  const char* name)
    : KDialogBase( KDialogBase::Plain, i18n("Settings"),
      KDialogBase::Ok | KDialogBase::Cancel, KDialogBase::Ok, parent,name)
{
    QVBoxLayout* PSLayout = new QVBoxLayout( plainPage(), 11, 6, "PSLayout");
    QSpacerItem* spacer = new QSpacerItem( 0, 0, QSizePolicy::Minimum, QSizePolicy::Expanding );

    spinRotation = new KIntNumInput( plainPage() );
    spinRotation->setLabel( i18n("Rotation:") );
    spinRotation->setRange( 0, 360, 1, false );
    spinRotation->setValue( 0 );
                       
    PSLayout->addWidget( spinRotation );
    PSLayout->addItem( spacer );   
}

/*********************************************************/

PositionDialog::PositionDialog( QWidget* parent,  const char* name )
    : KDialogBase( KDialogBase::Plain, i18n("Position"),
      KDialogBase::Ok | KDialogBase::Cancel, KDialogBase::Ok, parent,name)
{
    QVBoxLayout* PSLayout = new QVBoxLayout( plainPage(), 11, 6, "PSLayout");
    QSpacerItem* spacer = new QSpacerItem( 0, 0, QSizePolicy::Minimum, QSizePolicy::Expanding );

    spinX = new KIntNumInput( plainPage() );
    spinX->setLabel( i18n("X Position (mm):") );
    spinX->setRange( -10000, 10000, 1, false );

    spinY = new KIntNumInput( plainPage() );
    spinY->setLabel( i18n("Y Position (mm):") );
    spinY->setRange( -10000, 10000, 1, false );

    PSLayout->addWidget( spinX );
    PSLayout->addWidget( spinY );
    PSLayout->addItem( spacer );
}

