{-
    Kaya - My favourite toy language.
    Copyright (C) 2004, 2005 Edwin Brady

    This file is distributed under the terms of the GNU General
    Public Licence. See COPYING for licence.
-}

-- Read/execute/print loop for compiled kaya code.

module REPL(runREPL, REPLRes(..)) where

{- TODO:
 * Global initialisation
 * Catch all - compile the given code in its own executable if we don't know
   how to handle it. (This will be slow)
 * Tab completion of filenames (and maybe modules and functions in scope)

 * Commands for looking up functions by name or part of name, listing
   functions in a given module, querying memory usage, setting
   options, getting documentation for functions.
-}

import Parser
import Inference
import InfGadgets
import Language
import Errors
import Lib
import List

import CForeign
import Ptr
import IO
import System.Console.Readline
import System.Directory

foreign import ccall "repl_load.h" interp_load :: Ptr CChar -> IO ()
foreign import ccall "repl_load.h" interp_close :: IO ()
foreign import ccall "repl_load.h" interp_clear :: IO ()
foreign import ccall "repl_load.h" interp_prepare_call :: IO ()
foreign import ccall "repl_load.h" interp_push_int :: CInt -> IO ()
foreign import ccall "repl_load.h" interp_push_float :: Double -> IO ()
foreign import ccall "repl_load.h" interp_push_string :: Ptr CChar -> IO ()
foreign import ccall "repl_load.h" interp_call :: Ptr CChar -> IO Int
foreign import ccall "repl_load.h" interp_init :: IO ()
foreign import ccall "repl_load.h" interp_array :: CInt -> IO ()
foreign import ccall "repl_load.h" interp_op :: CInt -> IO ()
foreign import ccall "repl_load.h" interp_floatop :: CInt -> IO ()
foreign import ccall "repl_load.h" interp_unop :: CInt -> IO ()
foreign import ccall "repl_load.h" interp_floatunop :: CInt -> IO ()
foreign import ccall "repl_load.h" interp_int2str :: IO ()
foreign import ccall "repl_load.h" interp_str2int :: IO ()
foreign import ccall "repl_load.h" interp_real2str :: IO ()
foreign import ccall "repl_load.h" interp_str2real :: IO ()
foreign import ccall "repl_load.h" interp_chr2str :: IO ()
foreign import ccall "repl_load.h" interp_str2chr :: IO ()
foreign import ccall "repl_load.h" interp_real2int :: IO ()
foreign import ccall "repl_load.h" interp_int2real :: IO ()
foreign import ccall "repl_load.h" interp_bool2str :: IO ()
foreign import ccall "repl_load.h" interp_append :: IO ()

loadLib :: String -> IO ()
loadLib str = do cstr <- newCString str
                 interp_load cstr

call :: String -> IO Int
call fn = do cfn <- newCString fn
             interp_call cfn

pushString :: String -> IO ()
pushString str = do cstr <- newCString str
                    interp_push_string cstr

data REPLRes = Reload | Load String | Quit | Continue | CompError

-- Command; minimal abbreviation; function to run it; description
commands
    = [("quit", "q", quit, "Exits the top level"),
       ("load <file>", "l", load, "Load a new file"),
       ("reload", "r", reload, "Reload the current file"),
       ("find <name>", "f", showType, 
                             "Find functions with the given name"),
       ("help", "h", help, "Show help text"),
       ("?", "?", help, "Show help text")]

quit _ _ = do return Quit
load _ (f:_) = do return $ Load f
reload _ _ = do return Reload
help _ _ 
    = do putStrLn $ "\nKaya version " ++ version
         putStrLn $ "-------------" ++ take (length version) (repeat '-')
         putStrLn "Commands available:\n"
         putStrLn "\t<expression>     Execute the given expression"
         mapM_ (\ (com, _, _, desc) -> 
                       putStrLn $ "\t:" ++ com ++ (take (16-length com) (repeat ' ')) ++ desc) commands
         putStrLn "\nCommands may be given the shortest unambiguous abbreviation (e.g. :q, :l)\n"
         return Continue
showType ctxt (f:_) 
    = do let nms = nub (lookupname None (UN f) ctxt)
         mapM_ shownm nms
         return Continue
  where shownm (n,(Fn _ args ret,_)) 
            = putStrLn $ (show ret) ++ " " ++ showuser n ++ showlist args
        showlist [] = ""
        showlist xs = "(" ++ showlistargs xs ++ ")"
        showlistargs [x] = show x
        showlistargs (x:xs) = show x ++ ", " ++ showlistargs xs

runREPL :: String -> Name -> [FilePath] -> 
           Context -> EContext -> GContext -> Tags -> Types -> IO REPLRes
runREPL so mod libdirs ctxt ectxt gctxt tags types 
    = do loadLib so
--         removeFile so -- don't leave clutter, it's only used this once!
         interp_init
         res <- doREPL mod libdirs
         interp_close
         removeFile so -- have to wait until its closed on some OSes
         return res
  where
    doREPL :: Name -> [FilePath] -> IO REPLRes
    doREPL (UN mod) libdirs 
        = do let prompt = if (mod == "__REPLMain") then "Kaya" else mod
             r <- readline (prompt ++ "> ")
             res <- case r of
                Nothing -> return Continue
                Just (':':command) -> runCommand (words command) commands
                Just exprinput -> 
                    do processREPL (parserepl mod libdirs exprinput 
                                                  "(top level)") 
                       addHistory exprinput
                       return Continue
             case res of
                  Continue -> doREPL (UN mod) libdirs
                  _ -> return res

    runCommand (c:args) ((_, abbr, fun, _):xs) 
        | matchesAbbrev abbr c = fun ctxt args
        | otherwise = runCommand (c:args) xs
    runCommand _ _ = do putStrLn "Unrecognised command"
                        help ctxt []
                        return Continue

    matchesAbbrev [] _ = True
    matchesAbbrev (a:xs) (c:cs) | a == c = matchesAbbrev xs cs
                                | otherwise = False

    processREPL (Success raw)
        = do let prog = infertype mod ctxt gctxt ectxt tags types [] raw 
                                  (Prim Void) ity []
             case prog of
                  Success (xrv, xeq) -> do
                     phi <- mkSubst xeq
                     xfn <- substTerm mod ctxt phi xrv
                     let xinft = subst phi ity
                     catch (runProg xfn xinft)
                           (\e -> putStrLn(show e))
                  Failure err file line -> do reportError err
    processREPL (Failure err file line)
        = do reportError err

    ity = TyVar (MN ("i", 0))
    runProg expr (Prim Void)
        = do interp_clear
             execute expr
             hFlush stdout
    runProg expr t
        = do interp_clear
             interp_prepare_call
             execute expr
             putStr $ "Returned (" ++ show t ++ "): "
             hFlush stdout
             call (show dumpFun ++ dumpMangle)
             return ()
             
execute (Annotation _ e) = execute e
execute (Apply f args) = do interp_prepare_call
                            mapM_ execute (reverse args)
                            res <- callFn f
                            if res == 0 
                               then fail ""
                               else return ()
execute (GConst (Num x)) = interp_push_int (fromIntegral x)
execute (GConst (Bo True)) = interp_push_int 1
execute (GConst (Bo False)) = interp_push_int 0
execute (GConst (Re x)) = interp_push_float x
execute (GConst (Ch c)) = interp_push_int (fromIntegral (fromEnum c))
execute (GConst (Str str)) = pushString str
execute (Infix op l r) = do execute r
                            execute l
                            runOp op
execute (Unary op x) = do execute x
                          runUnaryOp op
execute (RealInfix op l r) = do execute r
                                execute l
                                runFloatOp op
execute (RealUnary op x) = do execute x
                              runFloatUnaryOp op
execute (Append l r) = do execute l
                          execute r
                          interp_append
execute (ArrayInit args) = do mapM_ execute (reverse args)
                              interp_array (fromIntegral (length args))
execute (Coerce from to e) = do execute e
                                coerceVal from to
execute x = fail $ "Can't execute that in a REPL " ++ show x

callFn (Global fn d _) = call (show fn ++ d)
callFn _ =  fail "Can't call an unnamed function in a REPL (yet)"

coerceVal (Prim Character) (Prim Number) = return ()
coerceVal (Prim Number) (Prim Character) = return ()
coerceVal (Prim StringType) (Prim Number) = interp_str2int
coerceVal (Prim Number) (Prim StringType) = interp_int2str
coerceVal (Prim StringType) (Prim RealNum) = interp_str2real
coerceVal (Prim RealNum) (Prim StringType) = interp_real2str
coerceVal (Prim StringType) (Prim Character) = interp_str2chr
coerceVal (Prim Character) (Prim StringType) = interp_chr2str
coerceVal (Prim Number) (Prim RealNum) = interp_int2real
coerceVal (Prim RealNum) (Prim Number) = interp_real2int
coerceVal (Prim Boolean) (Prim StringType) = interp_bool2str
coerceVal f t = fail $ "Can't coerce from " ++ show f ++ " to " ++ show t

runOp :: Op -> IO ()
runOp op = interp_op (fromIntegral (opNum op))

runFloatOp :: Op -> IO ()
runFloatOp op = interp_floatop (fromIntegral (opNum op))

runUnaryOp :: UnOp -> IO ()
runUnaryOp Not = interp_unop 0
runUnaryOp Neg = interp_unop 1

runFloatUnaryOp :: UnOp -> IO ()
runFloatUnaryOp Not = interp_floatunop 0
runFloatUnaryOp Neg = interp_floatunop 1

opNum Plus = 0
opNum Minus = 1
opNum Times = 2
opNum Divide = 3
opNum Modulo = 4
opNum Power = 5
opNum Equal = 6
opNum NEqual = 7
opNum OpLT = 8
opNum OpGT = 9
opNum OpLE = 10
opNum OpGE = 11
opNum OpAnd = 12
opNum OpOr = 13
opNum OpXOR = 14
opNum BAnd = 15
opNum BOr = 16
opNum OpShLeft = 17
opNum OpShRight = 18
opNum OpAndBool = 19
opNum OpOrBool = 20


