"""A script to generate the code to handle a database from
a configuration script

This script is a Python script which must define at least 2 variables :
- name   : the name of the database
- fields : a list of strings of the form "name:type" where name is the
           name of the field and type, its type
           Example : fields = ['name:str','author:str','issued:datetime.date']

Additional variables can be defined :
- dbmodule       : a string with the database module name. Defaults to 
                   'gadfly'. Must be one of 'gadfly','kirbybase','sqlite'
- data_directory : the directory where database will be stored (defaults to 
                   the same directory as the configuration script)
- pageTitle      : the title printed on top of the HTML pages (defaults to
                   the variable name)
- stylesheet     : the stylesheet url applied to all HTML pages (defaults to
                   default.css in this directory)
- security       : the security level. 3 possible values :
                   'low' : anyone can edit and remove records
                   'standard' : only logged in users can manage the base, only
                   the administrator can manage the users' base
                   'high' : only the administrator can manage the base
                   Default is 'low'
"""

try:
    SCRIPT_END
except NameError:
    pass
else:
    print "This script can't be executed by Karrigell"
    raise SCRIPT_END

import os
import cStringIO
import datetime
import sys
import md5

from records_skeleton import *
import db_specifics

class Field:

    def __init__(self,name,Type,input_format=None):
        self.name = name
        self.Type = Type
        self.type_name = self.Type.__name__
        if self.type_name in ['date','datetime']:
            self.type_name = 'datetime.'+self.type_name
        if input_format is not None:
            self.input_format = input_format
        else:
            self.input_format = 'input'

class BaseDescription:

    def __init__(self,args):
        for k in args.keys():
            setattr(self,k,args[k])

formats = {str:('input','textarea'),
    int:('input',),
    float:('input',),
    datetime.date:('input','calendar'),
    datetime.datetime:('input','calendar')}

types = {'int':int,'str':str,'date':datetime.date,
    'datetime':datetime.datetime,'float':float }

def save_admin_info(admin_file,login,password):
    out = open(admin_file,'w')
    out.write(login+'\n')
    # save md5 digest of the password
    out.write(md5.new(password).digest())

def get_security(configFile):
    # initialize the variables
    conf_vars = {}
    execfile(configFile,conf_vars)
    return conf_vars.get('security','low')

def get_vars(configFile):
    # initialize the variables
    conf_vars = {'filename':configFile}
    execfile(configFile,conf_vars)
    name = os.path.splitext(configFile)[0]
    conf_vars['name'] = name
    # defaults
    dbmodule = conf_vars.setdefault('dbmodule','kirbybase')
    security = conf_vars.setdefault('security','low')
    stylesheet = conf_vars.setdefault('stylesheet','../default.css')
    data_directory = conf_vars.setdefault('data_directory',
        os.path.dirname(configFile))
    page_title = conf_vars.setdefault('page_title', 'Managing base %s' %name)
    
    # next values used only if security is not low
    users_name = conf_vars.setdefault("users_name",'%s_users' %name)
    users_fields = conf_vars.setdefault("users_fields",
        [Field('login',str),Field('password',str)])
    abspath = os.path.join(os.path.normpath(data_directory),name)
    conf_vars['admin_file'] = '%s.ini' %abspath

    return BaseDescription(conf_vars)

def generate_script(name,login=None,password=None):
    # initialize the variables
    conf_vars = {}
    configFile = name + '.py'
    execfile(configFile,conf_vars)
    
    # check if 'fields' is defined in the config file

    if not conf_vars.has_key('fields'):
        print "Error - configuration script must specify a variable 'fields'"
    else:
        fields = conf_vars['fields']

    script_directory = os.path.dirname(configFile)
    # defaults
    security = conf_vars.get('security','low')
    stylesheet = conf_vars.get('stylesheet','../default.css')
    data_directory = conf_vars.get('data_directory',script_directory)
    pageTitle = conf_vars.get('pageTitle', 'Managing base %s' %name)

    # default for database module if undefined
    if not conf_vars.has_key('dbmodule'):
        try:
            import kirbybase
            dbmod_name = 'kirbybase'
        except:
            print "Error - no database module specified in configuration " \
                "script and KirbyBase not installed"
    else:
        dbmod_name = conf_vars['dbmodule']

    # if security level is not 'low', there must be a login/password file
    # for the administrator, created later in the script
    if security != 'low':
        abspath = os.path.join(os.path.normpath(data_directory),name)
        admin_file = '%s.ini' %abspath

        users_name = conf_vars.get("users_name",'%s_users' %name)
        users_fields = conf_vars.get("users_fields",
            ['login:str','password:str'])

    else:
        users_name = ""

    def make_script(name,
        fields,
        pageTitle,
        security,
        stylesheet):
        """Generate the ks script to manage the database called 'name'
        See parameters above
        If authent is True, an HTTP basic authentication test is made for
        every access to the management script
        """
        if security not in ['low','standard','high']:
            print "Invalid value for security : %s\n"
            print "Must be in ['low','standard','high']"

        authent = (security == 'high')

        def field_names(fields):
            return [ field.name for field in fields ]

        # initialize variables used in the db_specifics module
        db_specifics.fieldDefs(dbmod_name,data_directory,name,
            fields,users_name)

        # open the generated ks script
        out = open(os.path.join(script_directory,'%s.ks' %name),'w')

        # write header, imports, database module
        out.write('"""'+hdr1 %(name,
            datetime.datetime.now().strftime('%d-%m-%Y %H:%M:%S'))+'"""')
        out.write(imports)
        out.write("# database-specific modules\n")
        out.write(db_specifics.modulesToImport(dbmod_name))

        if authent:
            out.write(admin %{'admin_file':admin_file})
            # if only the administrator can run the script, 
            # no need for additional security
            security = 'low'

        # write configuration variables
        out.write("\n#parameters\n")
        params = ['name','pageTitle']
        for param in params:
            val = locals()[param]
            if isinstance(val,str):
                out.write('%s = "%s"\n' %(param,locals()[param]))
            else:
                out.write('%s = %s\n' %(param,locals()[param]))
        out.write('field_names = %s\n' %db_specifics.field_names)
        out.write('field_types = %s\n' %db_specifics.field_types)
        if security != 'low':
            out.write("users = r'%s'\n" %db_specifics.users_base)
            out.write('admin_file = r"%s"\n' %admin_file)    

        # code to open an existing base, or create it with the specified fields
        out.write(db_specifics.createCode(dbmod_name))

        if security != 'low':
            # generate code to open the users base
            out.write(db_specifics.openUsersCode(dbmod_name))

        # generate html HEAD part
        out.write('\n#header\nheader=')
        out.write(header %{'stylesheet':stylesheet,'title':pageTitle})
        out.write('\n')

        # default user test is True if security level is 'low', False otherwise
        out.write(hdr2 %{'default_user_test':(security=='low')})

        if security == 'standard':
            out.write(index_if_users1)
            out.write(db_specifics.selectByUserId(dbmod_name))
            out.write(index_if_users2)

        # generate code to select all records and print them in a table
        out.write(db_specifics.selectAllAsDict(dbmod_name,name))
        out.write(index_security_not_high)
        print_code = ['print TD(record["%s"],Class="main")' \
            %field for field in db_specifics.field_names]

        out.write(('\n'+' '*12).join(print_code))
        out.write(index_security_not_high_2)

        out.write(index_end_1)
        if not security == "low":
            out.write("""    print '<p><a href="../%s.ks">Admin</a>'\n""" 
                %users_name)
        out.write(index_end_2)

        # generate code to edit an existing record or create a new one
        out.write(edit_1 %(db_specifics.selectByRecordId(dbmod_name),
            db_specifics.field_names))
        input_code = []
        for field in fields:
            fn = field.name
            if field.input_format == 'textarea':
                input_code.append('print TR(TD("%s")+'
                    'TD(TEXTAREA(record["%s"],name="%s",rows="10",cols="50")))'
                        %(fn,fn,fn))            
            elif field.input_format == 'calendar':
                input_code.append('print TR(TD("%s")+'
                    'TD(INPUT(name="%s",id="%s",size="20",value=record["%s"])+' \
                        'A(IMG(src="../Calandar.gif",border=0),' \
                        'href= "javascript:scwShow(document.getElementById(\'%s\'), this);")))'
                        %(fn,fn,fn,fn,fn))
            else:
                input_code.append('print TR(TD("%s")+'
                    'TD(INPUT(name="%s",size="40",value=record["%s"])))'
                        %(fn,fn,fn))

        out.write(('\n'+' '*4).join(input_code))
        out.write(edit_2)

        # generate code to insert or remove a record, 
        # check users login/password
        out.write(rest %(db_specifics.insert(dbmod_name),
                        db_specifics.remove(dbmod_name),
                        db_specifics.selectUserByLogin(dbmod_name)))

        # generate code to redirect to the admin page
        if security != 'low':
            out.write(admin_method %users_name)

        out.close()

    # generate main script
    make_script(name, 
        fields,
        pageTitle,
        security,
        stylesheet)

    if not security == 'low':
        # generate users management script
        make_script(users_name,
            users_fields,
            "%s users" %name,
            "high",
            stylesheet)

    return name