
/*
 * Kanatest
 *
 * Copyright (C) 2001-2004, 2006 Tomasz Mąka <pasp@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>

#include <stdio.h>
#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include <gdk/gdkkeysyms.h>

#include <time.h>
#include <ctype.h>
#include <string.h>
#include <stdlib.h>

#include "gui.h"
#include "about.h"
#include "main.h"
#include "prefs.h"
#include "test.h"
#include "stats.h"
#include "chart.h"
#include "i18n.h"
#include "options.h"

#include "graphics.h"

GtkWidget   *main_window;

GtkWidget   *logo_area;
GtkWidget   *char_label;

GtkWidget   *start_button;
GtkWidget   *stat_button;
GtkWidget   *chart_button;
GtkWidget   *quit_button;
GtkWidget   *about_button;
GtkWidget   *prefs_button;
GtkWidget   *frame_ro;
GtkTooltips *tooltips;
GtkWidget   *combobox_lesson;
GtkWidget   *combobox_kana_mode;
GtkWidget   *label_ka;
GtkWidget   *label_le;

GtkWidget   *romaji_entry;
GtkWidget   *stop_button;
GtkProgressBar *progressbar;

GdkPixbuf   *icon, *logo;

time_t      start_time;
gint        old_kana_type = -1;
gboolean    any_key = FALSE;
gboolean    stop_flag = FALSE;
gboolean    test_state = FALSE;

/*--------------------------------------------------------------------*/

GtkWidget* 
gui_stock_label_button (gchar *blabel, const gchar *bstock) {

    GtkWidget   *button;
    GtkWidget   *alignment;
    GtkWidget   *hbox;
    GtkWidget   *image;

    button = g_object_new (GTK_TYPE_BUTTON, "visible", TRUE, NULL);

    alignment = gtk_alignment_new (0.5, 0.5, 0.0, 0.0);
    hbox = gtk_hbox_new (FALSE, 2);
    gtk_container_add (GTK_CONTAINER (alignment), hbox);

    image = gtk_image_new_from_stock (bstock, GTK_ICON_SIZE_BUTTON);
    if (image)
        gtk_box_pack_start (GTK_BOX (hbox), image, FALSE, TRUE, 0);

    if (blabel)
    gtk_box_pack_start (GTK_BOX (hbox),
        g_object_new (GTK_TYPE_LABEL, "label", blabel, "use_underline", TRUE, NULL),
        FALSE, TRUE, 0);

    gtk_widget_show_all (alignment);
    gtk_container_add (GTK_CONTAINER (button), alignment);

    return button;
}

/*--------------------------------------------------------------------*/

void 
gui_set_progress (void) {

    gchar buffer[BUFFER_SIZE];

    sprintf (buffer, "[ %d / %d ]", question_counter, max_entries_in_test);
    gtk_progress_bar_set_text (GTK_PROGRESS_BAR (progressbar), buffer);
    gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR (progressbar), (gfloat)question_counter / max_entries_in_test); 
    gtk_entry_set_text (GTK_ENTRY(romaji_entry), "");
}

/*--------------------------------------------------------------------*/

void 
gui_buttons_handler_cb (GtkWidget *widget, gpointer data) {

    switch((gint)data) {

    case B_STATS:   /* STATS */
            stats_create_window ();
            break;  

    case B_ABOUT:   /* ABOUT */
            about_create_window ();
            break;  

    case B_OPTIONS: /* OPTIONS */
            options_create_window ();
            break;  

    case B_CHART:   /* CHART */
            chart_create_window ();
            break;  

    case B_STOP:    /* STOP */
            test_state = FALSE;
            gui_disable_test ();
            break;  

    case B_START:   /* START */

            test_state = TRUE;
            gtk_widget_show(char_label);
            gtk_widget_hide(logo_area);

            gui_set_widgets_status (FALSE);

            gtk_widget_grab_focus (romaji_entry);

            test_init();
            test_generate_tables (max_entries_in_test);

            gui_display_kana (questions_table[question_counter], config.kana_mode);
            gui_set_progress ();

            start_time = time (NULL);
            break;  

    default:
            break;
    }
}

/*--------------------------------------------------------------------*/

void 
gui_disable_test (void) {

    gtk_widget_show(logo_area);
    gtk_widget_hide(char_label);

    gui_set_widgets_status (TRUE);

    gtk_progress_bar_set_text (GTK_PROGRESS_BAR (progressbar),
                                _("Select test mode, kana set and press START"));
    gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR (progressbar), 0.0);
    gtk_entry_set_text (GTK_ENTRY(romaji_entry), "");

    gtk_image_set_from_pixbuf (GTK_IMAGE (logo_area), logo);
    gtk_widget_grab_focus (start_button);
}

/*--------------------------------------------------------------------*/

void 
gui_set_widgets_status (gint mode) {

    gboolean st_a, st_b;

    st_a = TRUE;
    st_b = FALSE;

    if(mode) {
        st_a = FALSE;
        st_b = TRUE;
    }       

    gtk_widget_set_sensitive (romaji_entry, st_a);
    gtk_widget_set_sensitive (frame_ro, st_a);
    gtk_widget_set_sensitive (stop_button, st_a);

    gtk_widget_set_sensitive (start_button, st_b);
    gtk_widget_set_sensitive (stat_button, st_b);
    gtk_widget_set_sensitive (quit_button, st_b);
    gtk_widget_set_sensitive (about_button, st_b);
    gtk_widget_set_sensitive (prefs_button, st_b);
    gtk_widget_set_sensitive (chart_button, st_b);
    gtk_widget_set_sensitive (combobox_kana_mode, st_b);
    gtk_widget_set_sensitive (combobox_lesson, st_b);
    gtk_widget_set_sensitive (label_ka, st_b);
    gtk_widget_set_sensitive (label_le, st_b);

}

/*--------------------------------------------------------------------*/

void 
gui_display_kana (gint number, gint mode) {

    gchar tmpbuf[BUFFER_SIZE], letbuf[BUFFER_SIZE];


    switch (mode) {
        case HIRAGANA:
            if (number >= MIXED_SEPARATOR) {
                number -= MIXED_SEPARATOR;
            }
            sprintf (tmpbuf, "<span font_desc='80' face='%s' color='%s'>%s</span>", 
                     config.kana_font_face,
                     config.kana_color,
                     kana_signs[3*number+HIRAGANA]);
            old_kana_type = HIRAGANA;
            break;
        case KATAKANA:
            if (number >= MIXED_SEPARATOR) {
                number -= MIXED_SEPARATOR;
            }
            sprintf (tmpbuf, "<span font_desc='80' face='%s' color='%s'>%s</span>", 
                     config.kana_font_face,
                     config.kana_color,
                     kana_signs[3*number+KATAKANA]);
            old_kana_type = KATAKANA;
            break;
        case ROMAJI:
            if (number >= MIXED_SEPARATOR) {
                number -= MIXED_SEPARATOR;
            }

            sprintf(letbuf, "%s", kana_signs[3*number]);

            if (number == 57) {             /* di */
                sprintf(letbuf, "di, ji");
            } else if (number == 58) {      /* du */
                sprintf(letbuf, "du, zu");
            }

            if (old_kana_type == HIRAGANA) {
                sprintf (tmpbuf, "<span font_desc='60' face='%s' color='%s'>%s</span>"
                                 "<span font_desc='40' face='%s' color='%s'>(%s)</span>",
                                 config.kana_font_face,
                                 config.kana_color,
                                 kana_signs[3*number+HIRAGANA], 
                                 config.kana_font_face,
                                 config.romaji_color,
                                 letbuf);
            } else if (old_kana_type == KATAKANA) {
                sprintf (tmpbuf, "<span font_desc='60' face='%s' color='%s'>%s</span>"
                                 "<span font_desc='40' face='%s' color='%s'>(%s)</span>",
                                 config.kana_font_face,
                                 config.kana_color,
                                 kana_signs[3*number+KATAKANA], 
                                 config.kana_font_face,
                                 config.romaji_color,
                                 letbuf);
            } else {
                sprintf (tmpbuf, "<span font_desc='80' face='%s' color='%s'>%s</span>", 
                         config.kana_font_face,
                         config.romaji_color,
                         kana_signs[3*number]);
            }
            break;
        case MIXED:
            if (number >= MIXED_SEPARATOR) {
                sprintf (tmpbuf, "<span font_desc='80' face='%s' color='%s'>%s</span>", 
                         config.kana_font_face,
                         config.kana_color,
                         kana_signs[3*(number-MIXED_SEPARATOR)+KATAKANA]);
                old_kana_type = KATAKANA;
            } else {
                sprintf (tmpbuf, "<span font_desc='80' face='%s' color='%s'>%s</span>", 
                         config.kana_font_face,
                         config.kana_color,
                         kana_signs[3*number+HIRAGANA]);
                old_kana_type = HIRAGANA;
            }
            break;
        default:
            fprintf (stderr, "WARNING: Invalid mode selected.\n");
            break;
    }

    gtk_label_set_markup (GTK_LABEL (char_label), tmpbuf);
}

/*--------------------------------------------------------------------*/

gint 
gui_rm_key_press_cb (GtkWidget *widget, GdkEventKey *event, gpointer data) {

    switch (event->keyval) {

        case GDK_Escape:
            gtk_entry_set_text (GTK_ENTRY(romaji_entry), "");
            return TRUE;

        case GDK_Return:
            if (strlen ((gchar*)gtk_entry_get_text (GTK_ENTRY(romaji_entry)))) {
                test_check_answer ((gchar*)(gtk_entry_get_text (GTK_ENTRY(romaji_entry))));
            }
            return TRUE;

    }

    return FALSE;
}

gint 
gui_mw_key_press_cb (GtkWidget *widget, GdkEventKey *event, gpointer data) {

    if (test_state == FALSE) {

        switch (event->keyval) {
            case GDK_o:
                options_create_window ();
                return TRUE;
            case GDK_a:
                about_create_window ();
                return TRUE;
            case GDK_c:
                chart_create_window ();
                return TRUE;
            case GDK_s:
                stats_create_window ();
                return TRUE;
        }
    }

    any_key = TRUE;
    return FALSE;
}


/*--------------------------------------------------------------------*/

void 
gui_close_window_cb (GtkWidget *widget, gpointer data) {

    gtk_window_get_size (GTK_WINDOW(main_window),
                        &config.window_size_x, &config.window_size_y);
    gdk_window_get_root_origin (main_window->window,
                                &config.window_x, &config.window_y);
    gtk_main_quit ();
}

/*--------------------------------------------------------------------*/

void
gui_combobox_kana_handler_cb (GtkComboBox *widget, gpointer user_data) {

    config.kana_mode = HIRAGANA + gtk_combo_box_get_active(GTK_COMBO_BOX(widget));
} 

void
gui_combobox_kana_set_handler_cb (GtkComboBox *widget, gpointer user_data) {

    config.kana_set = gtk_combo_box_get_active(GTK_COMBO_BOX(widget));
} 


/*--------------------------------------------------------------------*/

void 
gui_create_window (void) {

    GtkWidget       *frame1;
    GtkWidget       *vbox0;
    GtkWidget       *vbox1;
    GtkWidget       *hseparator;
    GtkWidget       *frame2;
    GtkWidget       *frame;
    GtkWidget       *hbox1;
    GtkWidget       *hbox2;
    GtkWidget       *hbuttonbox;
    GtkWidget       *label;
    GtkWidget       *alignment;
    gint            i;
    gchar           buffer[BUFFER_SIZE];

    tooltips = gtk_tooltips_new();

    main_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
    gtk_container_set_border_width (GTK_CONTAINER (main_window), 4);
    gtk_window_set_title (GTK_WINDOW (main_window), "Kanatest");

    gtk_window_set_default_size (GTK_WINDOW(main_window),
                                config.window_size_x, config.window_size_y);

    gtk_window_set_resizable (GTK_WINDOW (main_window), TRUE);

    gtk_window_move (GTK_WINDOW (main_window),
                                config.window_x, config.window_y);

    g_signal_connect (G_OBJECT (main_window), "delete_event",
                        G_CALLBACK(gui_close_window_cb), NULL);
    g_signal_connect (G_OBJECT(main_window), "key_press_event",
                        G_CALLBACK(gui_mw_key_press_cb), NULL);

    gtk_widget_show (main_window);

    icon = gdk_pixbuf_new_from_inline(-1, kanatest_icon, FALSE, NULL);
    gtk_window_set_icon (GTK_WINDOW(main_window), icon);
    g_object_unref(icon);

    vbox0 = gtk_vbox_new (FALSE, 0);
    gtk_widget_show (vbox0);
    gtk_container_add (GTK_CONTAINER (main_window), vbox0);

    frame1 = gtk_frame_new (NULL);
    gtk_widget_show (frame1);
    gtk_container_set_border_width (GTK_CONTAINER (frame1), 4);
    gtk_frame_set_shadow_type (GTK_FRAME (frame1), GTK_SHADOW_ETCHED_OUT);
    gtk_box_pack_start (GTK_BOX (vbox0), frame1, TRUE, TRUE, 0);

    vbox1 = gtk_vbox_new (FALSE, 0);
    gtk_widget_show (vbox1);
    gtk_container_add (GTK_CONTAINER (frame1), vbox1);

    logo_area = gtk_image_new();
    logo = gdk_pixbuf_new_from_inline(-1, kanatest_logo, FALSE, NULL);
    gtk_widget_show(logo_area);
    gtk_box_pack_start (GTK_BOX (vbox1), logo_area, TRUE, TRUE, 0);

    char_label = gtk_label_new (NULL);
    gtk_box_pack_start (GTK_BOX (vbox1), char_label, TRUE, TRUE, 0);

    hseparator = gtk_hseparator_new ();
    gtk_widget_show (hseparator);
    gtk_box_pack_start (GTK_BOX (vbox1), hseparator, FALSE, TRUE, 0);

    progressbar = GTK_PROGRESS_BAR(gtk_progress_bar_new());
    gtk_widget_show (GTK_WIDGET(progressbar));
    gtk_box_pack_start (GTK_BOX (vbox1), GTK_WIDGET(progressbar), FALSE, FALSE, 0);

    hseparator = gtk_hseparator_new ();
    gtk_widget_show (hseparator);
    gtk_box_pack_start (GTK_BOX (vbox1), hseparator, FALSE, TRUE, 0);

    hbox1 = gtk_hbox_new (FALSE, 0);
    gtk_widget_show (hbox1);
    gtk_box_pack_start (GTK_BOX (vbox1), hbox1, FALSE, FALSE, 8);

    frame_ro = gtk_frame_new (NULL);
    gtk_widget_show (frame_ro);
    gtk_box_pack_start (GTK_BOX (hbox1), frame_ro, FALSE, FALSE, 0);
    gtk_frame_set_shadow_type (GTK_FRAME (frame_ro), GTK_SHADOW_NONE);

    alignment = gtk_alignment_new (0.5, 0.5, 1, 1);
    gtk_widget_show (alignment);
    gtk_container_add (GTK_CONTAINER (frame_ro), alignment);
    gtk_alignment_set_padding (GTK_ALIGNMENT (alignment), 0, 0, 12, 0);

    label = gtk_label_new (NULL);
    gtk_widget_show (label);
    gtk_frame_set_label_widget (GTK_FRAME(frame_ro), label);
    sprintf(buffer, "<b>%s </b>", _("Romaji:"));
    gtk_label_set_markup (GTK_LABEL (label), buffer);

    romaji_entry = gtk_entry_new();
    gtk_entry_set_max_length(GTK_ENTRY(romaji_entry), 3);
    g_signal_connect (G_OBJECT(romaji_entry), "key_press_event",
                        G_CALLBACK(gui_rm_key_press_cb), NULL);
    gtk_widget_set_size_request(romaji_entry, 130, -1);
    gtk_widget_show (romaji_entry);
    gtk_container_add (GTK_CONTAINER (alignment), romaji_entry);

    hbox2 = gtk_hbox_new (FALSE, 0);
    gtk_widget_show (hbox2);
    gtk_box_pack_start (GTK_BOX (hbox1), hbox2, TRUE, FALSE, 0);

    frame2 = gtk_frame_new (NULL);
    gtk_widget_show (frame2);
    gtk_box_pack_start (GTK_BOX (hbox1), frame2, FALSE, FALSE, 0);
    gtk_frame_set_shadow_type (GTK_FRAME (frame2), GTK_SHADOW_NONE);
    gtk_container_set_border_width (GTK_CONTAINER (frame2), 4);

    hbox2 = gtk_hbox_new (FALSE, 0);
    gtk_widget_show (hbox2);
    gtk_container_add (GTK_CONTAINER (frame2), hbox2);

    stat_button = gui_stock_label_button(NULL, GTK_STOCK_FIND);
    GTK_WIDGET_UNSET_FLAGS(stat_button, GTK_CAN_FOCUS);
    gtk_button_set_relief (GTK_BUTTON (stat_button), GTK_RELIEF_NONE);
    g_signal_connect (G_OBJECT (stat_button), "clicked",
                        G_CALLBACK (gui_buttons_handler_cb), (gpointer)B_STATS);
    gtk_widget_show (stat_button);
    gtk_box_pack_start (GTK_BOX (hbox2), stat_button, FALSE, FALSE, 0);
    gtk_container_set_border_width (GTK_CONTAINER (stat_button), 2);
    gtk_tooltips_set_tip (tooltips, stat_button, _("Statistics"), NULL);

    chart_button = gui_stock_label_button(NULL, GTK_STOCK_INDEX);
    GTK_WIDGET_UNSET_FLAGS(chart_button, GTK_CAN_FOCUS);
    gtk_button_set_relief (GTK_BUTTON (chart_button), GTK_RELIEF_NONE);
    g_signal_connect (G_OBJECT (chart_button), "clicked",
                        G_CALLBACK (gui_buttons_handler_cb), (gpointer)B_CHART);
    gtk_widget_show (chart_button);
    gtk_box_pack_start (GTK_BOX (hbox2), chart_button, FALSE, FALSE, 0);
    gtk_container_set_border_width (GTK_CONTAINER (chart_button), 2);
    gtk_tooltips_set_tip (tooltips, chart_button, _("Kana chart"), NULL);

    prefs_button = gui_stock_label_button(NULL, GTK_STOCK_PREFERENCES);
    GTK_WIDGET_UNSET_FLAGS(prefs_button, GTK_CAN_FOCUS);
    gtk_button_set_relief (GTK_BUTTON (prefs_button), GTK_RELIEF_NONE);
    g_signal_connect (G_OBJECT (prefs_button), "clicked",
                        G_CALLBACK (gui_buttons_handler_cb), (gpointer)B_OPTIONS);
    gtk_widget_show (prefs_button);
    gtk_box_pack_start (GTK_BOX (hbox2), prefs_button, FALSE, FALSE, 0);
    gtk_container_set_border_width (GTK_CONTAINER (prefs_button), 2);
    gtk_tooltips_set_tip (tooltips, prefs_button, _("Preferences"), NULL);

    about_button = gui_stock_label_button(NULL, GTK_STOCK_ABOUT);
    GTK_WIDGET_UNSET_FLAGS(about_button, GTK_CAN_FOCUS);
    gtk_button_set_relief (GTK_BUTTON (about_button), GTK_RELIEF_NONE);
    g_signal_connect (G_OBJECT (about_button), "clicked",
                        G_CALLBACK (gui_buttons_handler_cb), (gpointer)B_ABOUT);
    gtk_widget_show (about_button);
    gtk_box_pack_start (GTK_BOX (hbox2), about_button, TRUE, TRUE, 0);
    gtk_container_set_border_width (GTK_CONTAINER (about_button), 2);
    gtk_tooltips_set_tip (tooltips, about_button, _("About"), NULL);


    hbox1 = gtk_hbox_new (FALSE, 0);
    gtk_widget_show (hbox1);
    gtk_box_pack_start (GTK_BOX (vbox1), hbox1, FALSE, FALSE, 8);

    frame = gtk_frame_new (NULL);
    gtk_widget_show (frame);
    gtk_box_pack_start (GTK_BOX (hbox1), frame, FALSE, FALSE, 0);
    gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_NONE);

    alignment = gtk_alignment_new (0.5, 0.5, 1, 1);
    gtk_widget_show (alignment);
    gtk_container_add (GTK_CONTAINER (frame), alignment);
    gtk_alignment_set_padding (GTK_ALIGNMENT (alignment), 0, 0, 12, 0);

    combobox_kana_mode = gtk_combo_box_new_text ();
    gtk_widget_show (combobox_kana_mode);
    g_signal_connect (G_OBJECT (combobox_kana_mode), "changed",
                      G_CALLBACK (gui_combobox_kana_handler_cb), NULL);
    gtk_container_add (GTK_CONTAINER (alignment), combobox_kana_mode);
    gtk_container_set_border_width (GTK_CONTAINER (alignment), 4);

    for(i=0; i < KANA_MODE_NAMES; i++) {
        gtk_combo_box_append_text (GTK_COMBO_BOX (combobox_kana_mode), _(kana_mode_names[i+1]));
    }

    label_ka = gtk_label_new (NULL);
    gtk_widget_show (label_ka);
    gtk_frame_set_label_widget (GTK_FRAME(frame), label_ka);
    sprintf(buffer, "<b>%s </b>", _("Test mode:"));
    gtk_label_set_markup (GTK_LABEL (label_ka), buffer);

    frame = gtk_frame_new (NULL);
    gtk_widget_show (frame);
    gtk_box_pack_end (GTK_BOX (hbox1), frame, FALSE, FALSE, 8);
    gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_NONE);

    alignment = gtk_alignment_new (0.5, 0.5, 1, 1);
    gtk_widget_show (alignment);
    gtk_container_add (GTK_CONTAINER (frame), alignment);
    gtk_alignment_set_padding (GTK_ALIGNMENT (alignment), 0, 0, 12, 0);

    combobox_lesson = gtk_combo_box_new_text ();
    gtk_widget_show (combobox_lesson);
    g_signal_connect (G_OBJECT (combobox_lesson), "changed",
                      G_CALLBACK (gui_combobox_kana_set_handler_cb), NULL);
    gtk_container_add (GTK_CONTAINER (alignment), combobox_lesson);
    gtk_container_set_border_width (GTK_CONTAINER (alignment), 4);

    for(i=0; i < KANA_SET_NAMES; i++) {
        gtk_combo_box_append_text (GTK_COMBO_BOX (combobox_lesson), _(kana_signs_names[i]));
    }

    label_le = gtk_label_new (NULL);
    gtk_widget_show (label_le);
    gtk_frame_set_label_widget (GTK_FRAME(frame), label_le);
    sprintf(buffer, "<b>%s </b>", _("Kana set:"));
    gtk_label_set_markup (GTK_LABEL (label_le), buffer);

    hbuttonbox = gtk_hbutton_box_new ();
    gtk_widget_show (hbuttonbox);
    gtk_box_pack_start (GTK_BOX (vbox0), hbuttonbox, FALSE, TRUE, 0);
    gtk_box_set_spacing (GTK_BOX (hbuttonbox), 5);

    start_button = gui_stock_label_button(_("_Start"), GTK_STOCK_EXECUTE);
    g_signal_connect (G_OBJECT (start_button), "clicked",
                        G_CALLBACK (gui_buttons_handler_cb), (gpointer)B_START);
    gtk_widget_show (start_button);
    gtk_container_add (GTK_CONTAINER (hbuttonbox), start_button);
    gtk_container_set_border_width (GTK_CONTAINER (start_button), 4);
    GTK_WIDGET_SET_FLAGS (start_button, GTK_CAN_DEFAULT);
    gtk_tooltips_set_tip (tooltips, start_button, _("Press to begin testing..."), NULL);

    stop_button = gui_stock_label_button(_("_Stop"), GTK_STOCK_STOP);
    g_signal_connect (G_OBJECT (stop_button), "clicked",
                        G_CALLBACK (gui_buttons_handler_cb), (gpointer)B_STOP);
    gtk_widget_show (stop_button);
    gtk_container_add (GTK_CONTAINER (hbuttonbox), stop_button);
    gtk_container_set_border_width (GTK_CONTAINER (stop_button), 4);
    GTK_WIDGET_SET_FLAGS (stop_button, GTK_CAN_DEFAULT);
    gtk_tooltips_set_tip (tooltips, stop_button, _("Press to stop testing..."), NULL);

    quit_button = gui_stock_label_button(_("_Quit"), GTK_STOCK_QUIT);
    g_signal_connect (G_OBJECT (quit_button), "clicked",
                        G_CALLBACK (gui_close_window_cb), NULL);
    gtk_widget_show (quit_button);
    gtk_container_add (GTK_CONTAINER (hbuttonbox), quit_button);
    gtk_container_set_border_width (GTK_CONTAINER (quit_button), 4);
    GTK_WIDGET_SET_FLAGS (quit_button, GTK_CAN_DEFAULT);
    gtk_tooltips_set_tip (tooltips, quit_button, _("Exit!"), NULL);

    gui_disable_test ();

    gtk_combo_box_set_active (GTK_COMBO_BOX (combobox_lesson), config.kana_set);
    gtk_combo_box_set_active (GTK_COMBO_BOX (combobox_kana_mode), config.kana_mode - HIRAGANA);

    if (config.enable_tooltips == TRUE) {
        gtk_tooltips_enable(tooltips);
    } else {
        gtk_tooltips_disable(tooltips);
    }

}

/*--------------------------------------------------------------------*/

