package gnu.crypto.key.dh;

// ----------------------------------------------------------------------------
// $Id: GnuDHPrivateKey.java,v 1.1 2005/10/19 20:15:39 guilhem Exp $
//
// Copyright (C) 2003 Free Software Foundation, Inc.
//
// This file is part of GNU Crypto.
//
// GNU Crypto is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// GNU Crypto is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; see the file COPYING.  If not, write to the
//
//    Free Software Foundation Inc.,
//    51 Franklin Street, Fifth Floor,
//    Boston, MA 02110-1301
//    USA
//
// Linking this library statically or dynamically with other modules is
// making a combined work based on this library.  Thus, the terms and
// conditions of the GNU General Public License cover the whole
// combination.
//
// As a special exception, the copyright holders of this library give
// you permission to link this library with independent modules to
// produce an executable, regardless of the license terms of these
// independent modules, and to copy and distribute the resulting
// executable under terms of your choice, provided that you also meet,
// for each linked independent module, the terms and conditions of the
// license of that module.  An independent module is a module which is
// not derived from or based on this library.  If you modify this
// library, you may extend this exception to your version of the
// library, but you are not obligated to do so.  If you do not wish to
// do so, delete this exception statement from your version.
// ----------------------------------------------------------------------------

import gnu.crypto.Registry;
import gnu.crypto.key.IKeyPairCodec;

import java.math.BigInteger;
import javax.crypto.interfaces.DHPrivateKey;

/**
 * <p>An implementation of the Diffie-Hellman private key.</p>
 *
 * <p>Reference:</p>
 * <ol>
 *    <li><a href="http://www.ietf.org/rfc/rfc2631.txt">Diffie-Hellman Key
 *    Agreement Method</a><br>
 *    Eric Rescorla.</li>
 * </ol>
 *
 * @version $Revision: 1.1 $
 */
public class GnuDHPrivateKey extends GnuDHKey implements DHPrivateKey {

   // Constants and variables
   // -------------------------------------------------------------------------

   /** The private exponent. */
   private final BigInteger x;

   // Constructor(s)
   // -------------------------------------------------------------------------

   public GnuDHPrivateKey(BigInteger q, BigInteger p, BigInteger g, BigInteger x) {
      super(q, p, g);

      this.x = x;
   }

   // Class methods
   // -------------------------------------------------------------------------

   /**
    * <p>A class method that takes the output of the <code>encodePrivateKey()</code>
    * method of a DH keypair codec object (an instance implementing
    * {@link IKeyPairCodec} for DH keys, and re-constructs an instance of this
    * object.</p>
    *
    * @param k the contents of a previously encoded instance of this object.
    * @exception ArrayIndexOutOfBoundsException if there is not enough bytes,
    * in <code>k</code>, to represent a valid encoding of an instance of
    * this object.
    * @exception IllegalArgumentException if the byte sequence does not
    * represent a valid encoding of an instance of this object.
    */
   public static GnuDHPrivateKey valueOf(byte[] k) {
      // check magic...
      // we should parse here enough bytes to know which codec to use, and
      // direct the byte array to the appropriate codec.  since we only have one
      // codec, we could have immediately tried it; nevertheless since testing
      // one byte is cheaper than instatiating a codec that will fail we test
      // the first byte before we carry on.
      if (k[0] == Registry.MAGIC_RAW_DH_PRIVATE_KEY[0]) {
         // it's likely to be in raw format. get a raw codec and hand it over
         IKeyPairCodec codec = new DHKeyPairRawCodec();
         return (GnuDHPrivateKey) codec.decodePrivateKey(k);
      } else {
         throw new IllegalArgumentException("magic");
      }
   }

   // Instance methods
   // -------------------------------------------------------------------------

   // java.security.Key interface implementation ------------------------------

   /** @deprecated see getEncoded(int). */
   public byte[] getEncoded() {
      return getEncoded(IKeyPairCodec.RAW_FORMAT);
   }

   // javax.crypto.interfaces.DHPrivateKey interface implementation -----------

   public BigInteger getX() {
      return x;
   }

   // other methods -----------------------------------------------------------

   /**
    * <p>Returns the encoded form of this private key according to the
    * designated format.</p>
    *
    * @param format the desired format identifier of the resulting encoding.
    * @return the byte sequence encoding this key according to the designated
    * format.
    * @exception IllegalArgumentException if the format is not supported.
    * @see gnu.crypto.key.dh.DHKeyPairRawCodec
    */
   public byte[] getEncoded(int format) {
      byte[] result;
      switch (format) {
      case IKeyPairCodec.RAW_FORMAT:
         result = new DHKeyPairRawCodec().encodePrivateKey(this);
         break;
      default:
         throw new IllegalArgumentException("format");
      }
      return result;
   }
}
