// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Timothy M. Shead (tshead@k-3d.com)
		\author Romain Behar (romainbehar@yahoo.com)
*/

#include <k3dsdk/i18n.h>
#include <k3dsdk/axis.h>
#include <k3dsdk/bounding_box.h>
#include <k3dsdk/imaterial.h>
#include <k3dsdk/node.h>
#include <k3dsdk/persistent.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh_modifier.h>
#include <k3dsdk/mesh_selection_sink.h>
#include <k3dsdk/module.h>

#include <iterator>

namespace libk3ddeformation
{

/////////////////////////////////////////////////////////////////////////////
// shear_points

class shear_points :
	public k3d::mesh_selection_sink<k3d::mesh_modifier<k3d::persistent<k3d::node> > >
{
	typedef k3d::mesh_selection_sink<k3d::mesh_modifier<k3d::persistent<k3d::node> > > base;

public:
	shear_points(k3d::idocument& Document) :
		base(Document),
		m_direction(init_owner(*this) + init_name("direction") + init_label(_("Direction")) + init_description(_("Shear direction")) + init_value(k3d::Z) + init_enumeration(k3d::axis_values())),
		m_axis(init_owner(*this) + init_name("axis") + init_label(_("Axis")) + init_description(_("Shear axis")) + init_value(k3d::X) + init_enumeration(k3d::axis_values())),
		m_shear_factor(init_owner(*this) + init_name("shear_factor") + init_label(_("Shear factor")) + init_description(_("Shear factor")) + init_value(0.0) + init_step_increment(0.01) + init_units(typeid(k3d::measurement::scalar)))
	{
		m_mesh_selection.changed_signal().connect(make_reset_mesh_slot());

		m_direction.changed_signal().connect(make_update_mesh_slot());
		m_axis.changed_signal().connect(make_update_mesh_slot());
		m_shear_factor.changed_signal().connect(make_update_mesh_slot());
	}

	void on_create_mesh(const k3d::mesh& InputMesh, k3d::mesh& Mesh)
	{
		k3d::deep_copy(InputMesh, Mesh);
		k3d::replace_selection(m_mesh_selection.value(), Mesh);
	}

	void on_update_mesh(const k3d::mesh& Source, k3d::mesh& Target)
	{
		// Sanity checks ...
		assert(Source.points.size() == Target.points.size());

		const k3d::axis direction = m_direction.value();
		const k3d::axis axis = m_axis.value();
		const double shear_factor = m_shear_factor.value();

		const double xy = k3d::X == direction && k3d::Y == axis ? shear_factor : 0;
		const double xz = k3d::X == direction && k3d::Z == axis ? shear_factor : 0;
		const double yx = k3d::Y == direction && k3d::X == axis ? shear_factor : 0;
		const double yz = k3d::Y == direction && k3d::Z == axis ? shear_factor : 0;
		const double zx = k3d::Z == direction && k3d::X == axis ? shear_factor : 0;
		const double zy = k3d::Z == direction && k3d::Y == axis ? shear_factor : 0;

		const k3d::matrix4 shear_matrix = k3d::shearing3D(xy, xz, yx, yz, zx, zy);

		for(unsigned long i = 0; i != Target.points.size(); ++i)
			Target.points[i]->position = k3d::mix(Source.points[i]->position, shear_matrix * Source.points[i]->position, Target.points[i]->selection_weight);
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<shear_points>,
			k3d::interface_list<k3d::imesh_source,
			k3d::interface_list<k3d::imesh_sink > > > factory(
				k3d::uuid(0x0def4291, 0x51ce479b, 0xadf9aec8, 0xed522ccc),
				"ShearPoints",
				_("Shears mesh points along an axis"),
				"Deformation",
				k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_data(k3d::axis, immutable_name, change_signal, with_undo, local_storage, no_constraint, enumeration_property, with_serialization) m_direction;
	k3d_data(k3d::axis, immutable_name, change_signal, with_undo, local_storage, no_constraint, enumeration_property, with_serialization) m_axis;
	k3d_data(double, immutable_name, change_signal, with_undo, local_storage, no_constraint, measurement_property, with_serialization) m_shear_factor;
};

/////////////////////////////////////////////////////////////////////////////
// shear_points_factory

k3d::iplugin_factory& shear_points_factory()
{
	return shear_points::get_factory();
}

} // namespace libk3ddeformation


