// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Anders Dahnielson (anders@dahnielson.com)
*/

#include <k3dsdk/i18n.h>
#include "bitmap_functions.h"

#include <k3dsdk/bitmap_modifier.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/module.h>
#include <k3dsdk/node.h>
#include <k3dsdk/persistent.h>

namespace libk3dbitmap
{

class bitmap_size :
	public k3d::bitmap_modifier<k3d::persistent<k3d::node> >
{
	typedef k3d::bitmap_modifier<k3d::persistent<k3d::node> > base;

public:
	bitmap_size(k3d::idocument& Document) :
		base(Document),
		m_width(init_owner(*this) + init_name("pixel_width") + init_label(_("Output pixel width")) + init_description(_("Output pixel width")) + init_value(320) + init_constraint(constraint::minimum(1L)) + init_precision(0) + init_step_increment(1) + init_units(typeid(k3d::measurement::scalar))),
		m_height(init_owner(*this) + init_name("pixel_height") + init_label(_("Output pixel height")) + init_description(_("Output pixel height")) + init_value(240) + init_constraint(constraint::minimum(1L)) + init_precision(0) + init_step_increment(1) + init_units(typeid(k3d::measurement::scalar)))
	{
		m_width.changed_signal().connect(sigc::mem_fun(*this, &bitmap_size::on_reset_bitmap));
		m_height.changed_signal().connect(sigc::mem_fun(*this, &bitmap_size::on_reset_bitmap));
		m_input.changed_signal().connect(sigc::mem_fun(*this, &bitmap_size::on_reset_bitmap));
	}

	void on_reset_bitmap(k3d::iunknown* const Hint)
	{
		m_output.reset(0, Hint);
	}

	void on_create_bitmap(k3d::bitmap& Bitmap)
	{
		// If we don't have any input bitmap, we're done ...
		const k3d::bitmap* const input = m_input.value();
		if(!input)
			return;

		// Cache properties ...
		const k3d::pixel_size_t width = m_width.value();
		const k3d::pixel_size_t height = m_height.value();

		// Crop and pad bitmap ...
		const long width_diff = width - input->width();
		const long height_diff = height - input->height();

		Bitmap = *input;
		k3d::pixel_size_t left, right, top, bottom;

		left = right = width_diff > 0 ? (width_diff/2) : 0;
		top = bottom = height_diff > 0 ? (height_diff/2) : 0;
		if (left != 0 || right != 0 || top != 0 || bottom != 0)
		{
			k3d::bitmap* const output_padded = new k3d::bitmap(left + Bitmap.width() + right, top + Bitmap.height() + bottom);
			bitmap_padding(Bitmap, *output_padded, left, right, top, bottom);
			Bitmap = *output_padded;
		}

		left = right = width_diff < 0 ? std::abs(width_diff/2) : 0;
		top = bottom = height_diff < 0 ? std::abs(height_diff/2) : 0;
		if (left != 0 || right != 0 || top != 0 || bottom != 0)
		{
			k3d::bitmap* const output_cropped = new k3d::bitmap(Bitmap.width() - left - right, Bitmap.height() - top - bottom);
			bitmap_croping(Bitmap, *output_cropped, left, right, top, bottom);
			Bitmap = *output_cropped;
		}
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<bitmap_size>,
			k3d::interface_list<k3d::ibitmap_source,
			k3d::interface_list<k3d::ibitmap_sink> > > factory(
				k3d::uuid(0xc8a70bc7, 0xeef04dc1, 0x8b8a2001, 0x6ab63b4a),
				"BitmapSize",
				_("Set bitmap size by cropping or padding"),
				"Bitmap",
				k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_data(long, immutable_name, change_signal, with_undo, local_storage, with_constraint, measurement_property, with_serialization) m_width;
	k3d_data(long, immutable_name, change_signal, with_undo, local_storage, with_constraint, measurement_property, with_serialization) m_height;
};

/////////////////////////////////////////////////////////////////////////////
// bitmap_size_factory

k3d::iplugin_factory& bitmap_size_factory()
{
	return bitmap_size::get_factory();
}

} // namespace libk3dbitmap


