// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Anders Dahnielson (anders@dahnielson.com)
*/

#include <k3dsdk/i18n.h>
#include <k3dsdk/module.h>

#include "basic_bitmap_modifier.h"

namespace libk3dbitmap
{

/////////////////////////////////////////////////////////////////////////////
// bitmap_multiply

class bitmap_multiply :
	public basic_bitmap_modifier
{
	typedef basic_bitmap_modifier base;

public:
	bitmap_multiply(k3d::idocument& Document) :
		base(Document),
		m_value(init_owner(*this) + init_name("value") + init_label(_("Multiplicand")) + init_description(_("Multiply each pixel component with this value")) + init_value(0.0))
	{
		m_value.changed_signal().connect(sigc::mem_fun(*this, &bitmap_multiply::on_value_change));
	}

	void on_value_change(k3d::iunknown* const Hint)
	{
		m_value_cache = m_value.value();

		on_reset_bitmap(Hint);
	}

        void on_filter(const k3d::pixel& I, k3d::pixel& O)
	{
		double I_red   = k3d::color_traits<double>::convert(I.red);
		double I_green = k3d::color_traits<double>::convert(I.green);
		double I_blue  = k3d::color_traits<double>::convert(I.blue);
		double I_alpha = k3d::color_traits<double>::convert(I.alpha);

		// Matte divide ...
		if (I_alpha != 0)
		{
			I_red   = I_red   / I_alpha;
			I_green = I_green / I_alpha;
			I_blue  = I_blue  / I_alpha;
		}

		// Color operation ...
		const double O_red   = I_red   * m_value_cache;
		const double O_green = I_green * m_value_cache;
		const double O_blue  = I_blue  * m_value_cache;

		// Matte multiply ...
		O.red   = k3d::bitmap::pixel_type::sample_traits::convert(O_red   * I_alpha);
		O.green = k3d::bitmap::pixel_type::sample_traits::convert(O_green * I_alpha);
		O.blue  = k3d::bitmap::pixel_type::sample_traits::convert(O_blue  * I_alpha);
		O.alpha = I.alpha;
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<bitmap_multiply>,
			k3d::interface_list<k3d::ibitmap_source,
			k3d::interface_list<k3d::ibitmap_sink> > > factory(
				k3d::uuid(0x03d2ac85, 0x37af4255, 0x956c0def, 0x82c3c753),
				"BitmapMultiply",
				_("Multiply value of each pixel"),
				"Bitmap",
				k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_data(double, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_value;

	double m_value_cache;
};

/////////////////////////////////////////////////////////////////////////////
// bitmap_multiply_factory

k3d::iplugin_factory& bitmap_multiply_factory()
{
	return bitmap_multiply::get_factory();
}

} // namespace libk3dbitmap


