#ifndef NGUI_VIEWPORT_H
#define NGUI_VIEWPORT_H

// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Tim Shead (tshead@k-3d.com)
*/

#include <gtkmm/drawingarea.h>

#include "panel.h"
#include "ui_component.h"

#include <k3dsdk/geometry.h>
#include <k3dsdk/gl.h>
#include <k3dsdk/ianimation_render_engine.h>
#include <k3dsdk/ipreview_render_engine.h>
#include <k3dsdk/istill_render_engine.h>
#include <k3dsdk/property_collection.h>
#include <k3dsdk/selection.h>
#include <k3dsdk/signal_system.h>

namespace Gdk { class GC; }
namespace Gtk { class Menu; }
namespace k3d { class inode; }
namespace k3d { class iselectable; }
namespace k3d { class vector2; }

namespace libk3dngui
{

class document_state;

namespace viewport
{

/////////////////////////////////////////////////////////////////////////////
// control

class control :
        public Gtk::DrawingArea,
	public ui_component,
	public k3d::property_collection,
	public k3d::istill_render_engine,
	public k3d::ianimation_render_engine,
	public panel::control
{
	typedef Gtk::DrawingArea base;

public:
	control(document_state& DocumentState, k3d::icommand_node& Parent);

	/// Returns the owning document
	k3d::idocument& document();
	/// Returns the camera for this viewport
	k3d::icamera* const camera();
	/// Returns the OpenGL render engine for this viewport
	k3d::gl::irender_engine* const gl_engine();
	/// Returns the preview render engine for this viewport
	k3d::ipreview_render_engine* const preview_engine();
	/// Returns the still render engine for this viewport
	k3d::istill_render_engine* const still_engine();
	/// Returns the animation render engine for this viewport
	k3d::ianimation_render_engine* const animation_engine();

	/// Sets the camera for this viewport
	void set_camera(k3d::icamera* const Camera);
	/// Sets the OpenGL render engine for this viewport
	void set_gl_engine(k3d::gl::irender_engine* const Engine);
	/// Sets the current preview render engine for this viewport
	void set_preview_engine(k3d::ipreview_render_engine* const Engine);
	/// Sets the current still render engine for this viewport
	void set_still_engine(k3d::istill_render_engine* const Engine);
	/// Sets the current animation render engine for this viewport
	void set_animation_engine(k3d::ianimation_render_engine* const Engine);

	/// Returns the viewport view matrix
	const k3d::matrix4 get_view_matrix();
	/// Sets the viewport view matrix
	void set_view_matrix(const k3d::matrix4& Matrix);
	/// Returns the viewport "target", the point (in world coordinates) around which the viewport orbits
	const k3d::vector3 get_target();
	/// Sets the viewport "target", the point (in world coordinates) around which the viewport orbits
	void set_target(const k3d::vector3& Target);
	/// Returns the viewport "up" axis, the axis (in world coordinates) that defines "up" for modeling purposes
	const k3d::normal3 get_up_axis();
	
	/// Returns the most recent OpenGL viewport parameters
	void get_gl_viewport(GLdouble ViewMatrix[16], GLdouble ProjectionMatrix[16], GLint Viewport[4]);
	/// Projects a point in world coordinates into screen space, returning the 2D widget coordinates
	const k3d::vector2 project(const k3d::vector3& WorldCoords);

	bool render_frame(k3d::icamera& Camera, const boost::filesystem::path& OutputImage, const bool ViewCompletedImage);
	bool render_animation(k3d::icamera& Camera, const boost::filesystem::path& OutputImages, const bool ViewCompletedImages);

	/// Returns all points contained in the given rectangle in widget coordinates
	k3d::selection::records get_selectable_points(const k3d::rectangle& SelectionRegion);
	/// Returns all lines that intersect the given rectangle in widget coordinates
	k3d::selection::records get_selectable_lines(const k3d::rectangle& SelectionRegion);
	/// Returns all faces that intersect the given rectangle in widget coordinates
	k3d::selection::records get_selectable_faces(const k3d::rectangle& SelectionRegion);
	/// Returns all nodes that intersect the given rectangle in widget coordinates
	k3d::selection::records get_selectable_nodes(const k3d::rectangle& SelectionRegion);
	/// Returns all objects (points, lines, faces, or nodes, depending on selection mode) that intersect the given rectangle in widget coordinates
	k3d::selection::records get_selectable_objects(const k3d::rectangle& SelectionRegion);

	/// Returns the closest point at the given widget coordinates (may return an empty record)
	k3d::selection::record pick_point(const k3d::vector2& Coordinates);
	/// Returns the closest line at the given widget coordinates (may return an empty record)
	k3d::selection::record pick_line(const k3d::vector2& Coordinates);
	/// Returns the closest face at the given widget coordinates (may return an empty record)
	k3d::selection::record pick_face(const k3d::vector2& Coordinates);
	/// Returns the closest node at the given widget coordinates (may return an empty record)
	k3d::selection::record pick_node(const k3d::vector2& Coordinates);
	/// Returns the closest object (point, line, face, or node, depending on selection mode) at the given coordinates (may return an empty record)
	k3d::selection::record pick_object(const k3d::vector2& Coordinates);

	/// Returns the closest point at the given widget coordinates (may return an empty record)
	k3d::selection::record pick_point(const k3d::vector2& Coordinates, k3d::selection::records& Records);
	/// Returns the closest line at the given widget coordinates (may return an empty record)
	k3d::selection::record pick_line(const k3d::vector2& Coordinates, k3d::selection::records& Records);
	/// Returns the closest face at the given widget coordinates (may return an empty record)
	k3d::selection::record pick_face(const k3d::vector2& Coordinates, k3d::selection::records& Records);
	/// Returns the closest node at the given widget coordinates (may return an empty record)
	k3d::selection::record pick_node(const k3d::vector2& Coordinates, k3d::selection::records& Records);
	/// Returns the closest object (point, line, face, or node, depending on selection mode) at the given coordinates (may return an empty record)
	k3d::selection::record pick_object(const k3d::vector2& Coordinates, k3d::selection::records& Records);

private:
	void on_camera_changed();
	void on_gl_engine_changed();

	void on_redraw_request(k3d::gl::irender_engine::redraw_type_t RedrawType);
	void on_style_changed(const Glib::RefPtr<Gtk::Style>& previous_style);
	bool on_redraw();
	const GLint select(const k3d::gl::select_state& SelectState, const k3d::rectangle& SelectionRegion);
	const GLint select(const k3d::gl::select_state& SelectState, const k3d::rectangle& SelectionRegion, GLdouble ViewMatrix[16], GLdouble ProjectionMatrix[16], GLint Viewport[4]);

	/// Returns an OpenGL selection as a collection of records
	const k3d::selection::records get_selection(const k3d::gl::select_state& SelectionState, const k3d::rectangle& SelectionRegion);
	/// Returns an OpenGL selection as a collection of records
	const k3d::selection::records get_selection(const k3d::gl::select_state& SelectionState, const k3d::rectangle& SelectionRegion, GLdouble ViewMatrix[16], GLdouble ProjectionMatrix[16], GLint Viewport[4]);

	/// Renders the current view to disk
	bool save_frame(k3d::icamera& Camera, const boost::filesystem::path& OutputImage, const bool ViewCompletedImage);
	/// Caches the current font as an OpenGL display list
	void create_font();

	/// Stores the "target", a point (in world coordinates) around which the viewport orbits.  Also affects navigation sensitivity.
	k3d::vector3 m_target;
	/// Called when a mouse button is pressed
	bool on_button_press_event(GdkEventButton* Event);
	/// Called when a mouse button is released
	bool on_button_release_event(GdkEventButton* Event);
	/// Called when the mouse wheel is scrolled
	bool on_scroll_event(GdkEventScroll* Event);
	/// Called when the mouse is moved
	bool on_motion_notify_event(GdkEventMotion* Event);
	/// Called when a key is pressed
	bool on_key_press_event(GdkEventKey* Event);
	/// Called when a key is released
	bool on_key_release_event(GdkEventKey* Event);

	/// Stores a reference to the owning document
	document_state& m_document_state;
	/// Stores a reference to the current camera
	k3d_data(k3d::icamera*, no_name, change_signal, no_undo, node_storage, no_constraint, no_property, no_serialization) m_camera;
	/// Stores a reference to the current OpenGL render engine
	k3d_data(k3d::gl::irender_engine*, no_name, change_signal, no_undo, node_storage, no_constraint, no_property, no_serialization) m_gl_engine;
	/// Stores a reference to the current preview render engine
	k3d_data(k3d::ipreview_render_engine*, no_name, change_signal, no_undo, node_storage, no_constraint, no_property, no_serialization) m_preview_engine;
	/// Stores a reference to the current still render engine
	k3d_data(k3d::istill_render_engine*, no_name, change_signal, no_undo, node_storage, no_constraint, no_property, no_serialization) m_still_engine;
	/// Stores a reference to the current animation render engine
	k3d_data(k3d::ianimation_render_engine*, no_name, change_signal, no_undo, node_storage, no_constraint, no_property, no_serialization) m_animation_engine;

	/// Stores the current set of OpenGL font glyphs (generated from Pango by gtkglext)
	unsigned long m_font_begin;
	/// Stores the current set of OpenGL font glyphs (generated from Pango by gtkglext)
	unsigned long m_font_end;

	/// Stores a connection to the attached OpenGL render engine redraw request signal
	sigc::connection m_gl_engine_redraw_request_connection;

	/// Buffers OpenGL hit records
	typedef std::vector<GLuint> gl_selection_buffer_t;
	gl_selection_buffer_t m_selection_buffer;

	// Buffers parameters from the most-recent render
	GLdouble m_gl_view_matrix[16];
	GLdouble m_gl_projection_matrix[16];
	GLint m_gl_viewport[4];
};

} // namespace viewport

/////////////////////////////////////////////////////////////////////////////
// widget_to_ndc

/// Converts widget coordinates to normalized camera coordinates - note: results may be outside the range [0, 1] because the viewport and camera aspect ratios may not match
const k3d::vector2 widget_to_ndc(viewport::control& Viewport, const k3d::vector2& WidgetCoords);

/////////////////////////////////////////////////////////////////////////////
// ndc_to_widget

/// Converts normalized camera coordinates to widget coordinates
const k3d::vector2 ndc_to_widget(viewport::control& Viewport, const k3d::vector2& NDC);

/////////////////////////////////////////////////////////////////////////////
// mouse_to_world

/// Converts widget coordinates to a line in world coordinates
const k3d::line3 mouse_to_world(viewport::control& Viewport, const k3d::vector2& WidgetCoords);

} // namespace libk3dngui

#endif // NGUI_VIEWPORT_H


