// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Tim Shead (tshead@k-3d.com)
*/

#include "button.h"
#include "file_selection.h"
#include "hotkey_entry.h"
#include "interactive.h"
#include "path_chooser.h"

#include <k3dsdk/i18n.h>
#include <k3dsdk/istate_recorder.h>
#include <k3dsdk/state_change_set.h>

#include <gtkmm/combobox.h>
#include <gtkmm/entry.h>
#include <gtkmm/liststore.h>

namespace libk3dngui
{

namespace path_chooser
{

/////////////////////////////////////////////////////////////////////////////
// control

control::control(k3d::icommand_node& Parent, const std::string& Name, std::auto_ptr<idata_proxy> Data) :
	base(false, 0),
	ui_component(Name, &Parent),
	m_entry(new hotkey_entry),
	m_button(new Gtk::Button("...")),
	m_combo(new Gtk::ComboBox()),
	m_data(Data)
{
	m_entry->signal_focus_out_event().connect(sigc::mem_fun(*this, &control::on_focus_out_event));
	m_entry->signal_activate().connect(sigc::mem_fun(*this, &control::on_activate));

	tooltips().set_tip(*m_button, _("Browse for a file ..."));
	m_button->signal_clicked().connect(sigc::mem_fun(*this, &control::on_browse));

	Glib::RefPtr<Gtk::ListStore> model = Gtk::ListStore::create(m_columns);
	Gtk::TreeModel::Row row;
	row = *model->append();
		row[m_columns.reference] = k3d::ipath_property::ABSOLUTE_REFERENCE;
		row[m_columns.label] = _("Absolute");
	row = *model->append();
		row[m_columns.reference] = k3d::ipath_property::RELATIVE_REFERENCE;
		row[m_columns.label] = _("Relative");
	row = *model->append();
		row[m_columns.reference] = k3d::ipath_property::INLINE_REFERENCE;
		row[m_columns.label] = _("Inline");
		
	m_combo->set_model(model);
	m_combo->pack_start(m_columns.label);
	tooltips().set_tip(*m_combo, _("Choose whether to store absolute or relative filepaths"));
	
	m_combo->signal_changed().connect(sigc::mem_fun(*this, &control::on_pick_reference_type));
	
	pack_start(*manage(m_entry), Gtk::PACK_EXPAND_WIDGET);
	pack_start(*manage(m_button), Gtk::PACK_SHRINK);
	pack_start(*manage(m_combo), Gtk::PACK_SHRINK);

	data_changed();

	if(m_data.get())
		m_data->changed_signal().connect(sigc::mem_fun(*this, &control::data_changed));

	show_all();
}

bool control::execute_command(const std::string& Command, const std::string& Arguments)
{
	if(Command == "browse")
		{
			interactive::activate(*m_button);
			return true;
		}
	else if(Command == "set_value")
		{
			interactive::set_text(*m_entry, Arguments);
			m_entry->select_region(0, 0);
			set_value();
			return true;
		}

	return ui_component::execute_command(Command, Arguments);
}

bool control::on_focus_out_event(GdkEventFocus* Event)
{
	set_value();
	return false;
}

void control::on_activate()
{
	set_value();
}

void control::on_browse()
{
	return_if_fail(m_data.get());

	// Record the command for posterity (tutorials) ...
	record_command("browse");

	boost::filesystem::path new_path;
	if(!get_file_path(m_data->mode(), m_data->type(), "Prompt", m_data->value(), new_path))
		return;

	// Turn this into an undo/redo -able event ...
	if(m_data->state_recorder)
		m_data->state_recorder->start_recording(k3d::create_state_change_set());

	// Update everything with the new value ...
	m_data->set_value(new_path);

	// Turn this into an undo/redo -able event ...
	if(m_data->state_recorder)
		m_data->state_recorder->commit_change_set(m_data->state_recorder->stop_recording(), m_data->change_message + " " + new_path.native_file_string());
}

void control::on_pick_reference_type()
{
	return_if_fail(m_data.get());
	return_if_fail(m_combo->get_active() != m_combo->get_model()->children().end());

	m_data->set_reference(m_combo->get_active()->get_value(m_columns.reference));
}

void control::set_value()
{
	return_if_fail(m_data.get());

	// If the value didn't change, we're done ...
	const std::string new_value = m_entry->get_text();
	if(new_value == m_data->value().native_file_string())
		return;

	// Record the command for posterity (tutorials) ...
	record_command("set_value", new_value);

	// Turn this into an undo/redo -able event ...
	if(m_data->state_recorder)
		m_data->state_recorder->start_recording(k3d::create_state_change_set());

	// Update everything with the new value ...
	m_data->set_value(boost::filesystem::path(new_value, boost::filesystem::native));

	// Turn this into an undo/redo -able event ...
	if(m_data->state_recorder)
		m_data->state_recorder->commit_change_set(m_data->state_recorder->stop_recording(), m_data->change_message + " " + new_value);
}

void control::data_changed()
{
	return_if_fail(m_data.get());
	
	m_entry->set_text(m_data->value().leaf());
	tooltips().set_tip(*m_entry, m_data->value().native_file_string());
	
	m_combo->set_active(m_data->reference());
}

void control::on_reference_type_changed()
{
	m_combo->set_active(m_data->reference());
}

} // namespace path_chooser

} // namespace libk3dngui


