// K-3D
// Copyright (c) 2005, Romain Behar
//
// Contact: romainbehar@yahoo.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "file_selection.h"
#include "user_interface.h"

#include <k3dsdk/i18n.h>
#include <k3dsdk/ideletable.h>
#include <k3dsdk/idocument.h>
#include <k3dsdk/idocument_plugin_factory.h>
#include <k3dsdk/options.h>
#include <k3dsdk/persistent_lookup.h>
#include <k3dsdk/plugins.h>
#include <k3dsdk/xml.h>

#include <k3dsdk/fstream.h>
#include <boost/filesystem/path.hpp>

#include <gtkmm/dialog.h>
#include <gtkmm/label.h>
#include <gtkmm/liststore.h>
#include <gtkmm/scrolledwindow.h>
#include <gtkmm/stock.h>
#include <gtkmm/treeview.h>

namespace libk3dngui
{

namespace merge_nodes
{

namespace detail
{

// Merge nodes UI
struct node_check_t
{
	node_check_t(const std::string& Name) :
		name(Name)
	{
	}

	std::string name;
	bool check;
};

typedef std::vector<node_check_t> document_nodes_t;

bool merge_nodes_dialog(document_nodes_t& DocumentNodes)
{
	class columns_t :
		public Gtk::TreeModel::ColumnRecord
	{
	public:
		columns_t()
		{
			add(name);
			add(check);
		}

		Gtk::TreeModelColumn<Glib::ustring> name;
		Gtk::TreeModelColumn<bool> check;
	};

	columns_t nodes;
	Glib::RefPtr<Gtk::ListStore> model = Gtk::ListStore::create(nodes);

	for(document_nodes_t::const_iterator node = DocumentNodes.begin(); node != DocumentNodes.end(); ++node)
		{
			Gtk::TreeModel::Row row = *(model->append());
			return_val_if_fail(row, false);

			row[nodes.name] = node->name;
			row[nodes.check] = false;
		}

	Gtk::TreeView treeview(model);
	treeview.append_column_editable("Node", nodes.name);
	treeview.append_column_editable("merge", nodes.check);

	Gtk::ScrolledWindow scrolled_window;
	scrolled_window.add(treeview);
	scrolled_window.set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);

	Gtk::Dialog dialog("Choose nodes to merge", true);
	dialog.set_default_size(500, 350);
	dialog.set_border_width(5);
	dialog.get_vbox()->pack_start(*Gtk::manage(new Gtk::Label("Check nodes :")), Gtk::PACK_SHRINK, 5);
	dialog.get_vbox()->pack_start(scrolled_window, Gtk::PACK_EXPAND_WIDGET);
	dialog.add_button(Gtk::Stock::OK, Gtk::RESPONSE_OK);
	dialog.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
	dialog.show_all();

	if(Gtk::RESPONSE_OK != dialog.run())
		return false;

	Gtk::TreeNodeChildren rows = model->children();
	document_nodes_t::iterator node = DocumentNodes.begin();
	for(Gtk::TreeNodeChildren::const_iterator row = rows.begin(); row != rows.end(); ++row)
		{
			return_val_if_fail(node != DocumentNodes.end(), false);

			node->check = (*row)[nodes.check];
			++node;
		}

	return true;
}

} // namespace detail

void merge_nodes(k3d::idocument& Document)
{
	// Query K-3D document
	boost::filesystem::path document_path;
	if(!get_file_path(k3d::ipath_property::READ, k3d::options::path::documents(), _("Merge K-3D nodes:"), boost::filesystem::path(), document_path))
		return;

	// Check for a K-3D document
	k3d::xml::element xml("k3dml");
	try
	{
		k3d::filesystem::ifstream stream(document_path);
		k3d::xml::hide_progress progress;
		parse(xml, stream, document_path.native_file_string(), progress);
	}
	catch(std::exception& e)
	{
		k3d::log() << error << e.what() << std::endl;
		return;
	}

	return_if_fail(xml.name == "k3dml");

	// Setup context
	const boost::filesystem::path root_path = document_path.branch_path();
	k3d::persistent_lookup persistent_lookup;
	k3d::ipersistent::load_context context(root_path, persistent_lookup);

	// Get document node list
	k3d::xml::element* const xml_document = k3d::xml::find_element(xml, "document");
	return_if_fail(xml_document);

	k3d::xml::element* xml_nodes = k3d::xml::find_element(*xml_document, "nodes");
	if(!xml_nodes)
		xml_nodes = k3d::xml::find_element(*xml_document, "objects");

	return_if_fail(xml_nodes);

	if(xml_nodes->name != "nodes")
		k3d::log() << warning << "Loading data from obsolete <" << xml_nodes->name << "> tag ... re-save document to eliminate this warning" << std::endl;

	detail::document_nodes_t document_nodes;
	for(k3d::xml::element::elements_t::iterator xml_node = xml_nodes->children.begin(); xml_node != xml_nodes->children.end(); ++xml_node)
		{
			if(xml_node->name != "node" && xml_node->name != "object")
				continue;

			if(k3d::xml::attribute_value<bool>(*xml_node, "do_not_load", false))
				continue;

			const std::string name = k3d::xml::attribute_text(*xml_node, "name");
			document_nodes.push_back(detail::node_check_t(name));
		}

	// Run node list UI
	if(!detail::merge_nodes_dialog(document_nodes))
		return;

	// Load checked nodes
	k3d::inode_collection::nodes_t nodes;
	std::vector<k3d::ipersistent*> persistent_nodes;
	std::vector<k3d::xml::element*> node_storage;

	detail::document_nodes_t::const_iterator user_node = document_nodes.begin();
	for(k3d::xml::element::elements_t::iterator xml_node = xml_nodes->children.begin(); xml_node != xml_nodes->children.end(); ++xml_node, ++user_node)
		{
			return_if_fail(user_node != document_nodes.end());

			if(xml_node->name != "node" && xml_node->name != "object")
				continue;

			if(k3d::xml::attribute_value<bool>(*xml_node, "do_not_load", false))
				continue;

			if(xml_node->name != "node")
				k3d::log() << warning << "Loading data from obsolete <" << xml_node->name << "> tag ... re-save document to eliminate this warning" << std::endl;

			const std::string name = attribute_text(*xml_node, "name");
			return_if_fail(name == user_node->name);

			// Skip unchecked nodes
			if(!user_node->check)
				continue;

			const k3d::uuid class_id = k3d::xml::attribute_value<k3d::uuid>(*xml_node, "class", k3d::uuid::null());
			if(class_id == k3d::uuid::null())
			{
				k3d::log() << error << "node [" << name << "] with unspecified class ID will not be loaded" << std::endl;
				continue;
			}

			const k3d::ipersistent_lookup::id_type node_id = k3d::xml::attribute_value<k3d::ipersistent_lookup::id_type>(*xml_node, "id", 0);
			if(node_id == 0)
			{
				k3d::log() << error << "node [" << name << "] with unspecified ID will not be loaded" << std::endl;
				continue;
			}

			k3d::iplugin_factory* const plugin_factory = k3d::plugin(class_id);
			if(!plugin_factory)
			{
				k3d::log() << error << "node [" << name << "] with unknown class ID [" << class_id << "] will not be loaded" << std::endl;
				continue;
			}

			k3d::idocument_plugin_factory* const document_plugin_factory = dynamic_cast<k3d::idocument_plugin_factory*>(plugin_factory);
			if(!document_plugin_factory)
			{
				k3d::log() << error << "Non-document plugin [" << name << "] will not be loaded" << std::endl;
				continue;
			}

			k3d::inode* const node = document_plugin_factory->create_plugin(Document);
			if(!node)
			{
				k3d::log() << error << "Error creating node [" << name << "] instance" << std::endl;
				continue;
			}

			k3d::ipersistent* const persistent = dynamic_cast<k3d::ipersistent*>(node);
			if(!persistent)
			{
				k3d::log() << error << "node [" << name << "] does not support persistence" << std::endl;

				delete dynamic_cast<k3d::ideletable*>(node);
				continue;
			}

			k3d::undoable_new(dynamic_cast<k3d::ideletable*>(node), Document);

			nodes.push_back(node);
			persistent_nodes.push_back(persistent);
			node_storage.push_back(&(*xml_node));

			persistent_lookup.insert_lookup(node_id, node);
		}

	Document.nodes().add_nodes(nodes);

	for(unsigned long i = 0; i != persistent_nodes.size(); ++i)
		persistent_nodes[i]->load(*node_storage[i], context);
}

} // namespace merge_nodes

} // namespace libk3dngui


