// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Implements functions and helpers for simulating user input and driving interactive tutorials
		\author Tim Shead (tshead@k-3d.com)
*/

#include "collapsible_frame.h"
#include "interactive.h"
#include "options.h"
#include "screen_overlay.h"
#include "utility.h"

#include <k3dsdk/application.h>
#include <k3dsdk/basic_math.h>
#include <k3dsdk/bezier.h>
#include <k3dsdk/geometry.h>
#include <k3dsdk/high_res_timer.h>
#include <k3dsdk/system.h>
#include <k3dsdk/vectors.h>

#include <gtkmm/button.h>
#include <gtkmm/checkbutton.h>
#include <gtkmm/combobox.h>
#include <gtkmm/entry.h>
#include <gtkmm/menu.h>
#include <gtkmm/menuitem.h>
#include <gtkmm/notebook.h>
#include <gtkmm/scrolledwindow.h>
#include <gtkmm/treeview.h>
#include <gtkmm/widget.h>
#include <gtkmm/window.h>

#if defined K3D_PLATFORM_WIN32

	#include <gdkwin32.h>
	#undef min
	#undef max

#else // K3D_PLATFORM_WIN32

	#include <gdk/gdkx.h>

#endif // !K3D_PLATFORM_WIN32

namespace libk3dngui
{

namespace interactive
{

namespace detail
{

void warp_pointer(const k3d::vector2& Offset)
{
#ifdef K3D_PLATFORM_WIN32

	// Make that pointer jump!
	SetCursorPos(static_cast<int>(Offset[0]), static_cast<int>(Offset[1]));

#else // K3D_PLATFORM_WIN32

	// Get the X display ...
	Display* xdisplay = GDK_WINDOW_XDISPLAY(Gdk::Display::get_default()->get_default_screen()->get_root_window()->gobj());
	return_if_fail(xdisplay);

	// Get our X window ...
	Window xwindow = GDK_WINDOW_XWINDOW(Gdk::Display::get_default()->get_default_screen()->get_root_window()->gobj());
	return_if_fail(xwindow);

	// Move that pointer!
	XWarpPointer(xdisplay, None, xwindow, 0, 0, 0, 0, static_cast<int>(Offset[0]), static_cast<int>(Offset[1]));
	XFlush(xdisplay);

#endif // !K3D_PLATFORM_WIN32
}

void warp_pointer(Glib::RefPtr<Gdk::Window> Window, const k3d::vector2& Offset)
{
	return_if_fail(Window);
	
	int left = 0;
	int top = 0;
	Window->get_origin(left, top);

	warp_pointer(k3d::vector2(left, top) + Offset);
}

/// Moves the mouse pointer from its current location to the given screen coordinates
void move_pointer(const k3d::vector2& Offset, const double Speed, const bool ManhattanStyle)
{
	return_if_fail(Speed);

	// Get the current mouse pointer position in screen coordinates
	int pointerx = 0;
	int pointery = 0;
	Gdk::ModifierType modifiers;
	Gdk::Display::get_default()->get_pointer(pointerx, pointery, modifiers);

	// Make it our starting point ...
	const k3d::vector2 from(pointerx, pointery);

	// Setup our end point ...
	const k3d::vector2 to = Offset;

	// Calculate the number of steps, based on the distance to travel (then adjust, based on our speed) ...
	const unsigned long steps = static_cast<unsigned long>((((from - to).Length() / 20) + 30) / Speed);
	const double delta = 1.0 / static_cast<double>(steps);
	const unsigned long delay = 10;

	// Setup some sloppiness ...
	const double sloppiness = 75;
	const double randomscale = RAND_MAX * 0.5;
	const k3d::vector2 slop = k3d::vector2((rand() - randomscale) / randomscale, (rand() - randomscale) / randomscale) * sloppiness;

	// Setup a Bezier curve for our path ...
	std::vector<k3d::vector2> pathpoints;
	pathpoints.push_back(from);
	if(ManhattanStyle)
	{
		pathpoints.push_back(k3d::vector2(to[0], from[1]));
		pathpoints.push_back(k3d::vector2(to[0], from[1]));
	}
	pathpoints.push_back(to);

	// Setup a Bezier curve for some nice non-linear motion ...
	std::vector<double> ratepoints;
	ratepoints.push_back(0);
	ratepoints.push_back(0.1);
	ratepoints.push_back(0.9);
	ratepoints.push_back(1);

	for(unsigned long i = 1; i <= steps; ++i)
	{
		const double percent = k3d::bezier<3, double>(ratepoints.begin(), ratepoints.end(), delta * i);

		k3d::vector2 actualposition;
		if(ManhattanStyle)
		{
			actualposition = k3d::bezier<3, k3d::vector2>(pathpoints.begin(), pathpoints.end(), percent);
		}
		else
		{
			const k3d::vector2 sloppyto = k3d::mix(to+slop, to, percent);
			actualposition = k3d::mix(from, sloppyto, percent);
		}

		detail::warp_pointer(actualposition);

		non_blocking_sleep(delay);
	}

	non_blocking_sleep(static_cast<unsigned long>(500.0 / Speed));
}

void move_pointer(Glib::RefPtr<Gdk::Window> Window, const k3d::vector2& Offset, const double Speed, const bool ManhattanStyle)
{
	return_if_fail(Window);
	
	int left = 0;
	int top = 0;
	Window->get_origin(left, top);

	move_pointer(k3d::vector2(left, top) + Offset, Speed, ManhattanStyle);
}


void highlight_coordinates(const double Theta, const double Radius, const double XRadius, const double YRadius, const double XCenter, const double YCenter, gint& X, gint& Y)
{
	X = static_cast<gint>(XCenter + (Radius * XRadius * cos(Theta)));
	Y = static_cast<gint>(YCenter + (Radius * YRadius * -sin(Theta)));
}

const k3d::vector2 screen_coordinates(Gtk::Widget& Widget)
{
	int left = 0;
	int top = 0;
	Widget.get_window()->get_origin(left, top);

	if(Widget.has_no_window())
	{
		left += Widget.get_allocation().get_x();
		top += Widget.get_allocation().get_y();
	}
	
	return k3d::vector2(left, top);
}

void move_pointer(Gtk::Widget& Widget, const double XPercent, const double YPercent, const double Speed, const bool ManhattanStyle)
{
	const int width = Widget.get_width();
	const int height = Widget.get_height();
	detail::move_pointer(screen_coordinates(Widget) + k3d::vector2(width * XPercent, height * YPercent), Speed, ManhattanStyle);

/*
	if(Widget.has_no_window())
	{
		xoffset += Widget.get_allocation().get_x();
		yoffset += Widget.get_allocation().get_y();

k3d::log() << debug << "offset after: " << xoffset << ", " << yoffset << std::endl;
	
		Gtk::Menu* const menu = dynamic_cast<Gtk::Menu*>(&Widget);
		if(menu)
			detail::move_pointer(menu->get_attach_widget()->get_window(), xoffset, yoffset, Speed, ManhattanStyle);
		else
		{
			detail::warp_pointer(k3d::vector2(xoffset, yoffset));
//			detail::move_pointer(Widget.get_parent_window(), xoffset, yoffset, Speed, ManhattanStyle);
		}
	}
	else
	{
		detail::move_pointer(Widget.get_window(), xoffset, yoffset, Speed, ManhattanStyle);
	}
*/
}

void move_pointer(Gtk::Widget& Widget, const double Speed, const bool ManhattanStyle)
{
	move_pointer(Widget, 0.5, 0.5, Speed, ManhattanStyle);
}

} // namespace detail

void warp_pointer(const k3d::vector2& Coords)
{
	detail::warp_pointer(Coords);
}

void warp_pointer(Gtk::Widget& Widget, const k3d::vector2& Coords)
{
	detail::warp_pointer(detail::screen_coordinates(Widget) + Coords);
}

void warp_pointer(Gtk::Widget& Widget, const k3d::vector2& Coords, const double Timestamp, const k3d::timer& Timer)
{
	handle_pending_events();
	k3d::system::sleep(Timestamp - Timer.elapsed());
	detail::warp_pointer(detail::screen_coordinates(Widget) + Coords);
}

void warp_pointer(Gtk::Widget& Widget)
{
	detail::warp_pointer(detail::screen_coordinates(Widget) + k3d::vector2(Widget.get_width() / 2, Widget.get_height() / 2));
}

void move_pointer(Gtk::Widget& Widget)
{
	detail::move_pointer(Widget, 0.5, 0.5, options::tutorial_speed(), false);
}

void move_pointer(Gtk::Widget& Widget, const k3d::vector2& Coords)
{
	detail::move_pointer(detail::screen_coordinates(Widget) + Coords, options::tutorial_speed(), false);
}

void show(Gtk::Widget& Widget)
{
	const double speed = options::tutorial_speed();
	return_if_fail(speed);

	// Build a list of parent widgets, starting with our "leaf" widget ...
	typedef std::deque<Gtk::Widget*> parents_t;
	parents_t parents;
	for(Gtk::Widget* parent = &Widget; parent; )
	{
		parents.push_front(parent);

		Gtk::Menu* const menu = dynamic_cast<Gtk::Menu*>(parent);
		if(menu)
			parent = menu->get_attach_widget();
		else
			parent = parent->get_parent();
	}

	// The top-level item ought to be a window, so make sure it's visible ...
	if(dynamic_cast<Gtk::Window*>(parents.front()))
	{
		parents.front()->get_window()->show();
		parents.front()->get_window()->raise();
	}

	// Keep track of scrolled windows as we go, so we can make the children visible as-required ...
	Gtk::ScrolledWindow* parent_scrolled_window = 0;

	// Walk the list, modifying widgets as needed ...
	bool topmenu = true;
	for(parents_t::iterator widget = parents.begin(); widget != parents.end(); ++widget)
	{
		// Keep track of scrolled windows as we go ...
		Gtk::ScrolledWindow* const scrolled_window = dynamic_cast<Gtk::ScrolledWindow*>(*widget);
		if(scrolled_window)
		{
			parent_scrolled_window = scrolled_window;
			continue;
		}

		const bool last_item = parents.end() == widget+1;
		Gtk::TreeView* const tree_view = dynamic_cast<Gtk::TreeView*>(*widget);
		Gtk::MenuItem* const menu_item = dynamic_cast<Gtk::MenuItem*>(*widget);
		Gtk::Notebook* const notebook = dynamic_cast<Gtk::Notebook*>(*widget);
		collapsible_frame::control* const collapsible_frame = dynamic_cast<collapsible_frame::control*>(*widget);

		// If this is a child of a scrolled window, ensure that it is visible ...
		if(parent_scrolled_window && (!tree_view) && (last_item || menu_item || notebook || (collapsible_frame && collapsible_frame->is_collapsed())))
		{
			Gtk::Adjustment* const hadj = parent_scrolled_window->get_hadjustment();
			Gtk::Adjustment* const vadj = parent_scrolled_window->get_vadjustment();

			const k3d::rectangle viewport_allocation(
				hadj->get_value(),
				hadj->get_value() + hadj->get_page_size(),
				vadj->get_value(),
				vadj->get_value() + vadj->get_page_size());

			const k3d::rectangle child_allocation(
				(*widget)->get_allocation().get_x(),
				(*widget)->get_allocation().get_x() + (*widget)->get_allocation().get_width(),
				(*widget)->get_allocation().get_y(),
				(*widget)->get_allocation().get_y() + (*widget)->get_allocation().get_height());

			const k3d::vector2 start(hadj->get_value(), vadj->get_value());

			k3d::vector2 end = start;
			if(child_allocation.right > viewport_allocation.right)
				end[0] = start[0] + (child_allocation.right - viewport_allocation.right);
			if(child_allocation.left < viewport_allocation.left)
				end[0] = start[0] - (viewport_allocation.left - child_allocation.left);
			if(child_allocation.bottom > viewport_allocation.bottom)
				end[1] = start[1] + (child_allocation.bottom - viewport_allocation.bottom);
			if(child_allocation.top < viewport_allocation.top)
				end[1] = start[1] - (viewport_allocation.top - child_allocation.top);

			end[0] = std::min(end[0], hadj->get_upper() - hadj->get_page_size());
			end[1] = std::min(end[1], vadj->get_upper() - vadj->get_page_size());

			const int count = 50;
			for(int i = 0; i != count; ++i)
			{
				const k3d::vector2 position = k3d::mix(start, end, (i+1) / static_cast<double>(count));
				parent_scrolled_window->get_hadjustment()->set_value(position[0]);
				parent_scrolled_window->get_vadjustment()->set_value(position[1]);
				handle_pending_events();
			}
		}

		// Open menus ...
		if(menu_item)
		{
			if(topmenu)
			{
				detail::move_pointer(**widget, 0.25, 0.5, speed, false);
				topmenu = false;
			}
			else
			{
				detail::move_pointer(**widget, 0.25, 0.5, speed, true);
			}

			menu_item->select();
			handle_pending_events();
			non_blocking_sleep(250);
			continue;
		}

		// Switch notebook tabs ...
		if(notebook)
		{
			parents_t::iterator notebook_child = widget + 1;
			if(notebook_child == parents.end())
				continue;

			const int page_number = notebook->page_num(**notebook_child);
			if(page_number == notebook->get_current_page())
				continue;

			Gtk::Widget* const tab_label = notebook->get_tab_label(**notebook_child);
			return_if_fail(tab_label);

			detail::move_pointer(*tab_label, 0.5, 0.5, speed, false);
			notebook->set_current_page(page_number);
			continue;
		}

		// Expand frames ...
		if(collapsible_frame)
		{
			if(collapsible_frame->is_collapsed())
			{
				const unsigned long delay = static_cast<unsigned long>(500 / speed);

				detail::move_pointer(collapsible_frame->toggle_button(), 0.5, 0.5, speed, false);

				collapsible_frame->toggle_button().pressed();
				handle_pending_events();
				non_blocking_sleep(delay);

				collapsible_frame->toggle_button().released();
				handle_pending_events();
				non_blocking_sleep(delay);
			}

			continue;
		}
	}
}

void highlight(Gtk::Widget& Widget)
{
	const double speed = options::tutorial_speed();
	return_if_fail(speed);

	show(Widget);

	// Get widget position relative to the root window ...
	int width = Widget.get_width();
	int height = Widget.get_height();
	int left = 0;
	int top = 0;
	Widget.get_window()->get_origin(left, top);

	if(Widget.has_no_window())
	{
		left += Widget.get_allocation().get_x();
		top += Widget.get_allocation().get_y();
	}

	const unsigned long samples = 400;

	// Adjust the rate at which we update the display, based on the size of the widget,
	// because making changes to the overlay will get slower as the overlay gets larger
	const unsigned long update_rate = static_cast<unsigned long>(std::max(1.0, sqrt(width * height / 2000.0)));

	const double start_angle = 130;
	const double end_angle = start_angle + 360 + 30;
	const double start_radius = 0.87;
	const double end_radius = 0.95;

	const int line_width = 6;
	left -= 2 * line_width;
	top -= 2 * line_width;
	width += 4 * line_width;
	height += 4 * line_width;

	screen_overlay overlay(Gdk::Rectangle(left, top, width, height), k3d::color(1, 0, 0));
	overlay.show_all();

	const double xradius = width * 0.5 - line_width;
	const double yradius = height * 0.5 - line_width;
	const double xcenter = width * 0.5;
	const double ycenter = height * 0.5;

	int x, y = 0;
	detail::highlight_coordinates(k3d::radians(start_angle), start_radius, xradius, yradius, xcenter, ycenter, x, y);
	detail::move_pointer(overlay.get_window(), k3d::vector2(x, y), speed, false);

	overlay.mask_gc()->set_line_attributes(line_width, Gdk::LINE_SOLID, Gdk::CAP_ROUND, Gdk::JOIN_ROUND);
	for(unsigned long i = 0, update = 0; i != samples; ++i, ++update)
	{
		const double percent1 = i / static_cast<double>(samples);
		const double percent2 = (i + 1) / static_cast<double>(samples);

		int x1, y1 = 0;
		const double radius1 = k3d::mix(start_radius, end_radius, percent1);
		detail::highlight_coordinates(k3d::radians(k3d::mix(start_angle, end_angle, percent1)), radius1, xradius, yradius, xcenter, ycenter, x1, y1);

		int x2, y2 = 0;
		const double radius2 = k3d::mix(start_radius, end_radius, percent2);
		detail::highlight_coordinates(k3d::radians(k3d::mix(start_angle, end_angle, percent2)), radius2, xradius, yradius, xcenter, ycenter, x2, y2);

		detail::warp_pointer(overlay.get_window(), k3d::vector2(x2, y2));
		overlay.mask()->draw_line(overlay.mask_gc(), x1, y1, x2, y2);

		if(0 == update % update_rate)
			overlay.update();

//			non_blocking_sleep(static_cast<unsigned long>(100.0 / speed));
	}

	non_blocking_sleep(static_cast<unsigned long>(750.0 / speed));

	// Special-case menu items, closing the menu so it doesn't stick-around forever ...
	if(dynamic_cast<Gtk::MenuItem*>(&Widget))
	{
		typedef std::vector<Gtk::MenuItem*> menu_items_t;
		menu_items_t menu_items;

		for(Gtk::Widget* ancestor = &Widget; ancestor; )
		{
			Gtk::MenuItem* const menu_item = dynamic_cast<Gtk::MenuItem*>(ancestor);
			if(menu_item)
				menu_items.push_back(menu_item);

			Gtk::Menu* const menu = dynamic_cast<Gtk::Menu*>(ancestor);
			if(menu)
				ancestor = menu->get_attach_widget();
			else
				ancestor = ancestor->get_parent();
		}

		for(menu_items_t::iterator menu_item = menu_items.begin(); menu_item != menu_items.end(); ++menu_item)
			(*menu_item)->deselect();
	}
}

void set_text(Gtk::Entry& Entry, const std::string& Text)
{
	const double speed = options::tutorial_speed();
	return_if_fail(speed);

	show(Entry);
	detail::move_pointer(Entry, 0.1, 0.5, speed, false);

	Entry.grab_focus();
	Entry.set_position(-1);
	Entry.select_region(0, -1);
	handle_pending_events();
	non_blocking_sleep(static_cast<unsigned long>(500 / speed));

	Entry.delete_text(0, -1);
	handle_pending_events();
	non_blocking_sleep(static_cast<unsigned long>(500 / speed));

	for(unsigned int i = 0; i < Text.size(); ++i)
	{
		int position = i;
		Entry.insert_text(Glib::ustring(1, Text[i]), 1, position);
		handle_pending_events();
		non_blocking_sleep(static_cast<unsigned long>((25 + (rand() % 150)) / speed));
	}

	handle_pending_events();
	non_blocking_sleep(static_cast<unsigned long>(500 / speed));

	GdkEventKey event;
	event.type = GDK_KEY_PRESS;
	event.window = static_cast<Gtk::Widget&>(Entry).gobj()->window;
	event.send_event = TRUE;
	event.time= GDK_CURRENT_TIME;
	event.state = 0;
	event.keyval = GDK_Tab;
	event.length = 1;
	event.string = "\t";

	Gdk::Event(reinterpret_cast<GdkEvent*>(&event)).put();
}

void activate(Gtk::Button& Button)
{
	const double speed = options::tutorial_speed();
	return_if_fail(speed);

	show(Button);
	move_pointer(Button);

	const unsigned long delay = static_cast<unsigned long>(500 / speed);

	if(!Button.is_sensitive())
		return;

	Button.pressed();
	handle_pending_events();
	non_blocking_sleep(delay);

	Button.released();
	handle_pending_events();
	non_blocking_sleep(delay);
}

void activate(Gtk::MenuItem& MenuItem)
{
	static const unsigned long delay = 500;

	if(!MenuItem.is_sensitive())
		return;

	show(MenuItem);

	Gtk::MenuItem* root_menu_item = 0;
	Gtk::Menu* root_menu = 0;

	for(Gtk::Widget* ancestor = &MenuItem; ancestor; )
	{
		Gtk::Menu* const menu = dynamic_cast<Gtk::Menu*>(ancestor);
		Gtk::MenuItem* const menu_item = dynamic_cast<Gtk::MenuItem*>(ancestor);

		if(menu)
		{
			root_menu = menu;
			ancestor = menu->get_attach_widget();
		}
		else if(menu_item)
		{
			root_menu_item = menu_item;
			ancestor = ancestor->get_parent();
		}
		else
		{
			ancestor = ancestor->get_parent();
		}
	}

	return_if_fail(root_menu);
	root_menu->popdown();

	return_if_fail(root_menu_item);
	root_menu_item->deselect();

	MenuItem.activate();
	handle_pending_events();
	non_blocking_sleep(delay);
}

void toggle(Gtk::ToggleButton& Button)
{
	static const unsigned long delay = 500;

	show(Button);
	move_pointer(Button);

	if(!Button.is_sensitive())
		return;

	Button.pressed();
	handle_pending_events();
	non_blocking_sleep(delay);

	Button.released();
	handle_pending_events();
	non_blocking_sleep(delay);
}

void move_pointer(Gtk::TreeView& TreeView, Gtk::TreeViewColumn& Column, const Gtk::TreeIter& Row)
{
	// Get the center of the row in tree coordinates ...
	TreeView.scroll_to_row(Gtk::TreePath(Row), 0.5);
	handle_pending_events();
	Gdk::Rectangle cell_area;
	TreeView.get_cell_area(Gtk::TreePath(Row), Column, cell_area);
	int tx = cell_area.get_x() + (cell_area.get_width() / 2);
	int ty = cell_area.get_y() + (cell_area.get_height() / 2);

	// Convert to widget coordinates and move the pointer ...
	int wx = tx;
	int wy = ty;

/** \todo The gtkmm / gtk+ documentation says that Gtk::TreeView::get_cell_area() returns tree coordinates, but the returned coordinates are widget coordinates, so the following transformation isn't needed */
//	TreeView.tree_to_widget_coords(tx, ty, wx, wy);

	detail::move_pointer(TreeView.get_window(), k3d::vector2(wx, wy), options::tutorial_speed(), false);
}

void select_row(Gtk::TreeView& TreeView, Gtk::TreeViewColumn& Column, const Gtk::TreeIter& Row)
{
	static const unsigned long delay = 500;

	show(TreeView);
	move_pointer(TreeView, Column, Row);

	// Select the row ...
	TreeView.set_cursor(Gtk::TreePath(Row));
	handle_pending_events();
	non_blocking_sleep(delay);
}

/** \todo Get this thing working */
void set_text(Gtk::TreeView& TreeView, Gtk::TreeViewColumn& Column, Gtk::CellRenderer& Cell, const Gtk::TreeIter& Row, const std::string& Text)
{
	static const unsigned long delay = 500;
	const double speed = options::tutorial_speed();

	show(TreeView);
	move_pointer(TreeView, Column, Row);

	// Get the center of the row in tree coordinates ...
	TreeView.scroll_to_row(Gtk::TreePath(Row), 0.5);
	handle_pending_events();
	Gdk::Rectangle cell_area;
	TreeView.get_cell_area(Gtk::TreePath(Row), Column, cell_area);

	TreeView.set_cursor(Gtk::TreePath(Row), Column, true);

	Gdk::Rectangle background_area;
	TreeView.get_background_area(Gtk::TreePath(Row), Column, background_area);

	Gtk::CellEditable* const editable = Cell.start_editing(static_cast<GdkEvent*>(0), TreeView, Gtk::TreePath(Row).to_string(), background_area, cell_area, Gtk::CELL_RENDERER_FOCUSED);
	handle_pending_events();
	non_blocking_sleep(delay);

	Gtk::Entry* const entry = dynamic_cast<Gtk::Entry*>(editable);
	return_if_fail(entry);

//	entry->grab_focus();
//	entry->set_position(-1);
//	entry->select_region(0, -1);
	handle_pending_events();
	non_blocking_sleep(static_cast<unsigned long>(delay / speed));

/** \todo For some reason, we don't get any feedback while entering text interactively */
	entry->set_text(Text);
/*
	entry->delete_text(0, -1);
	handle_pending_events();
	non_blocking_sleep(static_cast<unsigned long>(delay / speed));

	for(unsigned int i = 0; i < Text.size(); ++i)
	{
		int position = i;
		entry->insert_text(Glib::ustring(1, Text[i]), 1, position);
		handle_pending_events();
		non_blocking_sleep(static_cast<unsigned long>((25 + (rand() % 150)) / speed));
	}
*/
	handle_pending_events();
	non_blocking_sleep(static_cast<unsigned long>(delay / speed));

	editable->editing_done();
}

/** \todo Improve the implementation of this so we see the mouse pointer move over the item to be chosen */
void select_row(Gtk::ComboBox& ComboBox, const Gtk::TreeIter& Row)
{
	static const unsigned long delay = 400;

	show(ComboBox);
	move_pointer(ComboBox);

	ComboBox.popup();
	handle_pending_events();
	non_blocking_sleep(delay);

	ComboBox.set_active(Row);
	ComboBox.popdown();
	handle_pending_events();
	non_blocking_sleep(delay);
}

} // namespace interactive

} // namespace libk3dngui


