// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Timothy M. Shead (tshead@k-3d.com)
		\author Romain Behar (romainbehar@yahoo.com)
*/

#include <k3dsdk/i18n.h>
#include <k3dsdk/bitmap_modifier.h>
#include <k3dsdk/node.h>
#include <k3dsdk/persistent.h>
#include <k3dsdk/module.h>

#include <iterator>

namespace libk3dbitmap
{

/////////////////////////////////////////////////////////////////////////////
// bitmap_matte_invert

class bitmap_matte_invert :
	public k3d::bitmap_modifier<k3d::persistent<k3d::node> >
{
	typedef k3d::bitmap_modifier<k3d::persistent<k3d::node> > base;

public:
	bitmap_matte_invert(k3d::idocument& Document) :
		base(Document)
	{
		m_input.changed_signal().connect(sigc::mem_fun(*this, &bitmap_matte_invert::on_reset_bitmap));
		m_output.need_data_signal().connect(sigc::mem_fun(*this, &bitmap_matte_invert::on_create_bitmap));
	}
	
	void on_reset_bitmap()
	{
		m_output.reset();
	}

	struct invert_alpha
	{
		void operator()(k3d::pixel& Pixel)
		{
			Pixel.alpha = k3d::pixel::sample_traits::invert(Pixel.alpha);
		}
	};

	k3d::bitmap* on_create_bitmap()
	{
		// If we don't have any input bitmap, we're done ...
		const k3d::bitmap* const input = m_input.value();
		if(!input)
			return 0;

		// Otherwise, we make a copy of the input bitmap and modify the copy ...
		k3d::bitmap* const output = new k3d::bitmap(*input);
		std::for_each(output->begin(), output->end(), invert_alpha());

		return output;
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<bitmap_matte_invert>,
			k3d::interface_list<k3d::ibitmap_source,
			k3d::interface_list<k3d::ibitmap_sink> > > factory(
				k3d::uuid(0x6856b0e9, 0x36b645d9, 0xb324d58b, 0x624072e9),
				"BitmapMatteInvert",
				_("Inverts a bitmap's alpha channel"),
				"Bitmap",
				k3d::iplugin_factory::STABLE);

		return factory;
	}
};

/////////////////////////////////////////////////////////////////////////////
// bitmap_matte_invert_factory

k3d::iplugin_factory& bitmap_matte_invert_factory()
{
	return bitmap_matte_invert::get_factory();
}

} // namespace libk3dbitmap

