// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		 \author Anders Dahnielson (anders@dahnielson.com)
*/

#include <k3dsdk/i18n.h>
#include <k3dsdk/module.h>

#include "bitmap_composite_simple.h"

namespace libk3dbitmap
{

/////////////////////////////////////////////////////////////////////////////
// bitmap_image_mix

class bitmap_image_mix :
	public bitmap_composite_simple
{
	typedef bitmap_composite_simple base;

public:
	bitmap_image_mix(k3d::idocument& Document) :
		base(Document),
		m_value(init_owner(*this) + init_name("value") + init_label(_("Mix percent")) + init_description(_("Mixing percent between the two input bitmaps")) + init_value(0.0))
	{
		m_value.changed_signal().connect(sigc::mem_fun(*this, &bitmap_image_mix::on_value_change));
	}

	void on_value_change()
	{
		m_value_cache = m_value.value();

		on_reset_bitmap();
	}

	void on_composite(const k3d::pixel& A, const k3d::pixel& B, k3d::pixel& O)
	{
		double val = m_value_cache;
		double invval = 1 - m_value_cache;

		O.red   = (val * A.red)   + (invval * B.red);
		O.green = (val * A.green) + (invval * B.green);
		O.blue  = (val * A.blue)  + (invval * B.blue);
		O.alpha = (val * A.alpha) + (invval * B.alpha);
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<bitmap_image_mix>,
			k3d::interface_list<k3d::ibitmap_source,
			k3d::interface_list<k3d::ibitmap_sink> > > factory(
				k3d::uuid(0x6d3f3951, 0x5f9a416f, 0xae464e2a, 0xd8fdc947),
				"BitmapImageMix",
				_("Two images averaged together"),
				"Bitmap",
				k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_data(double, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_value;

	double m_value_cache;
};

/////////////////////////////////////////////////////////////////////////////
// bitmap_image_mix_factory

k3d::iplugin_factory& bitmap_image_mix_factory()
{
	return bitmap_image_mix::get_factory();
}

} // namespace libk3dbitmap


