// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Timothy M. Shead (tshead@k-3d.com)
*/

#include "log.h"
#include "mesh.h"
#include "mesh_selection.h"
#include "selection.h"

namespace k3d
{

namespace detail
{

/// Stores mesh component selection data in the given collection of selection records
class store_selection
{
public:
	store_selection(mesh_selection::records_t& Records) :
		records(Records),
		index(0)
	{
	}

	/** \todo Consolidate ranges of identical selection data */
	template<typename component_t>
	void operator()(component_t Component)
	{
		records.insert(std::make_pair(index, mesh_selection::record(index+1, Component->selection_weight)));
		++index;
	}
	
private:
	mesh_selection::records_t& records;
	size_t index;
};
		
/// Updates mesh components using stored selection data.  Note: if selection data doesn't exist for a given component, leaves it alone
class replace_selection
{
public:
	replace_selection(const mesh_selection::records_t& Records) :
		records(Records),
		current_record(Records.begin()),
		end_record(Records.end()),
		index(0)
	{
	}

	template<typename component_t>
	void operator()(component_t Component)
	{
		if(current_record == end_record)
			return;

		if(index >= current_record->second.end)
			++current_record;

		if(current_record == end_record)
			return;
			
		if(index >= current_record->first && index < current_record->second.end)
		{
			Component->selection_weight = current_record->second.weight;
		}
		
		++index;
	}

private:
	const mesh_selection::records_t& records;
	mesh_selection::records_t::const_iterator current_record;
	const mesh_selection::records_t::const_iterator end_record;
	size_t index;
};

/// Updates mesh components so their visible_selection flag matches their selection_weight, or is diabled altogether
class set_visible_selection
{
public:
	set_visible_selection(const bool VisibleSelection) :
		visible_selection(VisibleSelection)
	{
	}

	template<typename component_t>
	void operator()(component_t& Component)
	{
		Component.visible_selection = visible_selection && Component.selection_weight;
	}

private:
	const bool visible_selection;
};

} // namespace detail

/////////////////////////////////////////////////////////////////////////////
// mesh_selection

const mesh_selection mesh_selection::select_all()
{
	mesh_selection result;
	result.points = component_select_all();
	result.edges = component_select_all();
	result.faces = component_select_all();
	result.linear_curves = component_select_all();
	result.cubic_curves = component_select_all();
	result.nucurves = component_select_all();
	result.bilinear_patches = component_select_all();
	result.bicubic_patches = component_select_all();
	result.nupatches = component_select_all();
	
	return result;
}

const mesh_selection mesh_selection::deselect_all()
{
	mesh_selection result;
	result.points = component_deselect_all();
	result.edges = component_deselect_all();
	result.faces = component_deselect_all();
	result.linear_curves = component_deselect_all();
	result.cubic_curves = component_deselect_all();
	result.nucurves = component_deselect_all();
	result.bilinear_patches = component_deselect_all();
	result.bicubic_patches = component_deselect_all();
	result.nupatches = component_deselect_all();
	
	return result;
}

const mesh_selection mesh_selection::select_null()
{
	return mesh_selection();
}

bool mesh_selection::empty() const
{
	return
		points.empty() &&
		edges.empty() &&
		faces.empty() &&
		linear_curves.empty() &&
		cubic_curves.empty() &&
		nucurves.empty() &&
		bilinear_patches.empty() &&
		bicubic_patches.empty() &&
		nupatches.empty();
}

bool mesh_selection::operator==(const mesh_selection& RHS) const
{
	return
		points == RHS.points &&
		edges == RHS.edges &&
		faces == RHS.faces &&
		linear_curves == RHS.linear_curves &&
		cubic_curves == RHS.cubic_curves &&
		nucurves == RHS.nucurves &&
		bilinear_patches == RHS.bilinear_patches &&
		bicubic_patches == RHS.bicubic_patches &&
		nupatches == RHS.nupatches;
}

bool mesh_selection::operator!=(const mesh_selection& RHS) const
{
	return !(operator==(RHS));
}

const mesh_selection::records_t mesh_selection::component_select_all()
{
	mesh_selection::records_t result;
	result.insert(std::make_pair(0, record(size_t(-1), 1.0)));

	return result;
}

const mesh_selection::records_t mesh_selection::component_deselect_all()
{
	mesh_selection::records_t result;
	result.insert(std::make_pair(0, record(size_t(-1), 0.0)));

	return result;
}

void mesh_selection::insert(const size_t Begin, const record& Record, records_t& Selections)
{
	records_t temp;
	
	for(mesh_selection::records_t::iterator selection = Selections.begin(); selection != Selections.end(); )
	{
		if(selection->first < Record.end && selection->second.end > Record.end)
		{
			temp.insert(std::make_pair(Record.end, selection->second));
		}

		if(selection->first < Begin && selection->second.end > Begin)
		{
			selection->second.end = Begin;
			++selection;
		}
		else if(Begin <= selection->first && selection->first < Record.end)
		{
			mesh_selection::records_t::iterator lamb = selection++;
			Selections.erase(lamb);
		}
		else
		{
			++selection;
		}
	}

	Selections.insert(temp.begin(), temp.end());
	Selections.insert(std::make_pair(Begin, Record));
}

std::ostream& operator<<(std::ostream& Stream, const mesh_selection::records_t& RHS)
{
	for(mesh_selection::records_t::const_iterator record = RHS.begin(); record != RHS.end(); ++record)
	{
		if(record != RHS.begin())
			Stream << " ";

		Stream << "[" << record->first << ", " << record->second.end << ") " << record->second.weight;
	}

	return Stream;
}

std::ostream& operator<<(std::ostream& Stream, const mesh_selection& RHS)
{
	Stream << "points:           " << RHS.points << "\n";
	Stream << "edges:            " << RHS.edges << "\n";
	Stream << "faces:            " << RHS.faces << "\n";
	Stream << "linear curves:    " << RHS.linear_curves << "\n";
	Stream << "cubic curves:     " << RHS.cubic_curves << "\n";
	Stream << "nucurves:         " << RHS.nucurves << "\n";
	Stream << "bilinear patches: " << RHS.bilinear_patches << "\n";
	Stream << "bicubic patches:  " << RHS.bicubic_patches << "\n";
	Stream << "nupatches:        " << RHS.nupatches << "\n";

	return Stream;
}

/////////////////////////////////////////////////////////////////////////////
// store_selection

void store_selection(const mesh& Mesh, mesh_selection& MeshSelection)
{
	MeshSelection.points.clear();
	MeshSelection.edges.clear();
	MeshSelection.faces.clear();
	MeshSelection.linear_curves.clear();
	MeshSelection.cubic_curves.clear();
	MeshSelection.nucurves.clear();
	MeshSelection.bilinear_patches.clear();
	MeshSelection.bicubic_patches.clear();
	MeshSelection.nupatches.clear();

	std::for_each(Mesh.points.begin(), Mesh.points.end(), detail::store_selection(MeshSelection.points));

	detail::store_selection store_edge_selection(MeshSelection.edges);
	for(mesh::polyhedra_t::const_iterator polyhedron = Mesh.polyhedra.begin(); polyhedron != Mesh.polyhedra.end(); ++polyhedron)
	{
		for(polyhedron::faces_t::const_iterator face = (*polyhedron)->faces.begin(); face != (*polyhedron)->faces.end(); ++face)
		{
			split_edge* edge = (*face)->first_edge;
			do
			{
				store_edge_selection(edge);
				edge = edge->face_clockwise;
			}
			while(edge != (*face)->first_edge);
		}
	}

	detail::store_selection store_face_selection(MeshSelection.faces);
	for(mesh::polyhedra_t::const_iterator polyhedron = Mesh.polyhedra.begin(); polyhedron != Mesh.polyhedra.end(); ++polyhedron)
		std::for_each((*polyhedron)->faces.begin(), (*polyhedron)->faces.end(), store_face_selection);

	detail::store_selection store_linear_curve_selection(MeshSelection.linear_curves);
	for(mesh::linear_curve_groups_t::const_iterator group = Mesh.linear_curve_groups.begin(); group != Mesh.linear_curve_groups.end(); ++group)
		std::for_each((*group)->curves.begin(), (*group)->curves.end(), store_linear_curve_selection);

	detail::store_selection store_cubic_curve_selection(MeshSelection.cubic_curves);
	for(mesh::cubic_curve_groups_t::const_iterator group = Mesh.cubic_curve_groups.begin(); group != Mesh.cubic_curve_groups.end(); ++group)
		std::for_each((*group)->curves.begin(), (*group)->curves.end(), store_cubic_curve_selection);

	detail::store_selection store_nucurve_selection(MeshSelection.nucurves);
	for(mesh::nucurve_groups_t::const_iterator group = Mesh.nucurve_groups.begin(); group != Mesh.nucurve_groups.end(); ++group)
		std::for_each((*group)->curves.begin(), (*group)->curves.end(), store_nucurve_selection);

	std::for_each(Mesh.bilinear_patches.begin(), Mesh.bilinear_patches.end(), detail::store_selection(MeshSelection.bilinear_patches));
	std::for_each(Mesh.bicubic_patches.begin(), Mesh.bicubic_patches.end(), detail::store_selection(MeshSelection.bicubic_patches));
	std::for_each(Mesh.nupatches.begin(), Mesh.nupatches.end(), detail::store_selection(MeshSelection.nupatches));
}

/////////////////////////////////////////////////////////////////////////////
// replace_selection

void replace_selection(const mesh_selection& MeshSelection, mesh& Mesh)
{
	// This is a policy decision that may belong elsewhere ... if the mesh_selection is
	// empty, don't modify the mesh at all.  This makes it possible for a plugin to have an
	// "empty" selection property, in which-case the plugin will alter its input using whatever
	// selection the input already carries.
	if(MeshSelection.empty())
		return;

	std::for_each(Mesh.points.begin(), Mesh.points.end(), detail::replace_selection(MeshSelection.points));

	detail::replace_selection replace_edge_selection(MeshSelection.edges);
	for(mesh::polyhedra_t::const_iterator polyhedron = Mesh.polyhedra.begin(); polyhedron != Mesh.polyhedra.end(); ++polyhedron)
	{
		for(polyhedron::faces_t::const_iterator face = (*polyhedron)->faces.begin(); face != (*polyhedron)->faces.end(); ++face)
		{
			split_edge* edge = (*face)->first_edge;
			do
			{
				replace_edge_selection(edge);
				edge = edge->face_clockwise;
			}
			while(edge != (*face)->first_edge);

			for(face::holes_t::iterator hole = (*face)->holes.begin(); hole != (*face)->holes.end(); ++hole)
			{
				split_edge* edge = *hole;
				do
				{
					replace_edge_selection(edge);
					edge = edge->face_clockwise;
				}
				while(edge != (*hole));
			}
		}
	}

	detail::replace_selection replace_face_selection(MeshSelection.faces);
	for(mesh::polyhedra_t::const_iterator polyhedron = Mesh.polyhedra.begin(); polyhedron != Mesh.polyhedra.end(); ++polyhedron)
		std::for_each((*polyhedron)->faces.begin(), (*polyhedron)->faces.end(), replace_face_selection);

	detail::replace_selection replace_linear_curve_selection(MeshSelection.linear_curves);
	for(mesh::linear_curve_groups_t::const_iterator group = Mesh.linear_curve_groups.begin(); group != Mesh.linear_curve_groups.end(); ++group)
		std::for_each((*group)->curves.begin(), (*group)->curves.end(), replace_linear_curve_selection);

	detail::replace_selection replace_cubic_curve_selection(MeshSelection.cubic_curves);
	for(mesh::cubic_curve_groups_t::const_iterator group = Mesh.cubic_curve_groups.begin(); group != Mesh.cubic_curve_groups.end(); ++group)
		std::for_each((*group)->curves.begin(), (*group)->curves.end(), replace_cubic_curve_selection);

	detail::replace_selection replace_nucurve_selection(MeshSelection.nucurves);
	for(mesh::nucurve_groups_t::const_iterator group = Mesh.nucurve_groups.begin(); group != Mesh.nucurve_groups.end(); ++group)
		std::for_each((*group)->curves.begin(), (*group)->curves.end(), replace_nucurve_selection);

	std::for_each(Mesh.bilinear_patches.begin(), Mesh.bilinear_patches.end(), detail::replace_selection(MeshSelection.bilinear_patches));
	std::for_each(Mesh.bicubic_patches.begin(), Mesh.bicubic_patches.end(), detail::replace_selection(MeshSelection.bicubic_patches));
	std::for_each(Mesh.nupatches.begin(), Mesh.nupatches.end(), detail::replace_selection(MeshSelection.nupatches));
}

//////////////////////////////////////////////////////////////////////////////////
// set_visible_selection

void set_visible_selection(const bool VisibleSelection, mesh& Mesh)
{
	k3d::for_each_component(Mesh, detail::set_visible_selection(VisibleSelection));
}

} // namespace k3d

