// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Declares the application_implementation class, which encapsulates the running K-3D program
		\author Tim Shead (tshead@k-3d.com)
*/

#include "application_implementation.h"
#include "data.h"
#include "document.h"
#include "iapplication_plugin_factory.h"
#include "ideletable.h"
#include "idocument.h"
#include "iscript_engine.h"
#include "ishader_collection.h"
#include "iuser_interface.h"
#include "plugins.h"
#include "result.h"
#include "state_change_set.h"

#include <boost/filesystem/operations.hpp>
#include <boost/filesystem/path.hpp>

#include <fstream>
#include <iostream>

#ifdef	interface
#undef	interface
#endif	// interface

namespace k3d
{

/////////////////////////////////////////////////////////////////////////////
// application_implementation::implementation

/// Encapsulates the running K-3D server application
class application_implementation::implementation :
	public k3d::iapplication,
	public k3d::ideletable
{
public:
	implementation(k3d::iplugin_factory_collection& Plugins, k3d::ishader_collection& Shaders, k3d::irender_farm& RenderFarm, k3d::iuser_interface& UserInterface, const boost::filesystem::path& ShaderCachePath, const boost::filesystem::path& SharePath) :
		m_plugins(Plugins),
		m_shaders(Shaders),
		m_render_farm(RenderFarm),
		m_user_interface(UserInterface),
		m_shader_cache_path(ShaderCachePath),
		m_share_path(SharePath)
	{
		// Sanity checks ...
		assert_warning(boost::filesystem::exists(m_shader_cache_path));
		assert_warning(boost::filesystem::exists(m_share_path));
	}

	~implementation()
	{
		// Notify observers that we're going away ...
		m_close_signal.emit();

		// Delete any remaining documents ...
		for(document_list_t::iterator document = m_documents.begin(); document != m_documents.end(); document = m_documents.begin())
			close_document(**document);
	}

	k3d::iuser_interface& user_interface()
	{
		return m_user_interface;
	}

	bool exit()
	{
		if(application().safe_to_close_signal().empty() || application().safe_to_close_signal().emit())
			return m_exit_signal.emit();

		return false;
	}

	k3d::idocument* create_document()
	{
		// Notify observers that we're about to create a new document ...
		m_pre_create_document_signal.emit();

		// Create a new doc ...
		k3d::idocument* const document = k3d::create_document();
		return_val_if_fail(document, 0);

		// Add it to the doc list ...
		m_documents.push_back(document);

		return document;
	}

	void close_document(k3d::idocument& Document)
	{
		// Find the document iterator ...
		const document_list_t::iterator document = std::find(m_documents.begin(), m_documents.end(), &Document);

		// Sanity checks ...
		return_if_fail(document != m_documents.end());

		// Notify observers that the document will be closed
		m_close_document_signal.emit(Document);

		// Remove it from the document list ...
		m_documents.erase(document);

		// Delete the document ...
		k3d::close_document(Document);
	}

	const k3d::iapplication::document_collection_t documents()
	{
		document_collection_t results;
		std::copy(m_documents.begin(), m_documents.end(), std::back_inserter(results));

		return results;
	}

	k3d::iapplication::startup_message_signal_t& startup_message_signal()
	{
		return m_startup_message_signal;
	}

	k3d::iapplication::safe_to_close_signal_t& safe_to_close_signal()
	{
		return m_safe_to_close_signal;
	}

	k3d::iapplication::close_signal_t& close_signal()
	{
		return m_close_signal;
	}

	k3d::iapplication::pre_create_document_signal_t& pre_create_document_signal()
	{
		return m_pre_create_document_signal;
	}

	k3d::iapplication::close_document_signal_t& close_document_signal()
	{
		return m_close_document_signal;
	}

	const boost::filesystem::path shader_cache_path()
	{
		return m_shader_cache_path;
	}

	const boost::filesystem::path share_path()
	{
		return m_share_path;
	}

	const k3d::iplugin_factory_collection::factories_t& plugins()
	{
		return m_plugins.factories();
	}

	const sl::shaders_t& shaders()
	{
		return m_shaders.shaders();
	}

	k3d::irender_farm& render_farm()
	{
		return m_render_farm;
	}

	application_implementation::exit_signal_t& exit_signal()
	{
		return m_exit_signal;
	}

private:
	implementation(const implementation& RHS);
	implementation& operator=(const implementation& RHS);

	// "Environment" options specified at startup

	/// Stores a reference to the collection of available plugin factories
	k3d::iplugin_factory_collection& m_plugins;
	/// Stores a reference to the collection of available RenderMan shaders
	k3d::ishader_collection& m_shaders;
	/// Stores a reference to a renderfarm object for rendering jobs/frames
	k3d::irender_farm& m_render_farm;
	/// User interface object
	k3d::iuser_interface& m_user_interface;

	/// Stores the shader cache path
	const boost::filesystem::path m_shader_cache_path;
	/// Stores the share path
	const boost::filesystem::path m_share_path;

	/// Stores a collection of open documents
	typedef std::list<k3d::idocument*> document_list_t;
	/// Stores the collection of open documents
	document_list_t m_documents;
	/// Signal emitted to display progress during application startup
	startup_message_signal_t m_startup_message_signal;
	/// Signal emitted prior to closing the application
	safe_to_close_signal_t m_safe_to_close_signal;
	/// Signal emitted when the application is closing
	close_signal_t m_close_signal;
	/// Signal emitted immediately prior to creating / opening a document
	pre_create_document_signal_t m_pre_create_document_signal;
	/// Signal emitted when an open document is closed
	close_document_signal_t m_close_document_signal;

	/// Signal emitted to request application close
	application_implementation::exit_signal_t m_exit_signal;
};

//////////////////////////////////////////////////////////////////////////////////////////////////////////
// application_implementation

application_implementation::application_implementation(iplugin_factory_collection& PluginFactories, ishader_collection& Shaders, irender_farm& RenderFarm, iuser_interface& UserInterface, const boost::filesystem::path& ShaderCachePath, const boost::filesystem::path& SharePath) :
	m_implementation(new implementation(PluginFactories, Shaders, RenderFarm, UserInterface, ShaderCachePath, SharePath))
{
}

application_implementation::~application_implementation()
{
	delete m_implementation;
}

iapplication& application_implementation::interface()
{
	return *m_implementation;
}

application_implementation::exit_signal_t& application_implementation::exit_signal()
{
	return m_implementation->exit_signal();
}

} // namespace k3d


