// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Timothy M. Shead (tshead@k-3d.com)
		\author Bart Janssens <bart.janssens@lid.kviv.be>
*/

#include "subdiv_algorithms.h"

#include <k3dsdk/basic_math.h>
#include <k3dsdk/object.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh_filter.h>
#include <k3dsdk/module.h>
#include <k3dsdk/plugins.h>
#include <k3dsdk/utility.h>

#include <boost/multi_array.hpp>

#include <iomanip>
#include <iterator>

namespace libk3dmesh
{

/////////////////////////////////////////////////////////////////////////////
// split_faces_parallel_implementation

class split_faces_parallel_implementation :
	public k3d::mesh_filter<k3d::persistent<k3d::object> >
{
	typedef k3d::mesh_filter<k3d::persistent<k3d::object> > base;

public:
	split_faces_parallel_implementation(k3d::idocument& Document) :
		base(Document),
		m_split_position(k3d::init_name("split_position") + k3d::init_description("Position of the split point [number]") + k3d::init_value(0.5) + k3d::init_constraint(k3d::constraint::minimum(0.0, k3d::constraint::maximum(1.0))) + k3d::init_precision(3) + k3d::init_step_increment(0.01) + k3d::init_units(typeid(k3d::measurement::scalar)) + k3d::init_document(Document))
	{

		register_property(m_split_position);

		m_split_position.changed_signal().connect(SigC::slot(*this, &split_faces_parallel_implementation::on_reset_geometry));
		m_input_mesh.changed_signal().connect(SigC::slot(*this, &split_faces_parallel_implementation::on_reset_geometry));
		m_output_mesh.need_data_signal().connect(SigC::slot(*this, &split_faces_parallel_implementation::on_create_geometry));
	}

	void on_reset_geometry()
	{
		m_output_mesh.reset();
	}

	/// Returns true iff the given object is selected
	struct is_selected
	{
		bool operator()(k3d::split_edge* const Edge)
		{
			return Edge->selected;
		}
	};

	k3d::mesh* on_create_geometry()
	{
		// Get the input geometry ...
		k3d::mesh* const input = m_input_mesh.property_value();
		if(!input)
			return 0;

		// Create output geometry ...
		std::auto_ptr<k3d::mesh> output(new k3d::mesh());
		double factor = m_split_position.property_value();
		subdiv::split_faces_parallel(*input, *output, factor);
		return output.release();
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<split_faces_parallel_implementation>,
				k3d::interface_list<k3d::imesh_source,
				k3d::interface_list<k3d::imesh_sink > > > factory(
				k3d::uuid(0x28081462, 0x2ffb4786, 0x8804c6c9, 0x1cc9c319),
				"SplitFacesParallel",
				"Splits both faces along selected edges by inserting a new edge at a given fraction of the face width",
				"Objects",
				k3d::iplugin_factory::EXPERIMENTAL);

		return factory;
	}
private:
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::with_constraint) m_split_position;
};

/////////////////////////////////////////////////////////////////////////////
// split_faces_parallel_factory

k3d::iplugin_factory& split_faces_parallel_factory()
{
	return split_faces_parallel_implementation::get_factory();
}

} // namespace libk3dmesh


