// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Timothy M. Shead (tshead@k-3d.com)
		\author Bart Janssens <bart.janssens@lid.kviv.be>
*/

#include <k3dsdk/basic_math.h>
#include <k3dsdk/object.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh_filter.h>
#include <k3dsdk/module.h>
#include <k3dsdk/plugins.h>
#include <k3dsdk/utility.h>

#include <boost/multi_array.hpp>
#include <iomanip>
#include <iterator>

namespace libk3dmesh
{

/////////////////////////////////////////////////////////////////////////////
// select_clockwise_implementation

class select_clockwise_implementation :
	public k3d::mesh_filter<k3d::persistent<k3d::object> >
{
	typedef k3d::mesh_filter<k3d::persistent<k3d::object> > base;

public:
	select_clockwise_implementation(k3d::idocument& Document) :
		base(Document),
		m_number(k3d::init_name("number") + k3d::init_description("Edge number [integer]") + k3d::init_constraint(k3d::constraint::minimum(0UL)) + k3d::init_document(Document) + k3d::init_value(0) + k3d::init_precision(0) + k3d::init_step_increment(1) + k3d::init_units(typeid(k3d::measurement::scalar)))
	{

		register_property(m_number);
		
		m_input_mesh.changed_signal().connect(SigC::slot(*this, &select_clockwise_implementation::on_reset_geometry));
		m_number.changed_signal().connect(SigC::slot(*this, &select_clockwise_implementation::on_reset_geometry));
		m_output_mesh.need_data_signal().connect(SigC::slot(*this, &select_clockwise_implementation::on_create_geometry));
	}

	void on_reset_geometry()
	{
		m_output_mesh.reset();
	}

	/// Returns true iff the given object is selected
	struct is_selected
	{
		bool operator()(k3d::split_edge* const Edge)
		{
			return Edge->selected;
		}
	};

	k3d::mesh* on_create_geometry()
	{
		// Get the input geometry ...
		k3d::mesh* const input = m_input_mesh.property_value();
		if(!input)
			return 0;

		// Create output geometry ...
		k3d::mesh* const output = new k3d::mesh();
		k3d::deep_copy(*input, *output);

		// For each polyhedron ...
		for(k3d::mesh::polyhedra_t::iterator polyhedron = output->polyhedra.begin(); polyhedron != output->polyhedra.end(); ++polyhedron)
			{
				const bool was_solid = k3d::is_solid(**polyhedron);

				std::set<k3d::split_edge*> selected_edges;
				k3d::copy_if((*polyhedron)->edges.begin(), (*polyhedron)->edges.end(), std::inserter(selected_edges, selected_edges.end()), is_selected());

				for (std::set<k3d::split_edge*>::iterator it = selected_edges.begin(); it != selected_edges.end(); ++it)
					{
						(*it)->selected = false;
					}
				unsigned long number = m_number.property_value();
				k3d::split_edge* edge = *(selected_edges.begin());
				for (unsigned long i = 0; i <= number; ++i)
					edge = edge->face_clockwise;
				edge->selected = true;
				assert_warning(is_valid(**polyhedron));

				// If the polyhedron was solid when we started, it better be solid when we're done ...
				if(was_solid)
					assert_warning(is_solid(**polyhedron));
			}

		return output;
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<select_clockwise_implementation>,
				k3d::interface_list<k3d::imesh_source,
				k3d::interface_list<k3d::imesh_sink > > > factory(
				k3d::uuid(0xea623f75, 0x19264b07, 0xbfa105a4, 0xba070bed),
				"SelectClockwise",
				"Select the clockwise edge of an edge. If multiple edges are selected the first one in the list is selected and all others are deselected.",
				"Objects",
				k3d::iplugin_factory::EXPERIMENTAL);

		return factory;
	}
private:
	k3d_measurement_property(unsigned long, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::with_constraint) m_number;
};

/////////////////////////////////////////////////////////////////////////////
// select_clockwise_factory

k3d::iplugin_factory& select_clockwise_factory()
{
	return select_clockwise_implementation::get_factory();
}

} // namespace libk3dmesh


