// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Timothy M. Shead (tshead@k-3d.com)
		\author Romain Behar (romainbehar@yahoo.com)
*/

#include <k3dsdk/classes.h>
#include <k3dsdk/computed_property.h>
#include <k3dsdk/glutility.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/object.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/mesh.h>
#include <k3dsdk/mesh_filter.h>
#include <k3dsdk/module.h>
#include <k3dsdk/plugins.h>

namespace libk3dmesh
{

/////////////////////////////////////////////////////////////////////////////
// mesh_stats_implementation

class mesh_stats_implementation :
	public k3d::mesh_filter<k3d::persistent<k3d::object> >
{
	typedef k3d::mesh_filter<k3d::persistent<k3d::object> > base;

public:
	mesh_stats_implementation(k3d::idocument& Document) :
		base(Document),
		m_point_count("point_count", "Point count [number]", k3d::method_call(*this, &mesh_stats_implementation::get_point_count)),
		m_edge_count("edge_count", "Edge count [number]", k3d::method_call(*this, &mesh_stats_implementation::get_edge_count)),
		m_face_count("face_count", "Face count [number]", k3d::method_call(*this, &mesh_stats_implementation::get_face_count))
	{
		register_property(m_point_count);
		register_property(m_edge_count);
		register_property(m_face_count);

		m_input_mesh.changed_signal().connect(SigC::slot(*this, &mesh_stats_implementation::on_reset_geometry));

		m_input_mesh.changed_signal().connect(m_point_count.changed_signal().slot());
		m_input_mesh.changed_signal().connect(m_edge_count.changed_signal().slot());
		m_input_mesh.changed_signal().connect(m_face_count.changed_signal().slot());

		m_output_mesh.need_data_signal().connect(SigC::slot(*this, &mesh_stats_implementation::on_create_geometry));
	}

	void on_reset_geometry()
	{
		m_output_mesh.reset();
	}

	unsigned long get_point_count()
	{
		k3d::mesh* const input = m_input_mesh.property_value();
		if(!input)
			return 0;

		return input->points.size();
	}

	unsigned long get_edge_count()
	{
		k3d::mesh* const input = m_input_mesh.property_value();
		if(!input)
			return 0;

		unsigned long edge_count = 0;
		for(k3d::mesh::polyhedra_t::const_iterator polyhedron = input->polyhedra.begin(); polyhedron != input->polyhedra.end(); ++polyhedron)
			edge_count += (*polyhedron)->edges.size();

		return edge_count;
	}

	unsigned long get_face_count()
	{
		k3d::mesh* const input = m_input_mesh.property_value();
		if(!input)
			return 0;

		unsigned long face_count = 0;
		for(k3d::mesh::polyhedra_t::const_iterator polyhedron = input->polyhedra.begin(); polyhedron != input->polyhedra.end(); ++polyhedron)
			face_count += (*polyhedron)->faces.size();

		return face_count;
	}

	k3d::mesh* on_create_geometry()
	{
		// Get the input geometry ...
		k3d::mesh* const input = m_input_mesh.property_value();
		if(!input)
			return 0;

		// Create output geometry ...
		k3d::mesh* const output = new k3d::mesh();
		k3d::deep_copy(*input, *output);

		return output;
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<mesh_stats_implementation>,
			k3d::interface_list<k3d::imesh_source,
			k3d::interface_list<k3d::imesh_sink > > > factory(
				k3d::uuid(0x5e9aae8e, 0xc73345c1, 0x932bfe88, 0xb912bbf1),
				"MeshStats",
				"Displays statistics about the input mesh",
				"Objects",
				k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d::computed_property<unsigned long, k3d::method_call_t<mesh_stats_implementation, unsigned long> > m_point_count;
	k3d::computed_property<unsigned long, k3d::method_call_t<mesh_stats_implementation, unsigned long> > m_edge_count;
	k3d::computed_property<unsigned long, k3d::method_call_t<mesh_stats_implementation, unsigned long> > m_face_count;
};

/////////////////////////////////////////////////////////////////////////////
// mesh_stats_factory

k3d::iplugin_factory& mesh_stats_factory()
{
	return mesh_stats_implementation::get_factory();
}

} // namespace libk3dmesh


