// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Implements the ImageMagickReader K-3D object, which reads and writes a large number of image file formats, using the ImageMagick Magick++ API
		\author Brett W. McCoy (bmccoy@chapelperilous.net)
*/

#include <k3dsdk/ibitmap_read_format.h>
#include <k3dsdk/ideletable.h>
#include <k3dsdk/ifile_format.h>
#include <k3dsdk/module.h>

#include <Magick++.h>

using namespace Magick;

namespace
{

/////////////////////////////////////////////////////////////////////////////
// imagemagick_reader_implementation

class imagemagick_reader_implementation :
	public k3d::ifile_format,
	public k3d::ibitmap_read_format,
	public k3d::ideletable
{
public:
	imagemagick_reader_implementation()
	{
	}

	unsigned long priority()
	{
		return 1;
	}

	bool query_can_handle(const boost::filesystem::path& FilePath)
	{
		try
			{
				Image image;
				image.read(FilePath.native_file_string());
				return true;
			}
		catch(Exception& e)
			{
				return false;
			}
	}

	bool read_file(const boost::filesystem::path& FilePath, k3d::bitmap& Bitmap)
	{
		// Sanity checks ...
		return_val_if_fail(!FilePath.empty(), false);

		std::cerr << info << "Read " << FilePath.native_file_string() << " using ImageMagickReader" << std::endl;

		try
		{
			Image image;
			image.read(FilePath.native_file_string());
			
			unsigned long sourcewidth = image.columns();
			unsigned long sourceheight = image.rows();
			
			return_val_if_fail(sourcewidth, false);
			return_val_if_fail(sourceheight, false);
			
			//set the image buffer
			Bitmap.reset(sourcewidth, sourceheight);
			k3d::pixel* destination = Bitmap.data();
			
			ColorRGB color_rgb;
			
			//load up the bitmap
			for(unsigned long y = 0; y < sourceheight; y++) 
			{
				for(unsigned long x = 0; x < sourcewidth; x++) 
				{
					
					color_rgb = image.pixelColor(x, y);
					destination->red = color_rgb.red();
					destination->green = color_rgb.green();
					destination->blue = color_rgb.blue();
					destination->alpha = 1.0 - color_rgb.alpha();
					
					++destination;
				}
			}
		}
		catch(Exception& e)
		{
			std::cerr << error << "ImageMagickReader: Caught exception " << e.what() << std::endl;
			return false;
		}

		return true;
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
	  		k3d::application_plugin<imagemagick_reader_implementation>, 
			k3d::interface_list<k3d::ibitmap_read_format> > factory(
				k3d::uuid(0x4eb70e35, 0x4e654a53, 0xa8e4b07c, 0x4219d946),
				"ImageMagickReader",
				"ImageMagick Formats ( many )",
				"Bitmap File Formats");
 
		return factory;
	}
};

} //namespace

/////////////////////////////////////////////////////////////////////////////
// Helper functions

namespace libk3dimagemagick
{

/////////////////////////////////////////////////////////////////////////////
// imagemagick_reader_factory

  k3d::iplugin_factory& imagemagick_reader_factory()
  {
    return ::imagemagick_reader_implementation::get_factory();
  }

} // namespace libk3dimagemagick


