// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Implements Bulge tool, applies a "Bulge" transformation to points
		\author Andy Gill (buzz@ucky.com)
		\author Tim Shead (tshead@k-3d.com)
		\author Romain Behar (romainbehar@yahoo.com)
*/

#include <k3dsdk/axis.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh_filter.h>
#include <k3dsdk/module.h>
#include <k3dsdk/mouse_event_observer.h>
#include <k3dsdk/object.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/plugins.h>
#include <k3dsdk/selection.h>
#include <k3dsdk/transform.h>

namespace libk3ddeformation
{

namespace detail
{

class bulge_points
{
public:
	bulge_points(const k3d::vector3& Origin, const k3d::vector3& Min, const k3d::vector3& Max, const k3d::axis Axis, const bool Radial, const bool displace_x, const bool displace_y, const bool displace_z, const double bulge_factor) :
		m_origin(Origin),
		m_min(Min),
		m_max(Max),
		m_axis(Axis),
		m_radial(Radial),
		m_displace_x(displace_x),
		m_displace_y(displace_y),
		m_displace_z(displace_z),
		m_bulge_factor(bulge_factor),
		m_size(Max - Min)
	{
	}

	const k3d::vector3 bulge(k3d::vector3& Coords)
	{
		k3d::vector3 coords = Coords + m_origin;

		unsigned long index = 0;
		switch(m_axis)
			{
				case k3d::X:
					index = 0;
				break;
				case k3d::Y:
					index = 1;
				break;
				case k3d::Z:
					index = 2;
				break;
				default:
					assert_not_reached();

			}

		double delta = 0;
		double length = m_size[index];
		if(length != 0)
			{
				double min = m_min[index];
				double max = m_max[index];
				double value = coords[index];

				double down = value - min;
				double up = max - value;

				delta = 2 * down * up / (length*length);
			}

		double bulge_amount = delta;
		if(m_radial)
			{
				double distance = coords.Length();
				double scale;
				if(0 == distance)
					scale = 1.0;
				else
					scale = (distance + m_bulge_factor * bulge_amount) / distance;

				if(m_displace_x && (k3d::X != m_axis))
					coords[0] *= scale;

				if(m_displace_y && (k3d::Y != m_axis))
					coords[1] *= scale;

				if(m_displace_z && (k3d::Z != m_axis))
					coords[2] *= scale;
			}
		else
			{
				double offset = m_bulge_factor * bulge_amount;

				if(m_displace_x && (k3d::X != m_axis))
					coords[0] += offset;

				if(m_displace_y && (k3d::Y != m_axis))
					coords[1] += offset;

				if(m_displace_z && (k3d::Z != m_axis))
					coords[2] += offset;
			}

		return coords - m_origin;
	}

private:
	const k3d::vector3 m_origin;
	const k3d::vector3 m_min;
	const k3d::vector3 m_max;
	const k3d::axis m_axis;
	const bool m_radial;
	const bool m_displace_x;
	const bool m_displace_y;
	const bool m_displace_z;
	const double m_bulge_factor;
	const k3d::vector3 m_size;
};

} // namespace detail

/////////////////////////////////////////////////////////////////////////////
// bulge_points_implementation

class bulge_points_implementation :
	public k3d::mesh_filter<k3d::persistent<k3d::object> >,
	public k3d::mouse_event_observer
{
	typedef k3d::mesh_filter<k3d::persistent<k3d::object> > base;

public:
	bulge_points_implementation(k3d::idocument& Document) :
		base(Document),
		k3d::mouse_event_observer("LMB drag to bulge points"),
		m_bulge_factor(k3d::init_name("bulge_factor") + k3d::init_description("Bulge amount [number]") + k3d::init_value(1.0) + k3d::init_document(Document) + k3d::init_precision(2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::scalar))),
		m_displace_x(k3d::init_name("displace_x") + k3d::init_description("Displace X [boolean]") + k3d::init_value(true) + k3d::init_document(Document)),
		m_displace_y(k3d::init_name("displace_y") + k3d::init_description("Displace Y [boolean]") + k3d::init_value(true) + k3d::init_document(Document)),
		m_displace_z(k3d::init_name("displace_z") + k3d::init_description("Displace Z [boolean]") + k3d::init_value(true) + k3d::init_document(Document)),
		m_axis(k3d::init_name("axis") + k3d::init_description("Axis to bulge along [enumeration]") + k3d::init_value(k3d::Z) + k3d::init_enumeration(k3d::axis_values()) + k3d::init_document(Document)),
		m_type(k3d::init_name("type") + k3d::init_description("Bulge type [enumeration]") + k3d::init_value(RADIAL) + k3d::init_enumeration(type_values()) + k3d::init_document(Document))
	{
		enable_serialization(k3d::persistence::proxy(m_bulge_factor));
		enable_serialization(k3d::persistence::proxy(m_displace_x));
		enable_serialization(k3d::persistence::proxy(m_displace_y));
		enable_serialization(k3d::persistence::proxy(m_displace_z));
		enable_serialization(k3d::persistence::proxy(m_axis));
		enable_serialization(k3d::persistence::proxy(m_type));

		register_property(m_bulge_factor);
		register_property(m_displace_x);
		register_property(m_displace_y);
		register_property(m_displace_z);
		register_property(m_axis);
		register_property(m_type);

		m_input_mesh.changed_signal().connect(SigC::slot(*this, &bulge_points_implementation::on_reset_geometry));

		m_bulge_factor.changed_signal().connect(SigC::slot(*this, &bulge_points_implementation::on_reshape_geometry));
		m_displace_x.changed_signal().connect(SigC::slot(*this, &bulge_points_implementation::on_reshape_geometry));
		m_displace_y.changed_signal().connect(SigC::slot(*this, &bulge_points_implementation::on_reshape_geometry));
		m_displace_z.changed_signal().connect(SigC::slot(*this, &bulge_points_implementation::on_reshape_geometry));
		m_axis.changed_signal().connect(SigC::slot(*this, &bulge_points_implementation::on_reshape_geometry));
		m_type.changed_signal().connect(SigC::slot(*this, &bulge_points_implementation::on_reshape_geometry));

		m_output_mesh.need_data_signal().connect(SigC::slot(*this, &bulge_points_implementation::on_create_geometry));
	}

	bool OnLButtonDrag(const k3d::imouse_event_observer::event_state& State, const k3d::vector2& Current, const k3d::vector2& Last, const k3d::vector2& Start, const drag_type_t DragType)
	{
		// Compute scaling factors for each mouse axis ...
		double xfactor = Current[0] - Last[0];
		double yfactor = Current[1] - Last[1];

		m_bulge_factor.set_value(m_bulge_factor.value() - (fabs(xfactor) >= fabs(yfactor) ? xfactor : yfactor));

		return true;
	}

	void on_reset_geometry()
	{
		m_output_mesh.reset();
	}

	void on_reshape_geometry()
	{
		if(m_output_mesh.empty())
			return;

		const k3d::mesh* const input = m_input_mesh.property_value();
		if(!input)
			return;

		k3d::mesh* const output = const_cast<k3d::mesh*>(m_output_mesh.value());
		if(!output)
			return;

		reshape_geometry(*input, *output);

		m_output_mesh.changed_signal().emit();
	}

	k3d::mesh* on_create_geometry()
	{
		// If we don't have any input mesh, we're done ...
		const k3d::mesh* const input = m_input_mesh.property_value();
		if(!input)
			return 0;

		// Otherwise, we make a copy of the input mesh and modify the copy ...
		k3d::mesh* const output = new k3d::mesh();
		k3d::deep_copy(*input, *output);
		reshape_geometry(*input, *output);

		return output;
	}

	void reshape_geometry(const k3d::mesh& Source, k3d::mesh& Target)
	{
		return_if_fail(Source.points.size() == Target.points.size());

		const bool ignore_selection = !k3d::contains_selection(Source);
		const k3d::bounding_box bounds = k3d::bounds(Source);

		const double bulge_factor = m_bulge_factor.property_value();
		const double displace_x = m_displace_x.property_value();
		const double displace_y = m_displace_y.property_value();
		const double displace_z = m_displace_z.property_value();
		const k3d::axis axis = m_axis.property_value();

		detail::bulge_points bulger(
			k3d::vector3(0, 0, 0),
			k3d::vector3(bounds.nx, bounds.ny, bounds.nz),
			k3d::vector3(bounds.px, bounds.py, bounds.pz),
			axis,
			m_type.property_value() == RADIAL,
			displace_x,
			displace_y,
			displace_z,
			bulge_factor
			);

		for(unsigned long i = 0; i != Target.points.size(); ++i)
			{
				if(ignore_selection || Source.points[i]->selected)
					{
						Target.points[i]->position = bulger.bulge(Source.points[i]->position);
					}
				else
					{
						Target.points[i]->position = Source.points[i]->position;
					}
			}
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<k3d::document_plugin<bulge_points_implementation>,
			k3d::interface_list<k3d::imesh_source,
			k3d::interface_list<k3d::imesh_sink > > > factory(
				k3d::uuid(0xb7002ece, 0x8e6348f5, 0xa99ce9b0, 0xfbeba55f),
				"BulgePoints",
				"Bulges mesh points around a point",
				"Objects",
				k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	typedef enum
	{
		LINEAR,
		RADIAL,
	} Type;

	friend std::ostream& operator<<(std::ostream& Stream, const Type& Value)
	{
		switch(Value)
			{
				case LINEAR:
					Stream << "linear";
					break;
				case RADIAL:
					Stream << "radial";
					break;
			}

		return Stream;
	}

	friend std::istream& operator>>(std::istream& Stream, Type& Value)
	{
		std::string text;
		Stream >> text;

		if(text == "linear")
			Value = LINEAR;
		else if(text == "radial")
			Value = RADIAL;
		else
			std::cerr << __PRETTY_FUNCTION__ << ": unknown enumeration [" << text << "]" << std::endl;

		return Stream;
	}

	const k3d::ienumeration_property::values_t& type_values()
	{
		static k3d::ienumeration_property::values_t values;
		if(values.empty())
			{
				values.push_back(k3d::ienumeration_property::value_t("Linear", "linear", "Linear bulge"));
				values.push_back(k3d::ienumeration_property::value_t("Radial", "radial", "Radial bulge"));
			}

		return values;
	}

	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_bulge_factor;
	k3d_data_property(bool, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_displace_x;
	k3d_data_property(bool, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_displace_y;
	k3d_data_property(bool, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_displace_z;
	k3d_enumeration_property(k3d::axis, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_axis;
	k3d_enumeration_property(Type, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_type;
};

/////////////////////////////////////////////////////////////////////////////
// bulge_points_factory

k3d::iplugin_factory& bulge_points_factory()
{
	return bulge_points_implementation::get_factory();
}

} // namespace libk3ddeformation


