// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "application.h"
#include "ioptions.h"
#include "result.h"
#include "shaders.h"
#include "string_modifiers.h"
#include "system_functions.h"

#include <boost/filesystem/convenience.hpp>
#include <boost/filesystem/operations.hpp>
#include <boost/filesystem/path.hpp>
#include <boost/static_assert.hpp>

#include <iostream>

namespace k3d
{

/////////////////////////////////////////////////////////////////////////////
// compile_shader

bool compile_shader(const boost::filesystem::path& SourceFile, const std::string RenderEngineType, const std::string RenderEngine)
{
	// Sanity checks ...
	return_val_if_fail(boost::filesystem::exists(SourceFile), false);
	return_val_if_fail(RenderEngineType.size(), false);
	return_val_if_fail(RenderEngine.size(), false);

	// Get the set of available render engines
	const k3d::ioptions::render_engines_t engines = k3d::application().options().render_engines();

	// For each engine ...
	for(k3d::ioptions::render_engines_t::const_iterator engine = engines.begin(); engine != engines.end(); ++engine)
		{
			// If it's the wrong type, skip it ...
			if(engine->type != RenderEngineType)
				continue;

			// If it's the wrong engine, skip it ...
			if(engine->engine != RenderEngine)
				continue;

			// OK, first let's make sure the shader binary cache directory exists ...
			boost::filesystem::create_directories(k3d::application().shader_cache_path());
			return_val_if_fail(boost::filesystem::exists(k3d::application().shader_cache_path()), false);

			// Figure out what the compiled shader filename will be ...
			std::string shader_binary_file(engine->shader_binary);
			k3d::formatted_replace(shader_binary_file, '%', "p", SourceFile.leaf().substr(0, SourceFile.leaf().rfind(".")));

			// Figure out the full path to the source and compiled shaders ...
			const boost::filesystem::path shader_source_path(SourceFile);
			const boost::filesystem::path shader_binary_path(k3d::application().shader_cache_path() / boost::filesystem::path(shader_binary_file, boost::filesystem::native));

			// Get the last modification time of the shader source ...
			time_t source_modified = 0;
			return_val_if_fail(k3d::system::file_modification_time(shader_source_path, source_modified), false);

			// Get the last modification time of the shader binary (if it exists) ...
			time_t binary_modified = 0;
			k3d::system::file_modification_time(shader_binary_path, binary_modified);

			// If the binary is up-to-date, we're done ...
			if(source_modified <= binary_modified)
				return true;

			std::cerr << info << "Updating [" << shader_source_path.native_file_string() << "] to [" << shader_binary_path.native_file_string() << "]" << std::endl;

			// Create a command line that can compile the shader ...
			std::string commandline = engine->compile_shader;
			k3d::formatted_replace(commandline, '%', "s", shader_source_path.native_file_string()); // %s = shader source file
			k3d::formatted_replace(commandline, '%', "b", shader_binary_path.native_file_string()); // %b = shader binary file
			k3d::formatted_replace(commandline, '%', "c", k3d::application().shader_cache_path().native_file_string()); // %c = shader cache directory
			k3d::formatted_replace(commandline, '%', "d", shader_source_path.branch_path().native_file_string()); // %d = shader source directory

			std::cerr << info << commandline << std::endl;

			// Make it happen ...
			std::stringstream buffer;
			return_val_if_fail(k3d::system::run_process(commandline, buffer), false);

			return true;
		}

	return false;
}

/////////////////////////////////////////////////////////////////////////////
// lookup_shader

std::auto_ptr<sdpsl::shader> lookup_shader(const std::string ShaderName)
{
	const sdpsl::shaders_t shaders(k3d::application().shaders());
	for(sdpsl::shaders_t::const_iterator shader = shaders.begin(); shader != shaders.end(); ++shader)
		{
			if(shader->name == ShaderName)
				return std::auto_ptr<sdpsl::shader>(new sdpsl::shader(*shader));
		}

	return std::auto_ptr<sdpsl::shader>(0);
}

} // namespace k3d


