// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Implements the RenderManScript K-3D object, which can insert scripted data into RenderMan output
		\author Tim Shead (tshead@k-3d.com)
*/

#include <k3dsdk/iapplication.h>
#include <k3dsdk/iscript_engine.h>
#include <k3dsdk/classes.h>
#include <k3dsdk/renderman.h>
#include <k3dsdk/transformable.h>
#include <k3dsdk/object.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/module.h>
#include <k3dsdk/viewport.h>

#include <sdpgl/sdpgl.h>

namespace
{

/////////////////////////////////////////////////////////////////////////////
// renderman_script_implementation

class renderman_script_implementation :
	public k3d::viewport::drawable<k3d::ri::renderable<k3d::transformable<k3d::persistent<k3d::object> > > >
{
	typedef k3d::viewport::drawable<k3d::ri::renderable<k3d::transformable<k3d::persistent<k3d::object> > > > base;

public:
	renderman_script_implementation(k3d::idocument& Document) :
		base(Document),
		m_script(k3d::init_name("script") + k3d::init_description("Script") + k3d::init_value<std::string>("") + init_document(Document)),
		m_script_engine(0)
	{
		enable_serialization(k3d::persistence::proxy(m_script));
		
		register_property(m_script);
		
		m_position.changed_signal().connect(SigC::slot(*this, &renderman_script_implementation::async_redraw_all));
		m_orientation.changed_signal().connect(SigC::slot(*this, &renderman_script_implementation::async_redraw_all));
		m_scale.changed_signal().connect(SigC::slot(*this, &renderman_script_implementation::async_redraw_all));
	}

	~renderman_script_implementation()
	{
		delete dynamic_cast<k3d::ideletable*>(m_script_engine);
	}

	void on_viewport_draw(const k3d::viewport::render_state& State)
	{
		glDisable(GL_LIGHTING);
		glColor3d(1, 1, 0);

		glBegin(GL_POINTS);
		glVertex3d(0, 0, 0);
		glEnd();
	}

	void on_viewport_select(const k3d::viewport::render_state& State)
	{
		glBegin(GL_POINTS);
		glVertex3d(0, 0, 0);
		glEnd();
	}

	void on_renderman_render(const k3d::ri::render_state& State)
	{
		// Create our script engine as-needed
		if(!m_script_engine)
			m_script_engine = dynamic_cast<k3d::iscript_engine*>(k3d::create_application_plugin(k3d::classes::JavaScriptEngine()));

		// No script engine?  We're outta here ...
		return_if_fail(m_script_engine);

		// Execute that bad-boy ...
		k3d::iscript_engine::context_t context;
		context.push_back(&document());
		context.push_back(this);
		context.push_back(&State.engine);

		return_if_fail(m_script_engine->execute(name() + "Script", m_script.value(), context));
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<renderman_script_implementation>,
			k3d::interface_list<k3d::itransform_source,
			k3d::interface_list<k3d::itransform_sink > > > factory(
			k3d::uuid(0xf80c25ca, 0xa1f644c1, 0x8198f00c, 0xafbbf3cf),
			"RenderManScript",
			"Creates RenderMan output under script control",
			"Objects",
			k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_script_property(std::string, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_script;
	k3d::iscript_engine* m_script_engine;
};

} // namespace

namespace libk3drenderman
{

/////////////////////////////////////////////////////////////////////////////
// renderman_script_factory

k3d::iplugin_factory& renderman_script_factory()
{
	return renderman_script_implementation::get_factory();
}

} // namespace libk3drenderman



