// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Timothy M. Shead (tshead@k-3d.com)
*/

#include <k3dsdk/imaterial.h>
#include <k3dsdk/object.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/material.h>
#include <k3dsdk/material_collection.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh_source.h>
#include <k3dsdk/module.h>
#include <k3dsdk/plugins.h>
#include <k3dsdk/transform.h>

#include <iterator>

namespace libk3dmesh
{

/////////////////////////////////////////////////////////////////////////////
// poly_torus_implementation

class poly_torus_implementation :
	public k3d::material_collection<k3d::mesh_source<k3d::persistent<k3d::object> > >
{
	typedef k3d::material_collection<k3d::mesh_source<k3d::persistent<k3d::object> > > base;

public:
	poly_torus_implementation(k3d::idocument& Document) :
		base(Document),
		m_u_segments(k3d::init_name("u_segments") + k3d::init_description("Columns [integer]") + k3d::init_value(32) + k3d::init_document(Document) + k3d::init_constraint(k3d::constraint::minimum(3UL)) + k3d::init_precision(0) + k3d::init_step_increment(1) + k3d::init_units(typeid(k3d::measurement::scalar))),
		m_v_segments(k3d::init_name("v_segments") + k3d::init_description("Rows [integer]") + k3d::init_value(16) + k3d::init_document(Document) + k3d::init_constraint(k3d::constraint::minimum(1UL)) + k3d::init_precision(0) + k3d::init_step_increment(1) + k3d::init_units(typeid(k3d::measurement::scalar))),
		m_majorradius(k3d::init_name("majorradius") + k3d::init_description("Major Radius [number]") + k3d::init_document(Document) + k3d::init_value(5.0) + k3d::init_precision(2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::distance))),
		m_minorradius(k3d::init_name("minorradius") + k3d::init_description("Minor Radius [number]") + k3d::init_document(Document) + k3d::init_value(1.0) + k3d::init_precision(2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::distance))),
		m_u_power(k3d::init_name("u_power") + k3d::init_description("Major Power [number]") + k3d::init_document(Document) + k3d::init_value(1.0) + k3d::init_precision(2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::scalar))),
		m_v_power(k3d::init_name("v_power") + k3d::init_description("Minor Power [number]") + k3d::init_document(Document) + k3d::init_value(1.0) + k3d::init_precision(2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::scalar)))
	{
		enable_serialization(k3d::persistence::proxy(m_u_segments));
		enable_serialization(k3d::persistence::proxy(m_v_segments));
		enable_serialization(k3d::persistence::proxy(m_majorradius));
		enable_serialization(k3d::persistence::proxy(m_minorradius));
		enable_serialization(k3d::persistence::proxy(m_u_power));
		enable_serialization(k3d::persistence::proxy(m_v_power));

		register_property(m_u_segments);
		register_property(m_v_segments);
		register_property(m_majorradius);
		register_property(m_minorradius);
		register_property(m_u_power);
		register_property(m_v_power);

		m_material.changed_signal().connect(SigC::slot(*this, &poly_torus_implementation::on_reset_geometry));
		m_u_segments.changed_signal().connect(SigC::slot(*this, &poly_torus_implementation::on_reset_geometry));
		m_v_segments.changed_signal().connect(SigC::slot(*this, &poly_torus_implementation::on_reset_geometry));
		m_majorradius.changed_signal().connect(SigC::slot(*this, &poly_torus_implementation::on_reset_geometry));
		m_minorradius.changed_signal().connect(SigC::slot(*this, &poly_torus_implementation::on_reset_geometry));
		m_u_power.changed_signal().connect(SigC::slot(*this, &poly_torus_implementation::on_reset_geometry));
		m_v_power.changed_signal().connect(SigC::slot(*this, &poly_torus_implementation::on_reset_geometry));
	
		m_output_mesh.need_data_signal().connect(SigC::slot(*this, &poly_torus_implementation::on_create_geometry));
	}

	void on_reset_geometry()
	{
		m_output_mesh.reset();
	}
	
	k3d::mesh* on_create_geometry()
	{
		std::auto_ptr<k3d::mesh> mesh(new k3d::mesh());

		mesh->polyhedra.push_back(new k3d::polyhedron());
		k3d::polyhedron& polyhedron = *mesh->polyhedra.back();
		polyhedron.material = m_material.interface();

		// Create the torus body ...
		const k3d::grid_results_t grid = k3d::add_grid(*mesh, polyhedron, m_v_segments.property_value(), m_u_segments.property_value(), true, true);
		const boost::multi_array<k3d::point*, 2>& points = boost::get<0>(grid);
//		const boost::multi_array<k3d::split_edge*, 3>& edges = boost::get<1>(grid);
//		const boost::multi_array<k3d::face*, 2>& faces = boost::get<2>(grid);

/*
for(unsigned long v = 0; v != edges.shape()[0]; ++v)
{
	for(unsigned long col = 0; col != edges.shape()[1]; ++col)
		{
			for(unsigned long i = 0; i != 4; ++i)
				std::cerr << v << ", " << col << ", " << i << ": " << edges[v][col][i]->companion << std::endl;
		}
}
*/

		// Define the shape of the torus ...
		const unsigned long point_v_segments = points.shape()[0];
		const unsigned long point_u_segments = points.shape()[1];
		
		// Shape the torus points
		const double majorradius = m_majorradius.property_value();
		const double minorradius = m_minorradius.property_value();
		const double u_power = m_u_power.property_value();
		const double v_power = m_v_power.property_value();
		
		const double inv_u_power = u_power ? 1.0 / u_power : 1.0;
		const double inv_v_power = v_power ? 1.0 / v_power : 1.0;
		
		for(unsigned long v = 0; v != point_v_segments; ++v)
			{
				const double phi = k3d::pi_times_2() * static_cast<double>(v) / static_cast<double>(point_v_segments);
				double minor_x = cos(phi);
				double minor_y = sin(phi);
				minor_x = k3d::sign(minor_x) * std::pow(std::abs(minor_x), inv_v_power);
				minor_y = k3d::sign(minor_y) * std::pow(std::abs(minor_y), inv_v_power);

				const double radius = majorradius - (minorradius * minor_x);

				for(unsigned long u = 0; u != point_u_segments; ++u)
					{
						const double theta = k3d::pi_times_2() * static_cast<double>(u) / static_cast<double>(point_u_segments);

						double x = -sin(theta);
						double y = -cos(theta);
						double z = minorradius * minor_y;

						x = radius * k3d::sign(x) * std::pow(std::abs(x), inv_u_power);
						y = radius * k3d::sign(y) * std::pow(std::abs(y), inv_u_power);
						
						points[v][u]->position = k3d::vector3(x, y, z);
					}
			}

		return_val_if_fail(is_valid(polyhedron), 0);
		assert_warning(is_solid(polyhedron));
						
		return mesh.release();
	}
	
	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<k3d::document_plugin<poly_torus_implementation>, k3d::interface_list<k3d::imesh_source > > factory(
			k3d::uuid(0x9a5ea45b, 0xebc64e37, 0xa50b287a, 0x89e18b71),
			"PolyTorus",
			"Generates a polygonal torus",
			"Objects",
			k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_measurement_property(unsigned long, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::with_constraint) m_u_segments;
	k3d_measurement_property(unsigned long, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::with_constraint) m_v_segments;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_majorradius;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_minorradius;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_u_power;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_v_power;
};

/////////////////////////////////////////////////////////////////////////////
// poly_torus_factory

k3d::iplugin_factory& poly_torus_factory()
{
	return poly_torus_implementation::get_factory();
}

} // namespace libk3dmesh

