// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Romain Behar (romainbehar@yahoo.com)
*/

#include <k3dsdk/basic_math.h>
#include <k3dsdk/imaterial.h>
#include <k3dsdk/object.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/material.h>
#include <k3dsdk/material_collection.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh_source.h>
#include <k3dsdk/module.h>
#include <k3dsdk/plugins.h>
#include <k3dsdk/vectors.h>

namespace libk3dmesh
{

/////////////////////////////////////////////////////////////////////////////
// lissajous_curve_implementation

class lissajous_curve_implementation :
	public k3d::material_collection<k3d::mesh_source<k3d::persistent<k3d::object> > >
{
	typedef k3d::material_collection<k3d::mesh_source<k3d::persistent<k3d::object> > > base;

public:
	lissajous_curve_implementation(k3d::idocument& Document) :
		base(Document),
		m_edge_count(k3d::init_name("edgecount") + k3d::init_description("Edge count [integer]") + k3d::init_value(100) + k3d::init_constraint(k3d::constraint::minimum(3L)) + k3d::init_precision(0) + k3d::init_step_increment(1) + k3d::init_units(typeid(k3d::measurement::scalar)) + k3d::init_document(Document)),
		m_mamp(k3d::init_name("mamp") + k3d::init_description("Modulation amplitude [number]") + k3d::init_value(0.0) + k3d::init_precision(2) + k3d::init_step_increment(0.01) + k3d::init_units(typeid(k3d::measurement::distance)) + k3d::init_document(Document)),
		m_mfreq(k3d::init_name("mfreq") + k3d::init_description("Modulation frequency [number]") + k3d::init_value(20.0) + k3d::init_precision(2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::scalar)) + k3d::init_document(Document)),
		m_mphase(k3d::init_name("mphase") + k3d::init_description("Modulation phase [number]") + k3d::init_value(0.0) + k3d::init_precision(2) + k3d::init_step_increment(k3d::radians(1.0)) + k3d::init_units(typeid(k3d::measurement::angle)) + k3d::init_document(Document)),
		m_xamp(k3d::init_name("xamp") + k3d::init_description("X amplitude [number]") + k3d::init_value(5.0) + k3d::init_precision(2) + k3d::init_step_increment(0.01) + k3d::init_units(typeid(k3d::measurement::distance)) + k3d::init_document(Document)),
		m_xfreq(k3d::init_name("xfreq") + k3d::init_description("X frequency [number]") + k3d::init_value(1.0) + k3d::init_precision(2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::scalar)) + k3d::init_document(Document)),
		m_xphase(k3d::init_name("xphase") + k3d::init_description("X phase [number]") + k3d::init_value(0.0) + k3d::init_precision(2) + k3d::init_step_increment(k3d::radians(1.0)) + k3d::init_units(typeid(k3d::measurement::angle)) + k3d::init_document(Document)),
		m_yamp(k3d::init_name("yamp") + k3d::init_description("Y amplitude [number]") + k3d::init_value(5.0) + k3d::init_precision(2) + k3d::init_step_increment(0.01) + k3d::init_units(typeid(k3d::measurement::distance)) + k3d::init_document(Document)),
		m_yfreq(k3d::init_name("yfreq") + k3d::init_description("Y frequencey [number]") + k3d::init_value(1.0) + k3d::init_precision(2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::scalar)) + k3d::init_document(Document)),
		m_yphase(k3d::init_name("yphase") + k3d::init_description("Y phase [number]") + k3d::init_value(k3d::pi_over_2()) + k3d::init_precision(2) + k3d::init_step_increment(k3d::radians(1.0)) + k3d::init_units(typeid(k3d::measurement::angle)) + k3d::init_document(Document)),
		m_zamp(k3d::init_name("zamp") + k3d::init_description("Z amplitude [number]") + k3d::init_value(0.0) + k3d::init_precision(2) + k3d::init_step_increment(0.01) + k3d::init_units(typeid(k3d::measurement::distance)) + k3d::init_document(Document)),
		m_zfreq(k3d::init_name("zfreq") + k3d::init_description("Z frequency [number]") + k3d::init_value(1.0) + k3d::init_precision(2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::scalar)) + k3d::init_document(Document)),
		m_zphase(k3d::init_name("zphase") + k3d::init_description("Z phase [number]") + k3d::init_value(0.0) + k3d::init_precision(2) + k3d::init_step_increment(k3d::radians(1.0)) + k3d::init_units(typeid(k3d::measurement::angle)) + k3d::init_document(Document)),
		m_width(k3d::init_name("width") + k3d::init_description("Width [number]") + k3d::init_value(0.1) + k3d::init_precision(2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::distance)) + k3d::init_document(Document)),
		m_wrap(k3d::init_name("closed") + k3d::init_description("Closed Curve [boolean]") + k3d::init_value(true) + k3d::init_document(Document))
	{
		enable_serialization(k3d::persistence::proxy(m_edge_count));
		enable_serialization(k3d::persistence::proxy(m_mamp));
		enable_serialization(k3d::persistence::proxy(m_mfreq));
		enable_serialization(k3d::persistence::proxy(m_mphase));
		enable_serialization(k3d::persistence::proxy(m_xamp));
		enable_serialization(k3d::persistence::proxy(m_xfreq));
		enable_serialization(k3d::persistence::proxy(m_xphase));
		enable_serialization(k3d::persistence::proxy(m_yamp));
		enable_serialization(k3d::persistence::proxy(m_yfreq));
		enable_serialization(k3d::persistence::proxy(m_yphase));
		enable_serialization(k3d::persistence::proxy(m_zamp));
		enable_serialization(k3d::persistence::proxy(m_zfreq));
		enable_serialization(k3d::persistence::proxy(m_zphase));
		enable_serialization(k3d::persistence::proxy(m_width));
		enable_serialization(k3d::persistence::proxy(m_wrap));

		register_property(m_edge_count);
		register_property(m_mamp);
		register_property(m_mfreq);
		register_property(m_mphase);
		register_property(m_xamp);
		register_property(m_xfreq);
		register_property(m_xphase);
		register_property(m_yamp);
		register_property(m_yfreq);
		register_property(m_yphase);
		register_property(m_zamp);
		register_property(m_zfreq);
		register_property(m_zphase);
		register_property(m_width);
		register_property(m_wrap);

		m_material.changed_signal().connect(SigC::slot(*this, &lissajous_curve_implementation::on_reset_geometry));

		m_edge_count.changed_signal().connect(SigC::slot(*this, &lissajous_curve_implementation::on_reset_geometry));
		m_mamp.changed_signal().connect(SigC::slot(*this, &lissajous_curve_implementation::on_reset_geometry));
		m_mfreq.changed_signal().connect(SigC::slot(*this, &lissajous_curve_implementation::on_reset_geometry));
		m_mphase.changed_signal().connect(SigC::slot(*this, &lissajous_curve_implementation::on_reset_geometry));
		m_xamp.changed_signal().connect(SigC::slot(*this, &lissajous_curve_implementation::on_reset_geometry));
		m_xfreq.changed_signal().connect(SigC::slot(*this, &lissajous_curve_implementation::on_reset_geometry));
		m_xphase.changed_signal().connect(SigC::slot(*this, &lissajous_curve_implementation::on_reset_geometry));
		m_yamp.changed_signal().connect(SigC::slot(*this, &lissajous_curve_implementation::on_reset_geometry));
		m_yfreq.changed_signal().connect(SigC::slot(*this, &lissajous_curve_implementation::on_reset_geometry));
		m_yphase.changed_signal().connect(SigC::slot(*this, &lissajous_curve_implementation::on_reset_geometry));
		m_zamp.changed_signal().connect(SigC::slot(*this, &lissajous_curve_implementation::on_reset_geometry));
		m_zfreq.changed_signal().connect(SigC::slot(*this, &lissajous_curve_implementation::on_reset_geometry));
		m_zphase.changed_signal().connect(SigC::slot(*this, &lissajous_curve_implementation::on_reset_geometry));
		m_width.changed_signal().connect(SigC::slot(*this, &lissajous_curve_implementation::on_reset_geometry));
		m_wrap.changed_signal().connect(SigC::slot(*this, &lissajous_curve_implementation::on_reset_geometry));

		m_output_mesh.need_data_signal().connect(SigC::slot(*this, &lissajous_curve_implementation::on_create_geometry));
	}

	void on_reset_geometry()
	{
		m_output_mesh.reset();
	}

	k3d::mesh* on_create_geometry()
	{
		const unsigned long edges = m_edge_count.property_value();
		const double mamp = m_mamp.property_value();
		const double mfreq = m_mfreq.property_value();
		const double mphase = m_mphase.property_value();
		const double xamp = m_xamp.property_value();
		const double xfreq = m_xfreq.property_value();
		const double xphase = m_xphase.property_value();
		const double yamp = m_yamp.property_value();
		const double yfreq = m_yfreq.property_value();
		const double yphase = m_yphase.property_value();
		const double zamp = m_zamp.property_value();
		const double zfreq = m_zfreq.property_value();
		const double zphase = m_zphase.property_value();
		const double width = m_width.property_value();
		const double wrap = m_wrap.property_value();

		// Create a linear curve k3d::mesh ...
		std::auto_ptr<k3d::mesh> mesh(new k3d::mesh());

		k3d::linear_curve_group* const group = new k3d::linear_curve_group();
		group->material = m_material.interface();
		group->wrap = wrap;
		group->constant_data["width"] = width;

		k3d::linear_curve* const curve = new k3d::linear_curve();
		group->curves.push_back(curve);

		mesh->linear_curve_groups.push_back(group);

		// Create the curve ...
		for(unsigned long n = 0; n != edges; n++)
		{
			const double percent = static_cast<double>(n) / static_cast<double>(edges);
			const double theta = percent * k3d::pi_times_2();

			k3d::vector3 coords(
				xamp * sin(xfreq * theta + xphase),
				yamp * sin(yfreq * theta + yphase),
				zamp * sin(zfreq * theta + zphase));

			if(mamp)
				coords *= 1.0 + (mamp * sin(mfreq * theta + mphase));

			k3d::point* const p = new k3d::point(coords);

			mesh->points.push_back(p);
			curve->control_points.push_back(p);
		}

		return mesh.release();
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<k3d::document_plugin<lissajous_curve_implementation> > factory(
		k3d::uuid(0x7fdcbdc1, 0xd1c6404e, 0xae15a8eb, 0x45ffc793),
			"LissajousCurve",
			"Generates a lissajous (sine-wave) curve",
			"Objects",
			k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_data_property(long, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::with_constraint) m_edge_count;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_mamp;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_mfreq;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_mphase;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_xamp;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_xfreq;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_xphase;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_yamp;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_yfreq;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_yphase;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_zamp;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_zfreq;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_zphase;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_width;
	k3d_data_property(bool, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_wrap;
};

k3d::iplugin_factory& lissajous_curve_factory()
{
	return lissajous_curve_implementation::get_factory();
}

} // namespace libk3dmesh
